#
#    Copyright (c) 1996-2016 Brocade Communications Systems, Inc.
#    All rights reserved.
#
#    File name:   diagcommon.sh
#    Module name: diag/scripts
#
#    This file contains several useful functions used by the
#    diagnostics user command scripts.  They are stuck here
#    to save some disk image space.  To optimize execution time
#    copy this file into each of the command scripts instead.
#

PATH="/bin:/usr/bin:/sbin:/usr/sbin:/fabos/sbin:/fabos/rbin:/fabos/bin"
_default_home="/fabos"                 # FABOSHOME default
_cfg_cmd="config"		      # config program
_def_slot=0
_config_switch=0
# definitions that must match src/include/config/resource.h 
# define Resource* (line 68-line 74)
INTEGER=2			      # config modes
STRING=5
HEX=6
##################
# Utilities
#

# error
# $1 error string
err () {
  /bin/echo "" >&2
  /bin/echo "Error -- $1" >&2
  if [ -n "$syntax" ]
  then
    eval /bin/echo Syntax: $syntax >&2
  fi
}

# burninLogPath
#
# Get full path to burnin log files for a script
# $1 slot
#
burninLogPath () {
  # Find path to log files
  _tmp_dir=`getConfig diag.mode.burnin.logpath $STRING "/var/log"`
  if [ ! -d $_tmp_dir ]
  then
    _tmp_dir="/var/log"
  fi
  _log_path=`getConfig diag.$1.LogFilePath $STRING "$_tmp_dir"`
  if [ ! -d $_log_path ]
  then
    _log_path=$_tmp_dir
  fi
  echo $_log_path
}
# commandPath
#
# Get full path to FABOS command
# $1 command name
#
commandPath () {
  # First try fabos/cliexec/command
  if [ "$FABOSHOME" = "" ]
  then
    _cmd="$_default_home/cliexec/$1"
  else
    _cmd="$FABOSHOME/cliexec/$1"
  fi
  if [ -x $_cmd ]
  then
    /bin/echo $_cmd
    return 0
  fi

  # No luck -- try fabos/sbin/command
  if [ "$FABOSHOME" = "" ]
  then
    _cmd="$_default_home/sbin/$1"
  else
    _cmd="$FABOSHOME/sbin/$1"
  fi
  if [ -x $_cmd ]
  then
    /bin/echo $_cmd
    return 0
  fi


  # No luck -- try fabos/bin/command
  if [ "$FABOSHOME" = "" ]
  then
    _cmd="$_default_home/bin/$1"
  else
    _cmd="$FABOSHOME/bin/$1"
  fi
  if [ -x $_cmd ]
  then
    /bin/echo $_cmd
    return 0
  fi

  # Still no luck try fabos/command
  if [ "$FABOSHOME" = "" ]
  then
    _cmd="$_default_home/$1"
  else
    _cmd="$FABOSHOME/$1"
  fi
  if [ -x $_cmd ]
  then
    /bin/echo $_cmd
    return 0
  fi

  # Still no luck try fabos/share
  if [ "$FABOSHOME" = "" ]
  then
    _cmd="$_default_home/share/$1"
  else
    _cmd="$FABOSHOME/share/$1"
  fi
  if [ -x $_cmd ]
  then
    /bin/echo $_cmd
    return 0
  fi

  # Still no luck try fabos/factory
  if [ "$FABOSHOME" = "" ]
  then
    _cmd="$_default_home/factory/$1"
  else
    _cmd="$FABOSHOME/factory/$1"
  fi
  if [ -x $_cmd ]
  then
    /bin/echo $_cmd
    return 0
  fi

  # Still no luck try /bin
  if [ -x /bin/$1 ]
  then
    /bin/echo /bin/$1
    return 0
  fi

  # Still no luck try /usr/bin
  if [ -x /usr/bin/$1 ]
  then
    /bin/echo /usr/bin/$1
    return 0
  fi

  # Last but not least try plain old command.
  _cmd="$1"
  if [ -x $_cmd ]
  then
    /bin/echo $_cmd
    return 0
  fi

  # Ok we give up now.
  err "Could not find: $_cmd"
  return 3

}
#-------------------------------
# Configuration support section
#
# The configuration variables for diagnostics are intended to
# apply to the entire chassis. They are not switch based. When
# there are multiple switches in the same chassis, then a single
# switch's config database will be used for all blades in the
# switch. This switch number is defined in the variable
# _config_switch at the top of this file. At some point it may
# be necessary to initialize this variable dynamically. Currently,
# it is statically initialized to switch 0. The assumption is that
# the switches are umbered starting from 0, and that there will
# always be a switch 0.
#-------------------------------

# setConfig()
#
# Set configuration variable value
# $1 = config string to set
# $2 = config mode
# $3 = new value
#
setConfig () {
  _cmd=`commandPath $_cfg_cmd`
  if [ $? != 0 ]
  then
    exit 3
  fi

  _initial_switch=$FABOS_SWITCHNO
      export FABOS_SWITCHNO=$_config_switch
  if { $_cmd set "$1" "$2" "$3"; }
  then
    export FABOS_SWITCHNO=$initial_switch
    return 0
  else
    err "setConfig failed"
    export FABOS_SWITCHNO=$_initial_switch
    exit 3
  fi
}

# commitConfig()
#
# Push configuration changes to kernel.
#  -- no inputs required.
#
commitConfig () {
  _cmd=`commandPath $_cfg_cmd`
  if [ $? != 0 ]
  then
    exit 3
  fi

  _initial_switch=$FABOS_SWITCHNO
  export FABOS_SWITCHNO=$_config_switch
  if { $_cmd commit; }
  then
    /bin/echo "Config commit Succeeded" >&2
    export FABOS_SWITCHNO=$initial_switch
    return 0
  else
    err "commitConfig failed"
    export FABOS_SWITCHNO=$_initial_switch
    exit 3
  fi
}

# updateConfig() 
# 
# Commit configuration changes to NV storage. 
#  -- no inputs required. 
# 
updateConfig () { 
  _cmd=`commandPath $_cfg_cmd` 
  if [ $? != 0 ]
  then
    exit 3
  fi

  _initial_switch=$FABOS_SWITCHNO
  export FABOS_SWITCHNO=$_config_switch
  prevstate=`burninSetLevel -show | \
      /usr/bin/cut -d" " -f4 | \
      /usr/bin/cut -d"." -f1`
  if [ "$prevstate" != "0" ]
  then
     burninSetLevel 0
  fi
  for _try in 1 2 3 4 5 6 7 8 9 10
  do
    $_cmd update 0 0 > /dev/null 2>&1
    if [ $? = 0 ]
    then 
      /bin/echo "Config update Succeeded" >&2
      export FABOS_SWITCHNO=$_initial_switch
      if [ "$prevstate" != "0" ]
      then
        burninSetLevel 1 -noupdate
      fi
      return 0 
    fi
    if [ $_try -ne 10 ]
    then
      /bin/echo "Config update failed, retrying" >&2
      /bin/sleep 3
    fi
  done;

  if [ "$prevstate" != "0" ]
  then
    burninSetLevel 1 -noupdate
  fi
  # Should not get here
  err "Config update failed" 
  export FABOS_SWITCHNO=$_initial_switch
  exit 3 
}
 
# ConfigSetCheck()
#
# Check if a config variable has been set
# $1 = config string to set
# $2 = config mode
#
ConfigSetCheck () {
  _cmd=`commandPath $_cfg_cmd`
  if [ $? != 0 ]
  then
    exit 3
  fi

  _initial_switch=$FABOS_SWITCHNO
  export FABOS_SWITCHNO=$_config_switch
  _val=`$_cmd get "$1" "$2" 2>/dev/null`
  export FABOS_SWITCHNO=$_initial_switch
  if [ -z "$_val" ]
  then
    return 1
  fi
  return 0
}

# getConfig()
#
# Set configuration variable value
# $1 = config string to set
# $2 = config mode
# $3 = default
#
getConfig () {
  _cmd=`commandPath $_cfg_cmd`
  if [ $? != 0 ]
  then
    exit 3
  fi

  _initial_switch=$FABOS_SWITCHNO
  export FABOS_SWITCHNO=$_config_switch
  _val=`$_cmd get "$1" "$2" 2>/dev/null`
  export FABOS_SWITCHNO=$_initial_switch

  if [ -z "$_val" ]
  then
    _val="$3"
  fi
  /bin/echo $_val
  return 0
}

# testConfig()
#
# Set configuration variable value
# $1 = config string to test
# $2 = config mode
#
testConfig () {
  _cmd=`commandPath $_cfg_cmd`
  if [ $? != 0 ]
  then
    exit 3
  fi

  _initial_switch=$FABOS_SWITCHNO
  export FABOS_SWITCHNO=$_config_switch
  _val=`$_cmd get "$1" $2 2>/dev/null`
  export FABOS_SWITCHNO=$_initial_switch

  if [ -z "$_val" ]
  then
    /bin/echo "0"
  else
    /bin/echo "1"
  fi
  return 0
}
#-----------------------
# End config support
#-----------------------

# checkForShow()
#
# Display current setting and bail if this is a show
# command.  Also check for -inc and increment the current
# setting but only if the config_mode is not STRING and
# the value is not boolean.
#
# $1 = config_name
# $2 = config_string
# $3 = config_mode
# $4 = config_default
# $5 = argument to check. (must be last)
checkForShow () {
  _s=`/usr/bin/expr "$5" : "[-]*show"`
  if { [ -z "$5" ] || [ "0" -ne "$_s" ] ; }
  then
    _val=`getConfig $2 $3 $4`
    if [ $? != 0 ]
    then
      exit 3
    fi
    if { [ -n "$bool_true" ] && [ -n "$bool_false" ] ; }
    then
      if [ "$_val" -ne 0 ]
      then
        _res=$bool_true
      else
        _res=$bool_false
      fi
      /bin/echo "$1 is $_val ($_res)".
    else
      /bin/echo "$1 is $_val".
    fi

    exit 0
  else
    _s=`/usr/bin/expr "$5" : "[-]*inc"`
    if { [ "$3" != "$STRING" ] && [ "0" -ne "$_s" ] && \
         [ -z "$bool_true" ] && [ -z "$bool_false" ] ; }
    then
      _val=`getConfig $2 $3 $4`
      if [ $? != 0 ]
      then
        exit 3
      fi
      _val=`/usr/bin/expr $_val + 1`
      setConfig $2 $3 $_val
      /bin/echo "$1 is now ${_val}."
      exit 0
    fi
    return 0
  fi
}

# getValue()
#
# Converts input string into number or generates an
# error.  Special keywords [-]enable (1) [-]disable (0)
# [-]on (1) and [-off] (0) are also supported for
# boolean conditions
#
# $1 = parameter to check

getValue () {
  _s=`/usr/bin/expr "$1" : "[-]*enable"`
  if [ "0" -ne "$_s" ]
  then
    /bin/echo "1"
    return 0
  fi

  _s=`/usr/bin/expr "$1" : "[-]*on"`
  if [ "0" -ne "$_s" ]
  then
    /bin/echo "1"
    return 0
  fi

  _s=`/usr/bin/expr "$1" : "[-]*disable"`
  if [ "0" -ne "$_s" ]
  then
    /bin/echo "0"
    return 0
  fi

  _s=`/usr/bin/expr "$1" : "[-]*off"`
  if [ "0" -ne "$_s" ]
  then
    /bin/echo "0"
    return 0
  fi

  _n=`/bin/echo "$1" | \
    /usr/bin/wc -c | \
    /usr/bin/tr -s " " " " | \
    /usr/bin/cut -d" " -f2`
  _n=`/usr/bin/expr $_n - 1`
  _s=`/usr/bin/expr "$1" : "[-][0-9][0-9]*"`
  _t=`/usr/bin/expr "$1" : "[0-9][0-9]*"`
  if [ "$_n" -eq "$_s" -o  "$_n" -eq "$_t" ]
  then
    /bin/echo $1
    return 0
  fi

  err "Invalid parameter $1"
  /bin/echo "-1"
  exit 3
}

##################################
# NAME
#    enableSlotProc - enable /proc/fabos/blade/slot entries
#
# SYNOPSIS
#    enableSlotProc slot_no
#
# DESCRIPTION
#    
#    1) check slot existence
#    2) get proc entries by calling debugentryon
#    3) varify the exection by checking its return status
#	if fail, exit and tell the user to relinguish others
#	if success, continue
#
# OPERANDS
#    slot_no
##################################
function enableSlotProc() {
  if [ ! -e /proc/fabos/blade/$1 ]
  then
    err "Slot: $slot not found"
    exit 3
  fi
  /fabos/cliexec/debugentryon --slot $1 > /dev/null 2>&1
  if [ "0" -ne "$?" ]
  then
    if [ `isMultiBlade` = TRUE ]
    then
      /bin/echo "Slot$1: not enough proc entires, relinguish by running "debugEntryOff" first"
      /bin/echo "       Usage: debugEntryOff --slot x"
    else
      /bin/echo "Not enough proc entires, relinguish by running "debugEntryOff" first"
      /bin/echo "       Usage: debugEntryOff"
    fi
    exit 3
   fi
}

##################################
# NAME
#    disableSlotProc - disable /proc/fabos/blade/slot entries
#
# SYNOPSIS
#    disableSlotProc slot_no
#
# DESCRIPTION
#    
#    release previously enabled proc entries.
#
# OPERANDS
#    slot_no
##################################
function disableSlotProc() {
  # relinguish the proc entries
  /fabos/cliexec/debugentryoff --slot $1 > /dev/null 2>&1
}

#
# Begin burnin script support
# Bill Lewis 7/24/2001
#
##################################
#PRIVATE FUNCTIONS
##################################
##################################
# NAME
#    get_config_list - Get the output of the configShow diag entries
#		       to stdout.
#
# SYNOPSIS
#    get_config_list
#
# DESCRIPTION
#    This command executes the configShow function and places
#    a "@" seperator after every input line that belongs to the 
#    calling script. Any input line that is not related to the calling
#    script is eliminated from the output by the "/bin/grep diag.$MyName"
#    filter.
#
# OPERANDS
#    NONE
#
##################################
get_config_list()
{
  _initial_switch=$FABOS_SWITCHNO
  export FABOS_SWITCHNO=$_config_switch
  /bin/cat /etc/fabos/fabos.0.conf  2>/dev/null | /bin/grep diag.$MyName | while read _line
  do
    /bin/echo $_line@
  done
  export FABOS_SWITCHNO=$_initial_switch
}

##################################
# NAME
#    get_config_entry - Get the value of a "configVariable:value" pair
#		        from the output generated by get_config_list.
#
# SYNOPSIS
#    get_config_entry
#
# DESCRIPTION
#    This command searches the output of config_list until a match on
#    the traget is found. The sed outputs the string starting from the
#    match of $target until the "@" seperator. The "cut" on ":" delimiter
#    trims off $target and the ":". The "cut" on "@" delimiter
#    removes the "@" sight leaving the value of the config variable
#    named $target.
#
# OPERANDS
#    $1 = target to match in the string
#    config_list = variable containing the string to search
#
##################################
get_config_entry()
{
  target=$1
  if [ ! -z "`/bin/echo $config_list | /bin/grep -w $target`" ]
  then
    /bin/echo $config_list | \
      /bin/sed -e "s/^.*\(\.${target}:[[:print:]]\+\).*@/\1/g" |\
      /usr/bin/cut -d":" -f2 | \
      /usr/bin/cut -d"@" -f1
  fi
}

##################################
# NAME
#    get_param_list - Get the output of the create_param_list
#		        function to stdout
#
# SYNOPSIS
#    get_param_list
#
# DESCRIPTION
#    This command executes the create_param_list function and places
#    a "@" seperator after every line that is a paramdef. Any line that
#    is not a param is eliminated from the output by the "PARAMDEF"
#    filter.
#
#    Added filter for lines that are comments. They could be commented
#    out PARAMDEFs.
#
# OPERANDS
#    NONE
#
##################################
get_param_list()
{
  create_param_list | /bin/grep -v "^#" | while read _line
  do
    new_line=`/bin/echo $_line | /usr/bin/cut -d" " -f1,2,$DefaultField,$ModeField,$MinField,$MaxField,$CommentField- | /bin/grep -e ' V:' | /bin/sed "s/ V:/ /g"`
    if [ ! -z "$new_line" ]
    then
      /bin/echo $new_line@
    fi
  done
}

##################################
# NAME
#    get_paramconst_list - Get the output of the create_paramlist
#		        function to stdout
#
# SYNOPSIS
#    get_paramconst_list
#
# DESCRIPTION
#    This command executes the create_param_list function and places
#    a "@" seperator after every line that is a param constant.
#    Any line that is not a param is eliminated from the output by
#    the "PARAMCONST" filter and the "^#" filter.
#
# OPERANDS
#    NONE
#
##################################
get_paramconst_list()
{
  create_param_list | /bin/grep -v "^#" | while read _line
  do
    new_line=`/bin/echo $_line | /usr/bin/cut -d" " -f1,2,$DefaultField,$ModeField,$MinField,$MaxField,$CommentField- | /bin/grep -e ' C:' | /bin/sed "s/ C:/ /g"`
    if [ ! -z "$new_line" ]
    then
      /bin/echo $new_line@
    fi
  done
}

##################################
# NAME
#    get_match_list - Get a list of the lines that contain the string
#		      $2 to stdout
#
# SYNOPSIS
#    get_match_list <file> <match_string>
#
# DESCRIPTION
#    This searches file named $1 for string $2 and places a @seperated
#    list of the lines that matches were found.
#
#    The output is a single line with "@" seperated
#    match strings.
#
# OPERANDS
#    NONE
#
##################################
get_match_list()
{
  _fil=$1
  shift
  /bin/cat "$_fil" | /bin/grep -n $* | while read _line
  do
    /bin/echo $_line@
  done
}

##################################
# This function valide_integer() is an internal only function.
# It determines if an integer is between two values. It
# expects the first parameter to be the value in question. The
# second value to be the minimum acceptable value, and the
# third value to be the maximum acceptable value. The min and
# max values are considered valid.
# $1 = value to test
# $2 = minimum acceptable value
# $3 = maximum acceptable value
##################################
validate_integer()
{
  _pass=TRUE
  _x=`/usr/bin/expr $1 : "\([-]*[0-9]\+\)"`
  if [ "$_x" != "$1" ]
  then
    _pass=NOTNUMBER
  else
    if [ "$2" != "NA" ]
    then
      _x=`/usr/bin/expr $2 : "\([-]*[0-9]\+\)"`
      if [ "$_x" != "$2" ]
      then
        _pass=MINNOTNUMBER
      else
        if [ "$1" -lt "$2" ]
        then
          _pass=LESSTHANMIN
        fi
      fi
    fi
    if [ "$3" != "NA" ]
    then
      _x=`/usr/bin/expr $3 : "\([-]*[0-9]\+\)"`
      if [ "$_x" != "$3" ]
      then
        _pass=MAXNOTNUMBER
      else
        if [ "$1" -gt "$3" ]
        then
          _pass=GREATERTHANMAX
        fi
      fi
    fi
  fi
  /bin/echo $_pass
}

##################################
# NAME
#    getEmulator - Output the emulation model
#
# SYNOPSIS
#    getEmulator
#
# DESCRIPTION
#    This function parses the output of the sin command to extract the
#    emulation model for this platform.
#
# OPERANDS
#    NONE
##################################
getEmulator()
{
  _platform=`getSWBD`
  case "$_platform" in
  "SWBD21" | "SWBD42" | "SWBD62" | "SWBD77" )
    _cmd=`/fabos/rbin/voltshow -d 626`
    if [ "`$_cmd | /bin/grep -i i2c`" = "" ]
    then
      _abc=`$_cmd | /bin/grep "CP BLADE" | \
        /usr/bin/tr -s "	 " "  " | \
        /usr/bin/head -1 | \
        /usr/bin/cut -d":" -f2 | \
        /usr/bin/cut -d" " -f3`
      if [ "$_abc" = "" ]
      then
        _abc=128MOCKUP_DONT_CHANGE
      else
        case $_abc in
        "--" ) _abc="128MOCKUP_DONT_CHANGE" ;;
        * )    _abc="CORE_EDGE_MOCKUP_DONT_CHANGE" ;;
        esac
      fi
    else
      _abc="REAL"
    fi
    # if we can not determine what it is, then assume it is 128 port emulator
    ;;
  * )
    _abc="REAL"
    ;;
  esac
  /bin/echo "$_abc"
}
##################################
# NAME
#    getSWBD - Output the software board ID
#
# SYNOPSIS
#    getSWBD
#
# DESCRIPTION
#    This function parses the output of the sin command to extract the
#    SWBD value for this platform.
#
#    Use of this function is discouraged. We intend to move the scripts
#    toward a capability of property driven queries instead of parsing
#    model numbers to reduce future maintainence.
#
# OPERANDS
#    NONE
##################################
getSWBD()
{
  _abc=`/sbin/sin | \
    /bin/grep Platform | \
    /usr/bin/cut -d" " -f2 | \
    /usr/bin/cut -d"," -f1`
  /bin/echo $_abc
}
##################################
# NAME
#    getbladeID - Output the software board ID
#
# SYNOPSIS
#    getBladeID <slot_no>
#
# DESCRIPTION
#    This function parses the output of the slotshow command to extract the
#    blade ID for the given slot.
#
# OPERANDS
#    NONE
##################################
getBladeID()
{
  _bladeid="0"
  _abc="UNKNOWN"
  if [ ! -e /var/log/diag_slotshow.log ]
  then
    `/fabos/link_bin/slotshow -d575 | \
    /bin/grep BLADE | \
    /usr/bin/tr -s "    " " " | \
    /usr/bin/cut -d" " -f2,5 > \
    /var/log/diag_slotshow.log`
  fi
  _bladeid=`/bin/cat /var/log/diag_slotshow.log | \
    /bin/grep -w "^$1" | \
    /usr/bin/cut -d" " -f2`
  case "$_bladeid" in
  "1" ) _abc="CP" ;;
  "2" ) _abc="STILETTO" ;;
  "3" ) _abc="TERMINATOR" ;;
  "4" ) _abc="PORT" ;;
  "5" ) _abc="CP128" ;;
  "6" ) _abc="DAZZLER" ;;
  "7" ) _abc="DAZZLERJR" ;;
  "8" ) _abc="BLAZER" ;;
  "9" ) _abc="HAMPTON" ;;
  "10" ) _abc="PULSAR" ;;
  "15" ) _abc="PULSAR32" ;;
  "16" ) _abc="SATURN_CP" ;;
  "17" ) _abc="PROTON" ;;
  "18" ) _abc="ELECTRON" ;;
  "19" ) _abc="XENON" ;;
  "20" ) _abc="FORTE" ;;
  "21" ) _abc="ICECUBE" ;;
  "22" ) _abc="GLACIER" ;;
  "23" ) _abc="STEALTH" ;;
  "24" ) _abc="MARATHON" ;;
  "26" ) _abc="BRADLEY" ;;
  "27" ) _abc="VIKING" ;;
  "28" ) _abc="GLACIER2" ;;
  "29" ) _abc="SPRINT" ;;
  "31" ) _abc="IRONMAN" ;;
  "33" ) _abc="SCIMITAR" ;;
  "34" ) _abc="DRAGON" ;;
  "36" ) _abc="POSITRON" ;;
  "37" ) _abc="DESPINA" ;;
  "38" ) _abc="SABRE" ;;
  "39" ) _abc="XENON2" ;;
  "42" ) _abc="MACE" ;;
  "43" ) _abc="LANCE" ;;
  "45" ) _abc="ELARA" ;;
  "46" ) _abc="HYDRA" ;;
  "49" ) _abc="SUPERHAWK" ;;
  "50" ) _abc="ZENTRON_CP" ;;
  "51" ) _abc="TRITON" ;;
  "52" ) _abc="PROTEUS" ;;
  "55" ) _abc="LARISSA" ;;
  "60" ) _abc="BULOVA" ;;
  "61" ) _abc="BLAZER3" ;;
  "63" ) _abc="SUPERHAWK2" ;;
  "69" ) _abc="SPIKE" ;;
  "70" ) _abc="FALCON2" ;;
  "73" ) _abc="KESTREL2" ;;
  "74" ) _abc="EUROPA" ;;
  "75" ) _abc="STRIKER" ;;
  "77" ) _abc="QUATTRO" ;;
  "96" ) _abc="THALASSA" ;;
  "97" ) _abc="GALATEA" ;;
  "98" ) _abc="NESO" ;;
  "99" ) _abc="CHARON" ;;
  "100" ) _abc="STINGER" ;;
  "119" ) _abc="PHAROS" ;;
  "120" ) _abc="TOMTOO" ;;
  "123" ) _abc="DRAGON2" ;;
  "124" ) _abc="DRAGON2T" ;;
  "125" ) _abc="GALATEA8G" ;;
  "126" ) _abc="THALASSA8G" ;;
  "129" ) _abc="PHAROS_LITE" ;;
  "133" ) _abc="BULOVA2" ;;
  "134" ) _abc="SUPERHAWK3" ;;
  "137" ) _abc="ODIN" ;;
  "144" ) _abc="QILIN" ;;
  "152" ) _abc="YODA" ;;
  "153" ) _abc="WINDU" ;;
  "154" ) _abc="SKYBOLT" ;;
  "155" ) _abc="KESTREL3" ;;
  "156" ) _abc="FALCON3" ;;
  "185" ) _abc="SHENLONG" ;;
  "200" ) _abc="TIANLONG" ;;
  "165" ) _abc="REDBIRD" ;;
  "166" ) _abc="GRAPHITE" ;;
  "167" ) _abc="CARBON" ;;
  "170" ) _abc="WEDGE" ;;
  "171" ) _abc="WEDGE64" ;;
  "175" ) _abc="LANDO" ;;
  "176" ) _abc="MUNDI" ;;
  "177" ) _abc="FISTO" ;;
  "178" ) _abc="SKYWALKER" ;;
  "186" ) _abc="HARPOON" ;;
  "190" ) _abc="CHEWBACCA" ;;
  "196" ) _abc="CHEWBACCA24" ;;
  "202" ) _abc="TYR" ;;
  "204" ) _abc="ANAKIN" ;;
  "213" ) _abc="A_WING" ;;
  * )  _abc="UNKNOWN" ;;
  esac
  /bin/echo $_abc
}
##################################
# NAME
#    getHWversion - Output the hardware version ID
#
# SYNOPSIS
#    getHWversion <slot_no>
#
# DESCRIPTION
#    This function parses the output of the chassisshow command to extract the
#    hardware version ID for the given slot.
#
# OPERANDS
#    NONE
##################################
getHWversion()
{
chassisdata=`chassisshow -d575`
_data=`/bin/echo $chassisdata | /bin/sed "s/\// /g"`
_datacnt=`/bin/echo $_data | /usr/bin/wc -w`
i=1
rev=255

while [ $i -lt $_datacnt ]
do
   _data2=`/bin/echo $_data | /usr/bin/cut -d" " -f$i`
   if [ "$_data2" = "Slot:" ]
   then
        num=`/usr/bin/expr $i + 1`
        slotnum=`/bin/echo $_data | /usr/bin/cut -d" " -f$num`
        if [ "$slotnum" = "$1" ] 
   	then
   	   j=1
   	   while [ 1 ]
   	   do
   	      cnt1=`/usr/bin/expr $i + $j`
              if [ $cnt1 -gt $_datacnt ] 
   	      then
		 /bin/echo $rev
   	         return 1
   	      fi
   	         	      
   	      cnt2=`/usr/bin/expr $cnt1 + 1`
   	      cnt3=`/usr/bin/expr $cnt2 + 1`
   	      _data3=`/bin/echo $_data | /usr/bin/cut -d" " -f$cnt1`
   	      _data4=`/bin/echo $_data | /usr/bin/cut -d" " -f$cnt2`
              if { [ "$_data3" = "FRU" ]  && [ "$_data4" = "ID:" ]; }
   	      then
        	fru=`/bin/echo $_data | /usr/bin/cut -d" " -f$cnt3`
		rev=`/bin/echo $fru | /usr/bin/cut -c9-10` 
		/bin/echo $rev
        	return 0
   	      else
   	        if [ "$_data3" = "Manufacture:" ]
   	        then
		   /bin/echo $rev
   	           return 1
   	        fi
   	      fi   	     
   	      j=`/usr/bin/expr $j + 1`   	   
   	   done
   	   	
   	fi        
   fi
   i=`/usr/bin/expr $i + 1`
done
/bin/echo $rev
return 1
}
##################################
# NAME
#    getUserPortList - Output the slist of user ports for a given slot
#
# SYNOPSIS
#    getUserPortList <slot_no> [ -list ]
#
# DESCRIPTION
#    This function parses the output of the diagshow command to extract the
#    list of user ports for the given slot.
#
# OPERANDS
#    NONE
##################################
getUserPortList()
{
  _initial_slot=$FABOS_SLOTNO
  export FABOS_SLOTNO=$1
  _cmd=`commandPath exec_diag`
  _abc=""
  prev_port=""
  case `getSWBD` in
  "SWBD148" | "SWBD171" )
  USER_PORT_LIST=`bladeportmap | grep "CONDOR3" | awk '$3 ~ /^[0-9]+$/ { print$3 }'`;;
   * )
  USER_PORT_LIST=${USER_PORT_LIST:-$($_cmd diagshow -use_bports 1 | \
   sed -n '/^[ \t]*--[ \t]/d;/ OK/s/^[ \t]*\([^ \t]*\).*/\1/p' )};;
  esac
  if [ "$USER_PORT_LIST" != "" ]
  then
    USER_PORT_LIST=`/bin/echo $USER_PORT_LIST | \
       tr " " "\012" | \
       /usr/bin/cut -d"/" -f2 | \
       /bin/sort -n | \
       /usr/bin/xargs /bin/echo`
  else
      /bin/echo "NO_USER_PORTS"
      export FABOS_SLOTNO=$_initial_slot
      return
  fi
  if [ "$2" = "-list" ]
  then
    /bin/echo $USER_PORT_LIST
    return
  fi
  for port in $USER_PORT_LIST
  do
    if [ -z "$prev_port" ]
    then
      prev_port=$port
      last_port=$port
      first_port=$port
    else
      prev_port=`/usr/bin/expr $prev_port + 1`
      if [ $prev_port = $port ]
      then
        last_port=$port
      else
        # non-consecutive number, need to dump what we have so far
        if [ $first_port = $last_port ]
        then
          _abc="${_abc}${first_port},"
        else
	   if { [ "$first_port" -lt "16" ] && [ "$last_port" -gt "15" ]  && [ "$2" = "-split" ]; }
	   then
          	_abc="${_abc}${first_port}-15,16-${last_port},"
	   else
          	_abc="${_abc}${first_port}-${last_port},"
           fi
        fi
        
        first_port=$port
        last_port=$port
	prev_port=$port
      fi
    fi
  done
  if [ -z "$_abc" ]
  then
    if [ -z "$first_port" ]
    then
      /bin/echo "NO_USER_PORTS"
      export FABOS_SLOTNO=$_initial_slot
      return
    else
      if [ $first_port != $last_port ]
      then
	if { [ "$first_port" -lt "16" ] && [ "$last_port" -gt "15" ]  && [ "$2" = "-split" ]; }
	then
           _abc=${first_port}-15,16-${last_port}
	else
	   _abc=${first_port}-${last_port}
	fi
      else
	_abc=${first_port}
      fi
    fi
  else
    if [ $first_port != $last_port ]
    then
      if { [ "$first_port" -lt "16" ] && [ "$last_port" -gt "15" ]  && [ "$2" = "-split" ]; }
      then
           _abc=${_abc}${first_port}-15,16-${last_port}
      else
      	   _abc=${_abc}${first_port}-${last_port}
      fi
    else
      _abc=${_abc}${first_port}
    fi
  fi
  export FABOS_SLOTNO=$_initial_slot
  /bin/echo "$1/$_abc"
}
##################################
# NAME
#    getUserPortListContext - Output the slist of user ports for a given slot
#                             AND the current context
#
# SYNOPSIS
#    getUserPortListContext <slot_no> [ -list ]
#
# DESCRIPTION
#    This function parses the output of the diagshow command to extract the
#    list of user ports for the given slot and the current context.
#
# OPERANDS
#    NONE
##################################
getUserPortListContext()
{
  _initial_slot=$FABOS_SLOTNO
  export FABOS_SLOTNO=$1
  _cmd=`commandPath exec_diag`
  _abc=""
  prev_port=""
  
  list=`$_cmd diagshow | /bin/grep "$FABOS_SLOTNO/[0-9]" | \
       /bin/grep "   OK" | \
       /usr/bin/cut -d " " -f 2 | \
       /bin/grep -v "\-\-"`

  if [ "$list" != "" ]
  then
    list=`/bin/echo $list | \
       tr " " "\012" | \
       /usr/bin/cut -d"/" -f2 | \
       /bin/sort -n | \
       /usr/bin/xargs /bin/echo`
  else
      /bin/echo "NO_USER_PORTS"
      export FABOS_SLOTNO=$_initial_slot
      return
  fi
  if [ "$2" = "-list" ]
  then
    /bin/echo $list
    return
  fi
  for port in $list
  do
    if [ -z "$prev_port" ]
    then
      prev_port=$port
      last_port=$port
      first_port=$port
    else
      prev_port=`/usr/bin/expr $prev_port + 1`
      if [ $prev_port = $port ]
      then
        last_port=$port
      else
        # non-consecutive number, need to dump what we have so far
        if [ $first_port = $last_port ]
        then
          _abc="${_abc}${first_port},"
        else
	   if { [ "$first_port" -lt "16" ] && [ "$last_port" -gt "15" ]  && [ "$2" = "-split" ]; }
	   then
          	_abc="${_abc}${first_port}-15,16-${last_port},"
	   else
          	_abc="${_abc}${first_port}-${last_port},"
           fi
        fi
        
        first_port=$port
        last_port=$port
	prev_port=$port
      fi
    fi
  done
  if [ -z "$_abc" ]
  then
    if [ -z "$first_port" ]
    then
      /bin/echo "NO_USER_PORTS"
      export FABOS_SLOTNO=$_initial_slot
      return
    else
      if [ $first_port != $last_port ]
      then
	if { [ "$first_port" -lt "16" ] && [ "$last_port" -gt "15" ]  && [ "$2" = "-split" ]; }
	then
           _abc=${first_port}-15,16-${last_port}
	else
	   _abc=${first_port}-${last_port}
	fi
      else
	_abc=${first_port}
      fi
    fi
  else
    if [ $first_port != $last_port ]
    then
      if { [ "$first_port" -lt "16" ] && [ "$last_port" -gt "15" ]  && [ "$2" = "-split" ]; }
      then
           _abc=${_abc}${first_port}-15,16-${last_port}
      else
      	   _abc=${_abc}${first_port}-${last_port}
      fi
    else
      _abc=${_abc}${first_port}
    fi
  fi
  export FABOS_SLOTNO=$_initial_slot
  /bin/echo "$1/$_abc"
}
##################################
# NAME
#    hasRefClock - query for existance of reference clock control.
#
# SYNOPSIS
#    hasRefClock <slot_no>
#
# DESCRIPTION
#    This used the SWBD type to output TRUE or FALSE as applies to the
#    existance of the reference clock control on the platform.
#
# OPERANDS
#    NONE
##################################
hasRefClock()
{
  case `getSWBD` in
  "SWBD10" ) /bin/echo TRUE;;
  * )        /bin/echo FALSE;;
  esac
}
##################################
# NAME
#    getCPLDVersionAddress - returns the address and field offset to the
#				CPLD version register in the xtool output.
#
# SYNOPSIS
#    hasRefClock <slot_no>
#
# DESCRIPTION
#    This used the SWBD type to output TRUE or FALSE as applies to the
#    existance of the reference clock control on the platform.
#
# OPERANDS
#    NONE
##################################
getCPLDVersionAddress()
{
  case `getSWBD` in
  "SWBD10" ) /bin/echo f8000010 17;;
  "SWBD12" ) /bin/echo f8000010 17;;
  "SWBD21" ) /bin/echo f8000000 3;;
  "SWBD26" ) /bin/echo f8000010 17;;
  "SWBD27" ) /bin/echo f8000010 17;;
  "SWBD29" ) /bin/echo f8000010 17;;
  "SWBD33" ) /bin/echo f8000010 17;;
  "SWBD37" ) /bin/echo f8000010 17;;
  "SWBD42" ) /bin/echo 1f8000000 2;;
  "SWBD45" ) /bin/echo f8000010 17;;
  "SWBD51" ) /bin/echo f8000010 17;;
  "SWBD61" ) /bin/echo f8000010 17;; 
  "SWBD62" ) /bin/echo f8000000 0;;
  "SWBD77" ) /bin/echo f8000000 0;;
  * )        /bin/echo FALSE;;
  esac
}
##################################
# NAME
#    isMultiBlade - Output TRUE if product has multiple blades
#
# SYNOPSIS
#    isMultiBlade
#
# DESCRIPTION
#    This function parses the output of the sin command to extract the
#    SWBD value for this platform, then it uses that information to
#    determine a TRUE/FALSE answer.
#
# OPERANDS
#    NONE
##################################
isMultiBlade()
{
  _abc=`/sbin/sin | \
    /bin/grep Platform | \
    /usr/bin/cut -d" " -f2 | \
    /usr/bin/cut -d"," -f1`
  case $_abc in 
  "SWBD10" ) /bin/echo "TRUE";;
  "SWBD21" ) /bin/echo "TRUE";;
  "SWBD42" ) /bin/echo "TRUE";;
  "SWBD62" ) /bin/echo "TRUE";;
  "SWBD77" ) /bin/echo "TRUE";;
  "SWBD141" ) /bin/echo "TRUE";;
  "SWBD142" ) /bin/echo "TRUE";;
  "SWBD165" ) /bin/echo "TRUE";;
  "SWBD166" ) /bin/echo "TRUE";;
  * )        /bin/echo "FALSE";;
  esac
}
isPfmAllegiance()
{
  _abc=`/sbin/sin | \
    /bin/grep Platform | \
    /usr/bin/cut -d" " -f2 | \
    /usr/bin/cut -d"," -f1`
  case $_abc in 
  "SWBD165" ) /bin/echo "TRUE";;
  "SWBD166" ) /bin/echo "TRUE";;
  * )        /bin/echo "FALSE";;
  esac

}
##################################
# NAME
#    isActiveCP - Output TRUE if current CP is the Active CP.
#
# SYNOPSIS
#    isActiveCP
#
# DESCRIPTION
#    This function parses the output of the hashow command to extract the
#    Active vs. Standby status of the current CP then it uses that information to
#    determine a TRUE/FALSE answer.
#
# OPERANDS
#    NONE
##################################
isActiveCP()
{
  _cmd=`commandPath haShow`
  _abc=`$_cmd | /bin/grep Local |\
    /bin/grep -e Active`
  if [ "$_abc" != "" ]
  then
    /bin/echo "TRUE"
  else
    /bin/echo "FALSE"
  fi
}
##################################
# NAME
#    getNumSwitches - returns the number of switches in the chassis
#
# SYNOPSIS
#    getNumSwitches
#
# DESCRIPTION
#    This is kluged to parses the output of the getchassisconfig command
#    to extract the value for this platform, then it uses that information to
#    determine the number of switches in the box.
#
# OPERANDS
#    NONE
##################################
getNumSwitches()
{
  _abc=`/bin/getchassisconfig | \
      /bin/grep "Number of switches" | \
      /usr/bin/cut -d":" -f2 | \
      /usr/bin/cut -d" " -f2`
  case $_abc in
  "Two" ) /bin/echo "2";;
  "One" ) /bin/echo "1";;
  "2" )   /bin/echo "2";;
  "1" )   /bin/echo "1";;
  * )     /bin/echo "1";;
  esac
}
##################################
# NAME
#    getAvailSlots - Output a space seperated list of the slots on this switch
#
# SYNOPSIS
#    getAvailSlots [ -chassis | -chassisnofault | -switch | -switchnofault ]
#
# DESCRIPTION
#    This function parses the output of the diagshow command to extract the
#    list of slots available in this switch.
#
# OPERANDS
#    -chassis	This option will list all switch components in the chassis
#    -withcp	This option is same as chassis, but includes include Ulysses CP
#    -nofault	This option is same as chassis, but excludes faulted blades
#
##################################
getAvailSlots()
{
  if [ "`isPfmAllegiance`" = TRUE ]
  then
  case "$1" in
    "-chassis" | "-switch" )
      _abc=`/fabos/link_bin/slotshow -d575 | \
        /usr/bin/tr -s " " " " | \
        /bin/grep -v "OUT" | \
        /bin/grep -v "IN" | \
            /bin/grep -v "CP BLADE" | \
        /bin/grep "BLADE" | \
        /usr/bin/cut -d" " -f2 | \
        /usr/bin/xargs /bin/echo`
      ;;
    "-chassisnofault" | "-switchnofault" )
      # chassis minus the faulted blades
      # switch minus the faulted blades
      _abc=`/fabos/link_bin/slotshow -d575 | \
        /usr/bin/tr -s " " " " | \
        /bin/grep -v "OUT" | \
        /bin/grep -v "IN" | \
        /bin/grep -v "FAULT" | \
            /bin/grep -v "FLTY" | \
            /bin/grep -v "UNKNOWN" | \
            /bin/grep -v "CP BLADE" | \
        /bin/grep "BLADE" | \
        /usr/bin/cut -d" " -f2 | \
        /usr/bin/xargs /bin/echo`
      ;;
  esac
  else
    case "$1" in
        "-chassis" | "-switch" )
          _abc=`/fabos/link_bin/slotshow -d575 | \
            /usr/bin/tr -s " " " " | \
            /bin/grep -v "OUT" | \
            /bin/grep -v "IN" | \
            /bin/grep "BLADE" | \
            /usr/bin/cut -d" " -f2 | \
            /usr/bin/xargs /bin/echo`
          ;;
        "-chassisnofault" | "-switchnofault" )
          # chassis minus the faulted blades
          # switch minus the faulted blades
          _abc=`/fabos/link_bin/slotshow -d575 | \
            /usr/bin/tr -s " " " " | \
            /bin/grep -v "OUT" | \
            /bin/grep -v "IN" | \
            /bin/grep -v "FAULT" | \
            /bin/grep "BLADE" | \
            /usr/bin/cut -d" " -f2 | \
            /usr/bin/xargs /bin/echo`
          ;;
    esac
  fi
  /bin/echo "$_abc"
}
##################################
# NAME
#    isValidSlot - Output TRUE if valid slot number otherwise FALSE
#
# SYNOPSIS
#    isValidSlot slot [ -chassis | -chassisnofault | -switch | -switchnofault ]
#
# DESCRIPTION
#    This function validate the input user slot number if
#    it is a valid or not by comparing with the result of getAvailSlots.
#    Then it will return TRUE/FALSE answer
# OPERANDS
#     $1 = user input slot number for validation
#     $2 = -chassis	This option check list of all switch components in the chassis
#          -withcp	This option is same as above, but includes include the Ulysses CP
#          omitted	This option checks list of all switch components in the switch
#
##################################
isValidSlot()
{
  slot="$1"
  list=`getAvailSlots $2`
  if [ -z $1 ]
  then
    /bin/echo "FALSE"
  else
    if [ "`/bin/echo $list | /bin/grep -w $slot`" = "" ]
    then
      /bin/echo "FALSE"
    else
      /bin/echo "TRUE"
    fi
  fi
}
##################################
# NAME
#    getSwitchFromSlot - Output the switch number which the slot belongs to
#
# SYNOPSIS
#    getSwitchFromSlot slot
#
# DESCRIPTION
#    This function will return the switch that a slot is associated with.
#
# OPERANDS
#     $1 = user input slot number for validation
# RETURN
#     switch number associated with slot or nothing if associated with chassis
#
##################################
getSwitchFromSlot()
{
  if [ "$1" = "" ]
  then
    /bin/echo "NO_PARAM"
    return 0
  fi
  _switch=0
  _num_switch=`getNumSwitches`
  _initial_switch=$FABOS_SWITCHNO
  while [ $_switch -lt $_num_switch ]
  do
    export FABOS_SWITCHNO=$_switch
    if [ "`isValidSlot $1 -switch`" = TRUE ]
    then
      /bin/echo "$_switch"
      export FABOS_SWITCHNO=$_initial_switch
      return 0
    fi
    _switch=`/usr/bin/expr $_switch + 1`
  done
  export FABOS_SWITCHNO=$_initial_switch
  /bin/echo "$_switch"
  return 0
}

##################################
# NAME
#    saveBladeParams - Saves the burnin parameters into non-volitile storage
#
# SYNOPSIS
#    saveBladeParams <slot> [keyword value]
#
# DESCRIPTION
#    save all parameters to the serial eeprom 
#    Variables stores in order are:
#
#    1.  State
#    2.  FileName
#    3.  RunNumber
#    4.  RunStatus
#    5.  CumulativeStatus
#    6.  StartTime
#    7.  RunTime
#    8.  TotalThermal
#    9.  TotalVib
#    10. TotalCmdNumber
#
#    These values are stored as a space seperated string. In this form 
#    they can be parsed with the /usr/bin/cut -d" " command to get the individual
#    values when read.
#
# ALSO SEE
#    getBladeParam
#
# OPERANDS
#    NONE
##################################

saveBladeParams()
{
  if [ "$2" != "" ]
  then
    _cmd=`commandPath getBladeParameters`
    BladeParameters=`$_cmd -slot $1`
    case $2 in
    "STATE" ) 			_data=`/bin/echo $BladeParameters | /bin/sed "s/ST:[[:alpha:]_]\+/ST:$3/g"`;;
    "SCRIPT_NAM" )		_data=`/bin/echo $BladeParameters | /bin/sed "s/SN:[[:alpha:]_]\+/SN:$3/g"`;;
    "RUN_NUMBER" )		_data=`/bin/echo $BladeParameters | /bin/sed "s/RN:[[:digit:]]\+/RN:$3/g"`;;
    "RUN_STATUS" )		_data=`/bin/echo $BladeParameters | /bin/sed "s/RS:[[:alpha:]_]\+/RS:$3/g"`;;
    "CUM_STATUS" )		_data=`/bin/echo $BladeParameters | /bin/sed "s/CS:[[:digit:]]\+/CS:$3/g"`;;
    "START_TIME" )		_data=`/bin/echo $BladeParameters | /bin/sed "s/TS:[[:digit:]]\+/TS:$3/g"`;;
    "RUN_TIME" )		_data=`/bin/echo $BladeParameters | /bin/sed "s/TR:[[:digit:]]\+/TR:$3/g"`;;
    "TEMP_CYCLE" )		_data=`/bin/echo $BladeParameters | /bin/sed "s/TT:[[:digit:]]\+/TT:$3/g"`;;
    "VIB_CYCLE" )		_data=`/bin/echo $BladeParameters | /bin/sed "s/TV:[[:digit:]]\+/TV:$3/g"`;;
    "TOTAL_CMD_NUMBER" )	_data=`/bin/echo $BladeParameters | /bin/sed "s/CT:[[:digit:]]\+/TC:$3/g"`;;
    "CMD_NUMBER" )		_data=`/bin/echo $BladeParameters | /bin/sed "s/CN:[[:digit:]]\+/CN:$3/g"`;;
    "PID" )			_data=`/bin/echo $BladeParameters | /bin/sed "s/PD:[[:digit:]]\+/PD:$3/g"`;;
    esac
  else
    _data="ST:$State SN:$MyName RN:$RunNumber RS:$RunStatus CS:$CumulativeStatus TS:$StartTime TR:$RunTime TT:$TotalThermal TV:$TotalVib TC:$TotalCmdNumber CN:$CmdNumber PD:$$ "
  fi
  _cmd=`commandPath saveBladeParameters`
  $_cmd -slot $1 -offset 0 -data "$_data"
  if [ "$debug_on" != "0" ]
  then
    if [ "$LogFile" != "" ]
    then
      $LogCmd $LogFile "DEBUG saveparam: state $State filename $MyName runnumber $RunNumber runstatus $RunStatus cumstat $CumulativeStatus starttime $StartTime runtime $RunTime totaltime $TotalThermal totalvib $TotalVib totalcmd $TotalCmdNumber cmdnumber $CmdNumber pid $$ "
    fi
  fi
}

##################################
# NAME
#    getBladeParam - returns a burnin parameter from non-volitile storage
#
# SYNOPSIS
#    getBladeParam keyword
#
# DESCRIPTION
#    Get a specific data field from the seeprom. The field to keyword
#    mapping is as follows:
#
#    1.  STATE            State
#    2.  SCRIPT_NAME      FileName
#    3.  RUN_NUMBER       RunNumber
#    4.  RUN_STATUS       RunStatus
#    5.  CUM_STATUS       CumulativeStatus
#    6.  START_TIME       StartTime
#    7.  RUN_TIME         RunTime
#    8.  TEMP_CYCLE       TotalThermal
#    9.  VIB_CYCLE        TotalVib
#    10. TOTAL_CMD_NUMBER TotalCmdNumber
#    11. CMD_NUMBER       CmdNumber
#    12. PID		  PID
#
#    The entire value string is extracrted from non-volitile storage,
#    and parsed for the specified field value. If the keyword is not
#    one of the eleven above, the string invalid parameter is returned
#
#    This function works in concert with saveBladeParams().
#
# OPERANDS
#    $1 slot        Slot to access
#    $2 keyword     Named field of a value desired
#
##################################
getBladeParam()
{
  gBP_result=0
  _cmd=`commandPath getBladeParameters`
  BladeParameters=`$_cmd -slot $1`
  if [ "$debug_on" != "0" ]
  then
    if [ "$LogFile" != "" ]
    then
      $LogCmd $LogFile "DEBUG getparam - $BladeParameters"
    fi
  fi

  #
  # Start by looking in the default position for the data.
  #
  case $2 in
  "STATE")
    fld_str="ST"
    fld_result=`/bin/echo "$BladeParameters" | /bin/grep $fld_str | /usr/bin/cut -d" " -f1`
    ;;
  "SCRIPT_NAME")
    fld_str="SN"
    fld_result=`/bin/echo "$BladeParameters" | /bin/grep $fld_str | /usr/bin/cut -d" " -f2`
    ;;
  "RUN_NUMBER")
    fld_str="RN"
    fld_result=`/bin/echo "$BladeParameters" | /bin/grep $fld_str | /usr/bin/cut -d" " -f3`
    ;;
  "RUN_STATUS")
    fld_str="RS"
    fld_result=`/bin/echo "$BladeParameters" | /bin/grep $fld_str | /usr/bin/cut -d" " -f4`
    ;;
  "CUM_STATUS")
    fld_str="CS"
    fld_result=`/bin/echo "$BladeParameters" | /bin/grep $fld_str | /usr/bin/cut -d" " -f5`
    ;;
  "START_TIME")
    fld_str="TS"
    fld_result=`/bin/echo "$BladeParameters" | /bin/grep $fld_str | /usr/bin/cut -d" " -f6`
    ;;
  "RUN_TIME")
    fld_str="TR"
    fld_result=`/bin/echo "$BladeParameters" | /bin/grep $fld_str | /usr/bin/cut -d" " -f7`
    ;;
  "TEMP_CYCLE")
    fld_str="TT"
    fld_result=`/bin/echo "$BladeParameters" | /bin/grep $fld_str | /usr/bin/cut -d" " -f8`
    ;;
  "VIB_CYCLE")
    fld_str="TV"
    fld_result=`/bin/echo "$BladeParameters" | /bin/grep $fld_str | /usr/bin/cut -d" " -f9`
    ;;
  "TOTAL_CMD_NUMBER")
    fld_str="TC"
    fld_result=`/bin/echo "$BladeParameters" | /bin/grep $fld_str | /usr/bin/cut -d" " -f10`
    ;;
  "CMD_NUMBER")
    fld_str="CN"
    fld_result=`/bin/echo "$BladeParameters" | /bin/grep $fld_str | /usr/bin/cut -d" " -f11`
    ;;
  "PID")
    fld_str="PD"
    fld_result=`/bin/echo "$BladeParameters" | /bin/grep $fld_str | /usr/bin/cut -d" " -f12`
    ;;
  *)
    fld_str="XX"
    gBP_result=fld_result="invalid parameter"
    ;;
  esac

  #
  # Default the field if it is missing
  #
  if [ "$fld_result" = "" ]
  then
#    $LogCmd $LogFile "$slot_message $MyName: Missing the $fld_str data"
    case $fld_str in
    "ST")
      gBP_result=ACTIVE;;
    "SN")
      gBP_result=verify;;
    "RN")
      gBP_result=1;;
    "RS")
      gBP_result=0;;
    "C")
      gBP_result=0;;
    "TS")
      gBP_result=`/bin/date \+\%s`;;
    "TR")
      gBP_result=`/bin/date \+\%s`;;
    "TT")
      gBP_result=0;;
    "TV")
      gBP_result=0;;
    "TC")
      gBP_result=0;;
    "CN")
      gBP_result=1;;
    "PD")
      gBP_result=0;;
    "XX")
      gBP_result="invalid";;
    esac
  else
    #
    # Make sure the data is the correct field requested.
    #
    field=`/bin/echo $fld_result | /usr/bin/cut -d":" -f1 | /usr/bin/cut -d" " -f2`
    if [ "$field" = "$fld_str" ]
    then
      # We have the correct information, so extract the value.
      gBP_result=`/bin/echo $fld_result | /usr/bin/cut -d":" -f2`
    else
      #
      # Determine how many fields there are.
      #
      max_field=`/bin/echo $BladeParameters | /usr/bin/wc -w | /bin/sed "s/ //g"`
      if [ "$max_field" != "12" ]
      then
        $LogCmd $LogFile "$slot_message $MyName: incorrect bladeparameters: $BladeParameters."
      fi
      _pos=1
      while [ $_pos -le $max_field ]
      do
        field=`/bin/echo $BladeParameters | /usr/bin/cut -d":" -f$_pos | /usr/bin/cut -d" " -f2`
        if [ "$field" = "$fld_str" ]
        then
          gBP_result=`/bin/echo $BladeParameters | /usr/bin/cut -d":" -f$_pos | /usr/bin/cut -d" " -f1`
          break
        fi
      _pos=`expr $_pos + 1`
      done
    fi
  fi
  /bin/echo "$gBP_result"
}

##################################
# NAME
#    burninSetName - Set the diagnostics burnin test name.
#
# SYNOPSIS
#    burninSetName name | "-show" slot [ -noupdate ]
#
# DESCRIPTION
#    This command sets the name of the current burnin test.  This value
#    is used in messages to indicate which test was running.
#
#    The mode becomes active as soon as this command is executed.
#    It does not require a reboot to take effect.
#
# OPERANDS
#    The following operands are optional:
#
#    $1   name	   <blank>, "-show": The current burnin name will
#		   be displayed.  Otherwise the burnin name will
#		   be set to the specified string.
#
#    $2   slot	   Which slot to update.
#
#    $3 -noupdate  Inhibits the configuration update upon succesfull
#                  completion of the script
#
##################################
burninSetName()
{
  _x=`/usr/bin/expr "$2" : "\([0-9]\+\)"`
  if { [ -z "$_x" ] || [ $_x -lt $_def_slot ] || [ $_x -gt 100 ] ; }
  then
    _slot=$_def_slot
  else
    _slot=$_x
  fi

  #
  # Program customization
  #

  _config_string="diag.mode.burnin.$_slot.name" # config string to update
  if [ `isMultiBlade` = TRUE ]
  then
    _config_name="Slot $_slot burnin name"  # User name of config.
  else
    _config_name="burnin name"  # User name of config.
  fi
  if [ "$4" != "" ]
  then
    _config_default="$4"        # default value
  else
    _config_default="burnin"    # default value
  fi
  _config_mode=$STRING          # config mode

  #
  # burninname()
  #
  checkForShow "$_config_name" "$_config_string" "$_config_mode" "$_config_default" $1

  script_name=common_test_list
  _ok=0
  for _f in "$script_name" "$FABOSHOME/share/$script_name" "$FABOSHOME/sbin/$script_name" "$home/$script_name"
  do
    if [ -f $_f ]
    then
      if [ "`/bin/grep SCRIPTNAME $_f | /usr/bin/cut -d'	' -f3- | /bin/grep -w $1`" != "" ]
      then
        _ok=1
        break;
      fi
    fi
  done
  if [ $_ok = 0 ]
  then
    err "$1 is not found"
    exit 3
  fi

  #
  # Make sure this is good setting
  #
  _ok=`eval testConfig $_config_string $_config_mode`
  if [ $? != 0 ]
  then
    exit 3
  fi

  setConfig $_config_string $_config_mode "$1"
  /bin/echo "$_config_name is now $1"

  if [ "$3" != "-noupdate" ]
  then
    updateConfig
  fi
}

##################################
# NAME
#    burninSetLevel - Set the diagnostics burnin level.
#
# SYNOPSIS
#    burninSetLevel [ level | "-show" ] [ -noupdate ]
#
# DESCRIPTION
#    This command selects or displays the burnin level.  When the
#    burnin level is set to a value other than 0, the diagnostics
#    daemon program will perform burnin testing in place of POST 2
#    each time a switch blade is powered on.
#
#    The mode becomes active as soon as this command is executed.
#    It does not require a reboot to take effect.
#
#    When a burnin level other than 0 is selected, the actual behavior
#    is determined by the configuration of the diagnostics daemon and
#    the burnin scripts that are run.
#
#    Note:  This command sets both diag.mode.burnin.level and
#    diag.mode.burnin.
#
# OPERANDS
#    The following operand is optional:
#
#       level		Integer value or "-show".  If level
#			is missing or the string "-show" then
#			the current burnin level will be displayed.
#			If an integer is provided then the burnin
#			level will be set to that value.
#
#                       If no parameter is provided the current
#			debug level setting will be displayed.
#
#   -noupdate  Inhibits the configuration update upon succesfull
#              completion of the script
#
#       -noupdate      
#
# SEE ALSO
#     diagsetburnin, diagresetburnin, setburninscript
#
##################################
burninSetLevel()
{
  #
  # Program customization
  #
  _config_string="diag.mode.burnin.level" # config string to update
  _config_string2="diag.mode.burnin"      # config string to update
  _config_name="Burnin level"             # User name of config.
  _config_default=0                       # default value
  _config_mode=$INTEGER                   # config mode

  ########################
  #
  # burninlevel()
  #

  checkForShow "$_config_name" "$_config_string" "$_config_mode" "$_config_default" $1 
  _val=`getValue $1`
  if [ $? != 0 ]
  then
    exit 3
  fi

  if [ $_val -lt 0 ]
  then
    err "Burnin level must be >= 0"
    exit 3
  fi

  # set level
  setConfig $_config_string $_config_mode $_val
  # /bin/echo $_config_name is now $_val.

  # set enable
  if [ $_val != 0 ]
  then
    _val=1
  fi
  setConfig $_config_string2 $_config_mode $_val
  if [ "$_val" = "1" ]
  then
    _res=Enabled
  else
    _res=Disabled
  fi
  /bin/echo Burnin mode is $_res.
}

##################################
# NAME
#    burninSetScript - Set diagnostics post/burnin scripts.
#
# SYNOPSIS
#    burninSetScript type event [ path ] [ -noupdate ]
#
# DESCRIPTION
#    This command selects the scripts to be run by the diagnostics
#    daemon for various events.  When the specified event occurs for
#    the specified board type then the script will be run.
#
#
#
#    The mode becomes active as soon as this command is executed.
#    It does not require a reboot to take effect.
#
# OPERANDS
#
#    type      Board type.  Possible values include "CP", "GP", "WAN",
#	       and "SWITCH".  For normal operations on switch blades
#	       "SWITCH" should be specified.
#
#    event     Event type.  Valid settings are "POST1", "POST2" and
#	       "BURNIN"
#
#    path      Path name for script to run.  If this parameter is ommited
#	       or contains the string "-show" then the current setting will
#	       be displayed.
#
#   -noupdate  Inhibits the configuration update upon succesfull
#              completion of the script
#
#
# Program customization
#
##################################
burninSetScript()
{
  _typ=$1
  _event=$2

  # this change was made to handle the per platform burnin implementation.
  # This removed the per type differentiation for burnin
  if [ "$2" = "BURNIN" ]
  then
    _config_string='diag.script.$_event'      	  # config string to update
  else
    _config_string='diag.script.$_typ.$_event'    # config string to update
  fi
  _config_name='$_event script for $_typ boards'  # User name of config.
  _config_default="doburnin"                      # default value
  _config_mode=$STRING                            # config mode

  #
  # burninscript
  #

  _config_string=`eval /bin/echo $_config_string`
  _config_name=`eval /bin/echo $_config_name`
  checkForShow "$_config_name" "$_config_string" "$_config_mode" "$_config_default" "$3"

  setConfig $_config_string $_config_mode $3
  /bin/echo $_config_name is now $3

  if [ "$4" != "-noupdate" ]
  then
    updateConfig
  fi
}
##################################
# NAME
#    getNumAvailableScripts - Output the number of scripts that
#                             contain the specified keyword
#
# SYNOPSIS
#    getNumAvailableScripts Keyword [-config | -noconfig ]
#
# DESCRIPTION
#    Output the number of scripts available that support the specified
#    keyword activity. Include the current script if there has been
#    one selected.
#
#    The determination is triggered by the presence of the keyword
#    $1 in the files in $FABOSHOME/share. If there is a variable that
#    holds the current selection, then $2 contains "-config". If there
#    is no config variable then $2 = "-noconfig"
#
# OPERANDS
#    $1 - "-config" "-noconfig"
#    $2 - List of scripts
##################################
getNumAvailableScripts()
{
  add_config=$1
  shift
  num_scripts=1
  list=`/bin/echo $*`
  for word in $list
  do
    if [ $word != '$INTEGER' -a $word != '$STRING' ]
    then
      num_scripts=`expr $num_scripts + 1`
    else
      break
    fi
  done
  if [ "$add_config" = "-config" ]
  then
    adjust=0
  else
    # do not count flag and name field
    adjust=1
  fi
  num_scripts=`expr $num_scripts - $adjust`
  /bin/echo $num_scripts
}

##################################
# NAME
#    showAvailableScripts - Output the names of scripts that
#                           contain the specified keyword
#
# SYNOPSIS
#    showAvailableScripts Keyword [ -noconfig | configvar default ]
#
# DESCRIPTION
#    Output a list of the scripts available that support burnin activity.
#    Include the current burnin script if there has been one selected.
#
#    The determination is triggered by the presence of the keyword
#    $1 in the files in $FABOSHOME/share. If there is a variable that
#    holds the current selection, then the variable name is specified
#    in $2 and the default value in $3. The config variables are
#    assumed to be of type $STRING. 
#
#    If there is no config variable then $2 = "-noconfig"
#
# OPERANDS
#    $1 - Config variable that contains the current setting 
#	  (if none then "-noconfig")
#    $2 - List of scripts
##################################
showAvailableScripts()
{
  # Add the EXIT choice
  /bin/echo " 0) EXIT"
  add_config=$1
  shift
  if [ "$add_config" = "-noconfig" ]
  then
    _script=""
    list=`/bin/echo $*`
  else
    default=$1
    shift
    list=`/bin/echo $*`
    _script=`getConfig "$add_config" $STRING $default`
  fi

  _num=1
  if [ ! -z "$_script" ]
  then
    /bin/echo " $_num) -current ($_script)"
    _num=`/usr/bin/expr $_num + 1`
  fi
  
  for name in $list
  do
    if [ $name = '$INTEGER' -o $name = '$STRING' ]
    then
      break
    fi
    case $name in
      N);;
      SCRIPTNAME);;
      *) /bin/echo " $_num) $name";;
    esac
    _num=`/usr/bin/expr $_num + 1`
  done
}

##################################
# NAME
#    getScriptSelection - Output the name of nth script that
#                         contain the specified keyword
#
# SYNOPSIS
#    getScriptSelection N Keyword [ -noconfig | configvar default ]
#
# DESCRIPTION
#    Output the Nth script available that supports keyworded activity.
#    Include the current burnin script if there has been one selected.
#
#    The determination is triggered by the presence of the keyword
#    $1 in the files in $FABOSHOME/share. If there is a variable that
#    holds the current selection, then the variable name is specified
#    in $2 and the default value in $3. The config variables are
#    assumed to be of type $STRING. 
#
#    If there is no config variable then $2 = "-noconfig"
#
# OPERANDS
#    $1 - Nth selection
#    $2 - Config variable that contains the current setting 
#	  (if none then "-noconfig")
#    $3 - default value for config variable
#    $4 - List of scripts
##################################
getScriptSelection()
{
  selection=$1
  add_config=$2
  shift
  shift
  if [ "$add_config" = "-noconfig" ]
  then
    _script=""
    list=`/bin/echo $*`
    max_choice=`getNumAvailableScripts $add_config $list`
  else
    default=$1
    shift
    list=`/bin/echo $*`
    max_choice=`getNumAvailableScripts $add_config $default $list`
    _script=`getConfig "$add_config" $STRING $default`
    list="$_script $list"
  fi
  # Add the EXIT choice
  list="EXIT $list"
  max_choice=`/usr/bin/expr $max_choice + 1`

  # get the selected choice
  if [ $selection -lt 0 -o $selection -gt $max_choice ]
  then
    /bin/echo "Invalid"
  else
    _num=0
    for name in $list
    do
      if [ $_num -eq $selection ]
      then
        /bin/echo "$name"
        break
      fi
      _num=`/usr/bin/expr $_num + 1`
    done
  fi
}
# Handle the Gen6 FOS Quiescent here
do_sw_quiescent()
{
  _cmd=`commandPath exec_diag`
  lbMode=`getConfig diag.verify.min_lb_mode $INTEGER 2`
  if [ `supportsQuiescent` = TRUE ]
  then
    case $2 in
    "TRUE" )
      /bin/echo "Configuring DIAG_START flag on slot $1"
      $_cmd usdsetupfortest --slot $1 -test_step 2 -post 0 -lb_mode $lbMode > /dev/null 2>&1;;

    "FALSE" )
      /bin/echo "Configuring DIAG_END flag on slot $1"
      $_cmd usdsetupfortest --slot $1 -test_step 3 -post 0 -lb_mode $lbMode > /dev/null 2>&1;;
    esac
  fi
  ret=$?
  return $ret
}
# Gen6/Exec_diag3 embedded platforms needs to be added to the below list 
# to send DIAGS and DIAGD during Systemverification.
supportsQuiescent()
{
  _abc=`/sbin/sin | \
    /bin/grep Platform | \
    /usr/bin/cut -d" " -f2 | \
    /usr/bin/cut -d"," -f1`
  case $_abc in
  "SWBD161" ) /bin/echo "TRUE";;
  "SWBD162" ) /bin/echo "TRUE";;
  "SWBD165" ) /bin/echo "TRUE";;
  "SWBD166" ) /bin/echo "TRUE";;
  "SWBD169" ) /bin/echo "TRUE";;
  "SWBD170" ) /bin/echo "TRUE";;
  "SWBD172" ) /bin/echo "TRUE";;
  "SWBD173" ) /bin/echo "TRUE";;
  "SWBD178" ) /bin/echo "TRUE";;
  * )        /bin/echo "FALSE";;
  esac
}
