#!/bin/sh

# This script is to upgrade sysFPGA on BES or FS8-18. You need to input AP Blade No of the upgrade file. 
# Refer to usage for more details. Contact Brocade Support for any issues faced during this upgrade.
# For Any issues Contact Brocade Global Support 

# Don't allow CTRL+C to Kill the current exeuction
trap "" 1 2 3 6 15

# Declaration of Global Variables Here - The default values 
# for FPGA upgrade environment are declared as Global variables.

SYSFPGA_TASK=''                                          # Any of prepare, upgrade, clear_prepare, verify_upgrade
AP_BLADE_NO=0                                            # If AP_BLADE_NO = 0 then BES, otherwise Chassis
BP_IP=''                                                 # Set BP IP Here for BES and FS8-18

BES_BLADE_PL_CHECK_RV=0;                                 # BES FS8-18 Platform Return Value

OLD_FPGA_PATH_IN_BP='/boot/lancectl_rev67.xsvf'          # Path of the FPGA Version 0x67 Version file in BP - Hard Coded
NEW_FPGA_PATH_IN_BP='/boot/LanceCTL_rev77_20100603.xsvf' # Path of the FPGA Version 0x77 Version file in BP - Hard Coded
JTAGPGM_FILE_PATH_IN_BP='/fabos/bin/jtagpgm'             # JTAGPGM File path in BP - Hard Coded
PREEMPT_FOS_FILE_PATH='/etc/preempt_fos'                 # PREEMPT FOS File Path - Hard Coded
BP_IP_FILE_PATH='/etc/bp_ip_addr'                        # BP IP Address File PATH in BP - Hard Coded
PT_TOTAL_WAIT_TIME=900									 # Total wait time 
PT_TOTAL_POLL_TIME=15									 # Total Poll time 

OLD_FPGA_VERSION='0x6'                                   # Old FPGA Version - Hard Coded
NEW_FPGA_VERSION_STD='0x7'                               # Standard New FPGA Version - Hard Coded

CURRENT_FPGA_VERSION=''                                  # Actual FPGA version after upgrade
FPGA_UPGRADE_MESSAGES=''                                 # FPGA Upgrade Messages to be captured in this variable

LOG_FILE=''                                              # Logging of current execution
LOG_DIR='log_fpga_upgrade'	     						 # Log directory from PWD

# 	Exit Codes and Their meanings.
# Code 0 - Upgrade is successful or already done
# Code 1 - Argument issues
# Code 2 - Improper ap blade slot no
# Code 3 - This build doesnot have fpga upgrade utility
# Code 4 - Isshe with prepare. run clear_prepare and prepare
# Code 5 - Current FPGA Version is unknown not ox67 or ox77
# Code 6 - Switch is in unexpcted state

usage() {	# sysFPGA upgrade syntax helper

    script_name=`basename $0`
    echo
    echo "###############################################################"
    echo "# sysFPGA Upgrade Utility - Usage Instructions                #"
    echo "###############################################################"
    echo "#                                                             #"
    echo "# This script is to upgrade the sysFPGA on a BES or an        #"
    echo "# FS8-18. The complete upgrade process comprises multiple     #"
    echo "# steps. Please consult the example given.                    #"
    echo "#                                                             #"
    echo "# Synopsis: $script_name <ARG> [VALUE]                  #"
    echo "#                                                             #"
    echo "# ARG:                                                        #"
    echo "#   prepare -                                                 #"
    echo "#    Prepares the BES/FS8-18 for an upgrade. VALUE should be  #"
    echo "#    the slot number if FS8-18, else 0 if a BES. Giving VALUE #"
    echo "#    as empty also means a BES.                               #"
    echo "#    If the existing sysFPGA version is latest/updated,       #"
    echo "#    running the prepare step informs that an upgrade is not  #"
    echo "#    required and the utility exits.                          #"
    echo "#    e.g: $script_name prepare 0                        #"
    echo "#           Prepare a BES for upgrade.                        #"
    echo "#    e.g: $script_name prepare                          #"
    echo "#           Prepare a BES for upgrade.                        #"
    echo "#    e.g: $script_name prepare 2                        #"
    echo "#           Prepare a FS8-18 present in slot 2.               #"
    echo "#                                                             #"
    echo "#  clear_prepare -                                            #"
    echo "#    Undo the prepare step if a prepare has been done and it  #"
    echo "#    is not desired to continue with an upgrade. VALUE should #"
    echo "#    be the slot number if FS8-18, else 0 if a BES. Giving    #"
    echo "#    VALUE as empty also means a BES.                         #"
    echo "#    e.g: $script_name clear_prepare 0                  #"
    echo "#           Undo the prepare step done on a BES.              #"
    echo "#    e.g: $script_name clear_prepare                    #"
    echo "#           Undo the prepare step done on a BES.              #"
    echo "#    e.g: $script_name clear_prepare 2                  #"
    echo "#           Undo the prepare step done on a FS8-18 in slot 2. #"
    echo "#                                                             #"
    echo "#  upgrade -                                                  #"
    echo "#    Perform a sysFPGA upgrade on a BES or FS8-18. Upgrade    #" 
    echo "#    should be attempted only after a prepare. VALUE should   #"
    echo "#    be the slot number if FS8-18, else 0 if a BES. Giving    #"
    echo "#    VALUE as empty also means a BES.                         #"
    echo "#    e.g: $script_name upgrade 0                        #"
    echo "#           Perform a sysFPGA upgrade on a BES.               #"
    echo "#    e.g: $script_name upgrade                          #"
    echo "#           Perform a sysFPGA upgrade on a BES.               #"
    echo "#    e.g: $script_name upgrade 2                        #"
    echo "#           Perform a sysFPGA upgrade on a FS8-18 in slot 2.  #"
    echo "#                                                             #"
    echo "#  verify_upgrade -                                           #"
    echo "#    Verifies the status of the upgrade step performed. This  #"
    echo "#    should be run after the upgrade step.                    #"
    echo "#    e.g: $script_name verify_upgrade 0                 #"
    echo "#           Perform a verification of the previously          #"
    echo "#           performed upgrade on a BES.                       #"
    echo "#    e.g: $script_name verify_upgrade                   #"
    echo "#           Perform a verification of the previously          #"
    echo "#           performed upgrade on a BES.                       #"
    echo "#    e.g: $script_name verify_upgrade 2                 #"
    echo "#           Perform a verification of the previously          #"
    echo "#           performed upgrade on a FS8-18 in slot 2.          #"
    echo "#                                                             #"
    echo "###############################################################"
    echo "# Recommended Usage:                                          #"
    echo "#                                                             #"
    echo "# The recommended order of running the multiple steps of      #"
    echo "# the complete upgrade process is as follows.                 #"
    echo "# Step 1) prepare                                             #"
    echo "# Step 2) upgrade                                             #"
    echo "# Step 3) verify_upgrade                                      #"
    echo "# If the user wishes to cancel the prepare step which he      #"
    echo "# executed and if he doesn't wish to proceed with the upgrade #"
    echo "# and the verify_upgrade steps, he should run a clear_prepare #"
    echo "# step after the prepare step to undo the effects of prepare. #"
    echo "#                                                             #"
    echo "###############################################################"
    echo "# Example 1:                                                  #"
    echo "#                                                             #"
    echo "# Consider the sysFPGA upgrade is to be performed on a BES.   #"
    echo "# The upgrade steps are as follows assuming that a            #"
    echo "# clear_prepare is not attempted after the prepare step.      #"
    echo "#                                                             #"
    echo "# Step 1) ./$script_name prepare 0                      #"
    echo "# Step 2) ./$script_name upgrade 0                      #"
    echo "# Step 3) ./$script_name verify_upgrade 0               #"
    echo "#                                                             #"
    echo "###############################################################"
    echo "# Example 2:                                                  #"
    echo "#                                                             #"
    echo "# Consider the sysFPGA upgrade is to be performed on an       #"
    echo "# FS-18 present in the chassis at slot 8. The upgrade steps   #"
    echo "# are as follows assuming that a clear_prepare is not         #"
    echo "# attempted after the prepare step.                           #"
    echo "#                                                             #"
    echo "# Step 1) ./$script_name prepare 8                      #"
    echo "# Step 2) ./$script_name upgrade 8                      #"
    echo "# Step 3) ./$script_name verify_upgrade 8               #"
    echo "#                                                             #"
    echo "###############################################################"
    echo 
}

warning() {	# Warning message before the upgrade
    echo ;
    echo "#############################################################################"
    echo "# ***WARNING***                                                             #"
    echo "# - You are going to upgrade the sysFPGA image of the BES / FS8-18.         #"
    echo "# - During any phase of this upgrade process, the BES / FS8-18 is NOT       #"
    echo "#   expected to serve any I/O.                                              #"
    echo "# - There will be multiple ***REBOOTS*** during the upgrade process. Please #"
    echo "#   plan accordingly.                                                       #"
    if [ $SYSFPGA_TASK = 'upgrade' ]; then
        echo "#   Please do NOT **Power-Cycle** the box during sysFPGA image upgrade.     #"
    fi
    echo "#############################################################################"
    echo ; 	echo -n "Do you want to continue? \"yes\" to continue, \"no\" to abort: ";
    read confirm
    echo ;
    if [ -z "$confirm" ]; then
        echo "Upgrade cancelled."; echo;
        exit 7;
    fi
    if [ "$confirm" != 'yes' ]; then
        echo "Upgrade cancelled."; echo;
        exit 7;
    fi
}

get_switch_current_state() {	# Return 0 if online, Return 1 if faulty and 2 otherwise
    echo "In function: get_switch_current_state" >> $LOG_FILE
    echo "	BES_BLADE_PL_CHECK_RV Value is: $BES_BLADE_PL_CHECK_RV" >> $LOG_FILE
    echo "	FS8-18 blade Slot No is: $AP_BLADE_NO" >> $LOG_FILE
    if [ $BES_BLADE_PL_CHECK_RV -eq 0 ]; then 
        if [ "`switchshow | grep "switchState.*Online"`" ]; then
            echo "	Switch is Online." >> $LOG_FILE
            return 0		# Switch is Online
        elif [ "`switchshow | grep "switchState.*Offline"`" ]; then
            echo "	Switch is Offline." >> $LOG_FILE
            return 0		# Switch is Offline
        elif [ "`switchshow | grep "switchState.*Faulty"`" ]; then
            echo "	Switch is Faulty." >> $LOG_FILE
            return 1		# This is for Faulty
        else
            local echo_val=`switchshow`
            echo "	$echo_val" >> $LOG_FILE
            return 2		# Switch is other than Fault.
        fi
    elif [ $BES_BLADE_PL_CHECK_RV -eq $AP_BLADE_NO ]; then
        if [ "`slotshow | grep "$AP_BLADE_NO.*AP BLADE.*ENABLED"`" ]; then
            echo "	Blade is Enabled." >> $LOG_FILE
            return 0		# this is for Online
        elif [ "`slotshow | grep "$AP_BLADE_NO.*AP BLADE.*FAULTY"`" ]; then
            echo "	Blade is Faulty." >> $LOG_FILE
            return 1		# this is for Faulty
        else
            local echo_val=`slotshow` >> $LOG_FILE
            echo "	$echo_val" >> $LOG_FILE			
            return 2		# Other than fault.
        fi
    else
        echo "Neither a BES nor a FS8-18. Probably you wont be hitting this message.";
    fi
}

mace_lance_platform_check() {	# Return 0 If BES, Blade No if FS8-18, 100 If any issues.
    echo "In function: mace_lance_platform_check" >> $LOG_FILE
    if [ "`slotshow | grep "Not supported on this platform"`" ]; then
        return 0;			# This is BES
    else
        if [ "`slotshow | grep "$AP_BLADE_NO.*AP BLADE.*43"`" ]; then
            return $AP_BLADE_NO		# this is FS8-18
        else
            local echo_val=`slotshow`
            echo "	$echo_val" >> $LOG_FILE
            return 100		# This is not BES or Issue with Ap Blade
        fi
    fi
}

find_bp_ip() {	# Set BP IP to $BP_IP Variable. Exit otherwise.
    echo "In function: find_bp_ip" >> $LOG_FILE
    echo
    if [ $AP_BLADE_NO -eq 0 ]; then
        if [ $BES_BLADE_PL_CHECK_RV -ne 0 ]; then
            echo "[FAIL]	This is not a BES. Please give the correct Slot No."
            echo "	This is not a BES. Please give the correct Slot No." >> $LOG_FILE
            exit 2;
        else
            #echo "This is BES";
            BP_IP="127.1.10.1";
            echo "	BES BP_IP : $BP_IP" >> $LOG_FILE
        fi
    else 
        if [ $AP_BLADE_NO -eq $BES_BLADE_PL_CHECK_RV ]; then
            #echo "This is Chassis";
            # We are always going to upgrade AP Blade and Blade ID should be 43. Ok?
            local_cp=`hashow | grep Local.*CP0`
            if [ $? -ne 0 ]; then
                echo "	local_cp output: $local_cp and exit_status is $?" >> $LOG_FILE
                echo "[FAIL]	sysFPGA upgrade can be done only when Active CP is CP0. ";
                exit 2;
            else
                echo "	local_cp output: $local_cp and exit_status is $?" >> $LOG_FILE
            fi
            local_cp_value=`expr "$local_cp" : '.*Slot \([0-9]*\)'`
            BP_IP="127.1.`expr $local_cp_value + 10`.`expr $AP_BLADE_NO + 1`";	# Calculate BP IP and set it in BP_IP
            echo "	BES BP_IP : $BP_IP" >> $LOG_FILE
        else
            echo "[FAIL]	FS8-18 blade Slot No specified is incorrect.";
            exit 2;
        fi
    fi
}

find_current_fpga_version() { # Find current FPGA Version
    echo "In function: find_current_fpga_version" >> $LOG_FILE
    #local fpga_version_string=`rsh -l root $BP_IP '/fabos/bin/jtagpgm -i | grep "PB FPGA Release"'`
    local fpga_version_string=''
    if [ $BES_BLADE_PL_CHECK_RV -ne 0 ]; then
        fpga_version_string=`bladevershow $AP_BLADE_NO| grep -a "FPGA version"`
    else
        fpga_version_string=`cat /proc/fabos/blade/0/info| grep -a "FPGA version"`
    fi
    echo "	FPGA VERSION STRING: $fpga_version_string" >> $LOG_FILE
    #CURRENT_FPGA_VERSION=`expr "$fpga_version_string" : '.*PB FPGA Release : \(.*[0-9]*\)'`
    CURRENT_FPGA_VERSION=`expr "$fpga_version_string" : '.*FPGA version.*\(0x[0-9]*\)'`
    echo "	Current sysFPGA Version: :$CURRENT_FPGA_VERSION:" >> $LOG_FILE
}

check_current_fpga_version() { # Get 0 if old version, 1 if new and 2 if anything else
    echo "In function: check_current_fpga_version" >> $LOG_FILE
    echo "	CURRENT FPGA VERSION: $CURRENT_FPGA_VERSION" >> $LOG_FILE
    if [ "$CURRENT_FPGA_VERSION" = "$OLD_FPGA_VERSION" ]; then
        return 0;	# We can upgrade
    elif [ "$CURRENT_FPGA_VERSION" = "$NEW_FPGA_VERSION_STD" ]; then
        return 1;	# No Need to upgrade, Hauh?
    else
        return 2;	# Any other unknown versions.
    fi
}

upgrade_fpga() {	# Upgrade FPGA and set the console logs to the global var FPGA_UPGRADE_MESSAGES
    echo "In function: upgrade_fpga" >> $LOG_FILE
    echo "[INFO]	Upgrade in progress. Please do NOT POWER-CYCLE during upgrade."	
    #>/fabos/bin/jtagpgm -f FS8-18CTL_rev77_20100603.xsvf  -v 1
    local upgrade_msg=`rsh -l root $BP_IP "$JTAGPGM_FILE_PATH_IN_BP -f $NEW_FPGA_PATH_IN_BP -v 1 2>&1"`
    echo >> $LOG_FILE
    echo "	FULL UPGRADE MESSAGE: $upgrade_msg" >> $LOG_FILE
    echo >> $LOG_FILE
    echo "[INFO]	Upgrade completed."
    echo "	Upgrade completed." >> $LOG_FILE
    FPGA_UPGRADE_MESSAGES=$upgrade_msg
    return 0;
}
downgrade_fpga() {	# Upgrade FPGA and set the console logs to the global var FPGA_UPGRADE_MESSAGES
    echo "In function: downgrade fpga" >> $LOG_FILE
    echo "[INFO]	Downgrade in progress. Please do NOT POWER-CYCLE during upgrade."	
    #>/fabos/bin/jtagpgm -f FS8-18CTL_rev77_20100603.xsvf  -v 1
    local upgrade_msg=`rsh -l root $BP_IP "$JTAGPGM_FILE_PATH_IN_BP -f $OLD_FPGA_PATH_IN_BP -v 1 2>&1"`
    echo >> $LOG_FILE
    echo "	FULL UPGRADE MESSAGE: $upgrade_msg" >> $LOG_FILE
    echo >> $LOG_FILE
    echo "[INFO]	Upgrade completed."
    echo "	Upgrade completed." >> $LOG_FILE
    FPGA_UPGRADE_MESSAGES=$upgrade_msg
    return 0;
}

validate_fpga_upgrade_message() {	# Return 0 if upgrade is success, 1 if success is not found 2 if failure is found
    echo "In function: validate_fpga_upgrade_message" >> $LOG_FILE
    echo "[INFO]	Verifying the sysFPGA upgrade status.";
    echo >> $LOG_FILE
    echo "	FULL UPGRADE MESSAGE: $FPGA_UPGRADE_MESSAGES" >> $LOG_FILE	
    echo >> $LOG_FILE	

    echo "$FPGA_UPGRADE_MESSAGES" | grep -q "Programmed.*file successfully"
    exit_status=$?;

    echo "	Exit status of the sysFPGA image upgrade check $exit_status." >> $LOG_FILE

    if [ $exit_status -eq 0 ]; then
        echo "[SUCCESS]	***sysFPGA Upgrade Successful***. Proceeding..."
        echo "	***sysFPGA Upgrade Successful***. Proceeding..." >> $LOG_FILE		
        echo ;
    else
        echo "[FAIL]	sysFPGA upgrade successful message not found."
        echo "	sysFPGA upgrade successful message not found." >> $LOG_FILE		
        echo ; 	echo "Checking for sysFPGA failure message - $FPGA_UPGRADE_MESSAGES"; echo ; 	
        exit_status=1;
        echo "$FPGA_UPGRADE_MESSAGES" | grep -q "Failed to program"
        if [ $? -eq 0 ]; then
            exit_status=2;
        fi
    fi
    return $exit_status;	
}

check_jtagpgm_is_available_in_bp() { # Check if the build is having upgrade utility
    echo "In function: check_jtagpgm_is_available_in_bp" >> $LOG_FILE
    local jtag_file=`rsh -l root $BP_IP "ls $JTAGPGM_FILE_PATH_IN_BP 2>&1"`
    echo "	JTAG FILE availability in BP: $jtag_file" >> $LOG_FILE
    if [ "$JTAGPGM_FILE_PATH_IN_BP" = "$jtag_file" ]; then
        echo "[INFO]	This build has the utility to upgrade sysFPGA. Proceeding..."
    else
        echo "[FAIL]	This build doesn't have the sysFPGA upgrade utility."
        echo "Please upgrade to the build where you have the sysFPGA upgrade utility. "
        exit 3;
    fi
}

check_fpga_upgrade_image_is_available_in_bp() { # Check if the build is having fpga upgrade image
    echo "In function: check_fpga_upgrade_image_is_available_in_bp" >> $LOG_FILE
    local fpga_image=`rsh -l root $BP_IP "ls $NEW_FPGA_PATH_IN_BP 2>&1"`
    echo "	Searching for upgrade image in BP: $fpga_image" >> $LOG_FILE
    if [ "$NEW_FPGA_PATH_IN_BP" = "$fpga_image" ]; then
        echo "[INFO]	New sysFPGA image is located in BP. Proceeding...";
    else
        echo "[FAIL]	This build doesn't have the sysFPGA image required for upgradation."
        echo "Please upgrade to the build where you have the sysFPGA image required for upgrade. "
        exit 3;
    fi
}

check_preempt_fos_file_available_in_bp() {	# Check Preemptfos file in bp, exit otherwise
    echo "In function: check_preempt_fos_file_available_in_bp" >> $LOG_FILE
    local preemptfos_file=`rsh -l root $BP_IP "ls $PREEMPT_FOS_FILE_PATH 2>&1"`
    echo "	Checking preempt fos file in BP: $preemptfos_file" >> $LOG_FILE	
    #local preemptfos_file=`echo $preemptfos_file | sed 's/://g'`
    if [ "$PREEMPT_FOS_FILE_PATH" = "$preemptfos_file" ]; then
        echo "[INFO]	$PREEMPT_FOS_FILE_PATH is present in BP."
        echo "[FATAL]	Have you run \"prepare\" already? If yes, please run"
        echo "[FATAL]	\"clear_prepare\" to revert to the normal state. Else proceed with the \"upgrade\". "
        exit 4;
    fi
}

create_preempt_fos_file_in_bp() {	# Create preemptfos file in bp with content 1
    echo "In function: create_preempt_fos_file_in_bp" >> $LOG_FILE
    `rsh -l root $BP_IP "echo 1 > $PREEMPT_FOS_FILE_PATH"`
    echo "	$PREEMPT_FOS_FILE_PATH file is created with content 1 in BP." >> $LOG_FILE
}

delete_preempt_fos_file_in_bp() {	# Delete preemptfos file in bp
    echo "In function: delete_preempt_fos_file_in_bp" >> $LOG_FILE
    local val=`rsh -l root $BP_IP "rm $PREEMPT_FOS_FILE_PATH 2>&1"`
    echo "[INFO]	Deleted $PREEMPT_FOS_FILE_PATH file in BP. Proceeding..." ; 
    echo "	Deleted $PREEMPT_FOS_FILE_PATH file in BP." >> $LOG_FILE
}

check_content_of_preempt_fos_file_in_bp() {	# Check the content of the preempt fos file in bp.
    echo "In function: check_content_of_preempt_fos_file_in_bp" >> $LOG_FILE
    local preemptfos_file=`rsh -l root $BP_IP "cat $PREEMPT_FOS_FILE_PATH 2>&1"`
    echo "	Content of preempt fos file: $preemptfos_file" >> $LOG_FILE
    if [ "$preemptfos_file" -eq "1" ]; then
        echo "[INFO]	Created $PREEMPT_FOS_FILE_PATH. Proceeding..."; 	
        echo "	Content of the file $PREEMPT_FOS_FILE_PATH is correct." >> $LOG_FILE
    else
        echo "[FATAL]	Content of the file $PREEMPT_FOS_FILE_PATH is wrong.";
        echo "	Content of the file $PREEMPT_FOS_FILE_PATH is wrong." >> $LOG_FILE
        echo "Run \"clear_prepare\", \"prepare\" and proceed with \"upgrade\". "
        exit 4;
    fi
}

check_bp_ip_addr_file_in_bp() {	# Check bp_ip_addr file availability in BP
    local bpipaddr_file=`rsh -l root $BP_IP "ls $BP_IP_FILE_PATH 2>&1"`
    echo "	Checking bp ip addr file in BP: $bpipaddr_file" >> $LOG_FILE
    if [ "$BP_IP_FILE_PATH" = "$bpipaddr_file" ]; then
        echo "[INFO]	$BP_IP_FILE_PATH is present in BP."
        echo "Have you run \"prepare\" already? If yes please run"
        echo "[FAIL]	\"clear_prepare\" to revert to the normal state. Else continue with \"upgrade\".  "
        exit 4;
    fi
}

create_bp_ip_addr_file_in_bp() {	# Create bp_ip_addr file with content as $BP_IP in BP
    echo "In function: create_bp_ip_addr_file_in_bp" >> $LOG_FILE
    `rsh -l root $BP_IP "echo $BP_IP > $BP_IP_FILE_PATH"`
    echo "	Created $BP_IP_FILE_PATH file with content $BP_IP in BP." >> $LOG_FILE
}

delete_bp_ip_addr_file_in_bp() {	# Delete the file bp_ip_addr in BP
    echo "In function: delete_bp_ip_addr_file_in_bp" >> $LOG_FILE
    local val=`rsh -l root $BP_IP "rm $BP_IP_FILE_PATH 2>&1"`
    echo "[INFO]	Deleted $BP_IP_FILE_PATH file in BP. Proceeding..."
    echo "Deleted $BP_IP_FILE_PATH file in BP." >> $LOG_FILE
}

check_content_of_bp_ip_addr_file_in_bp() { 	# Validate the content of the file in BP
    echo "In function: check_content_of_bp_ip_addr_file_in_bp" >> $LOG_FILE
    local bpip_file=`rsh -l root $BP_IP "cat $BP_IP_FILE_PATH 2>&1"`
    echo "	Content of the bp ip addr file: $bpip_file" >> $LOG_FILE
    if [ "$bpip_file" = "$BP_IP"  ]; then
        echo "[INFO]	$BP_IP_FILE_PATH has been created. Proceeding..." ;
        echo "	Content of the file $BP_IP_FILE_PATH is correct. Proceeding..." >> $LOG_FILE
    else
        echo "[FATAL]	Content of the file $BP_IP_FILE_PATH is wrong."
        echo "	Content of the file $BP_IP_FILE_PATH is wrong." > $LOG_FILE
        echo "Run \"clear_prepare\" and \"prepare\" to proceed \"upgrade\". " ; 
        echo
        exit 4;
    fi
}

reboot_blade() {
    echo "In function: reboot_blade" >> $LOG_FILE
    echo "	BES / FS8-18? $BES_BLADE_PL_CHECK_RV" >> $LOG_FILE	
    if [ $BES_BLADE_PL_CHECK_RV -eq 0 ]; then 
        echo "	After this message the BES will be rebooted." >> $LOG_FILE
        echo 
        sleep $PT_TOTAL_POLL_TIME
        echo "Rebooting..."
        reboot	2>/dev/null 1>/dev/null		# rebooting the switch
        echo ;
        # Sleep in order not to execute further commands till it actually gets rebooted		
        sleep $PT_TOTAL_POLL_TIME			
    else
        echo "	After this message the FS8-18 at Slot No $AP_BLADE_NO will be POWER-CYCLED." >> $LOG_FILE
        sleep 5 ; 
        echo "[INFO]	Powering Off the FS8-18 at Slot No $AP_BLADE_NO..."; 
        echo "	Powering Off FS8-18 at Slot No $AP_BLADE_NO" >> $LOG_FILE
        slotpoweroff $AP_BLADE_NO 2>/dev/null 1>/dev/null
        sleep 5; 
        echo "[INFO]	Powering On Blade FS8-18 at Slot No $AP_BLADE_NO...";	
        echo "	Powering On FS8-18 at Slot No $AP_BLADE_NO" >> $LOG_FILE
        slotpoweron $AP_BLADE_NO 2>/dev/null 1>/dev/null
        # Time Out Logic
        local current_time=`date +%s`;
        local total_time_out=`expr $current_time + $PT_TOTAL_WAIT_TIME`;	# Max timeout is 15 Minutes
        local return_value=0		# 0 If timed Out - 15 Minutes are elapsed
        echo "Waiting for the FS8-18 blade in Slot No $AP_BLADE_NO to come up..."
        while [ $total_time_out -ge $current_time ]; do
            local echo_val=`slotshow`
            echo "	$echo_val" >> $LOG_FILE
            if [ "`slotshow | grep "$AP_BLADE_NO.*AP BLADE.*ENABLED"`" ]; then
                echo
                return 1;		# 1 is for Enabled
            elif [ "`slotshow | grep "$AP_BLADE_NO.*AP BLADE.*FAULTY"`" ]; then	
                echo
                return 2;		# 2 is for Faulty
            else
                echo -n ".";
                current_time=`expr $current_time + $PT_TOTAL_POLL_TIME`;
                sleep $PT_TOTAL_POLL_TIME;
            fi
        done
        if [ $return_value -eq 0 ]; then
            echo "[FATAL]	Timeout reached. Check the FS8-18 blade state or contact Brocade Support."
            exit 5;
        fi
    fi
}

if [ $# -eq 0 ]; then
    echo "No arguments supplied : `basename $0` $*"; 
    echo; 
    usage; 
    echo "" ; 
    echo ;
    exit 1
elif [ $# -gt 2 ]; then
    echo "Too many options are found : `basename $0` $*"; 
    echo; 
    usage; 
    echo "";
    exit 1
elif [ $# -le 2 ]; then
    blade_no_valid=`echo $2 | grep -E '[^0-9]'`;
    if [ ! -z $blade_no_valid ]; then
        echo "Blade slot number should be numeric : `basename $0` $*";
        echo;
        usage;
        echo "";
        exit 1;
    fi
    SYSFPGA_TASK=$1; shift;
    if [ $# -eq 0 ]; then
        AP_BLADE_NO=0
    else
        AP_BLADE_NO=$1
    fi

    if [ ! -d $LOG_DIR ]; then # Make room for logging.
        mkdir -p "/root/${LOG_DIR}"
    fi
    random_no=`date +%s`
    LOG_FILE="/root/${LOG_DIR}/sys_fpga_${SYSFPGA_TASK}_${random_no}";
    echo ; 
    echo "Log for this session is being captured at $LOG_FILE."
    echo "Log for this session is being captured at $LOG_FILE." >> $LOG_FILE

    mace_lance_platform_check				# Check for the Platform whether this is BES or FS8-18
    BES_BLADE_PL_CHECK_RV=$?
    find_bp_ip;								# Find the BP IP of the right AP Blade
    echo "Blade Processor IP = $BP_IP";

    # We don't have to warn the user when he is just verifying the upgrade.
    #if [ $SYSFPGA_TASK != 'verify_upgrade' ]; then
    #warning								# Warn the user and confirm user action.
    #fi
    if [ $SYSFPGA_TASK = 'upgrade' ] || [ $SYSFPGA_TASK = 'prepare' ]; then
        warning								# Warn the user and confirm user action.
    fi

    if [ $SYSFPGA_TASK = "verify_upgrade" ]; then
        echo >> $LOG_FILE
        echo "Inside verify_upgrade" >> $LOG_FILE
        echo "[INFO]	Verifying the upgrade."; 
        echo "[INFO]	Checking the state of the BES / FS8-18.";
        echo >> $LOG_FILE
        echo "Verifying the upgrade." >> $LOG_FILE
        echo "Checking the state of the BES / FS8-18." >> $LOG_FILE
        echo >> $LOG_FILE

        get_switch_current_state				# Get the current switch state.
        current_switch_state=$?
        echo "Current switch state: $current_switch_state" >> $LOG_FILE

        if [ $current_switch_state -eq 0 ]; then
            echo "[INFO]	BES / FS8-18 is in expected state."
            echo >> $LOG_FILE
            echo "BES / FS8-18 is in expected state." >> $LOG_FILE			
            echo >> $LOG_FILE			
        elif [ $current_switch_state -eq 1 ]; then
            echo "[FATAL]	BES / FS8-18 is in unexpected state after upgrade."; 
            echo ;
            if [ $BES_BLADE_PL_CHECK_RV -eq 0 ]; then
                echo "[INFO]	If you have run \"upgrade\" before and not POWER-CYCLED the BES, please POWER-CYCLE and re-run \"verify_upgrade\"."
            fi
            echo "[INFO]	If you have run \"prepare\" before \"verify_upgrade\" then this state is expected."
            echo "[FATAL]	If any one of above 2 is not correct, then this might be a problem. Please contact Brocade Support."; 
            echo ;
            echo "BES / FS8-18 is in unexpected state after upgrade." >> $LOG_FILE
            echo "If you have run \"prepare\" before \"verify_upgrade\" then this state is expected." >> $LOG_FILE
            echo "If you have run \"upgrade\" before and not POWER-CYCLED the BES, please POWER-CYCLE and re-run \"verify_upgrade\"." >> $LOG_FILE
            echo "If any one of above 2 is not right, then this might be a problem. Please contact Brocade Support." >> $LOG_FILE ; echo >> $LOG_FILE
            # No need to exit 6 here because it just verifies the upgrade.

        else
            echo "[INFO]	Wait 10 minutes and run \"verify_upgrade\" after BES / FS8-18 comes up."
            echo >> $LOG_FILE
            echo "Run \"verify_upgrade\" after BES / FS8-18 comes up." >> $LOG_FILE
            echo >> $LOG_FILE			
            exit 6;
        fi
        echo "[INFO]	Verifying the current sysFPGA version.";
        echo "Verifying the current sysFPGA version." >> $LOG_FILE
        echo >> $LOG_FILE
        find_current_fpga_version				# This finds current FPGA Version
        echo >> $LOG_FILE		
        check_current_fpga_version				# Get 0 if old version, 1 if new and 2 if anything else
        current_fpga_version=$?
        echo >> $LOG_FILE		
        if [ $current_fpga_version = 0 ]; then
            echo "[INFO]	Current sysFPGA version is old. This qualifies for \"upgrade\"."
            if [ $BES_BLADE_PL_CHECK_RV -eq 0 ]; then
                echo "[INFO]	If you have run \"upgrade\" already then POWER-CYCLE for new sysFPGA to load on BES."
            fi
            echo "Current FPGA version is old. This qualifies for upgrade. Run \"upgrade\"." >> $LOG_FILE
            echo "**NOTE**" >> $LOG_FILE
            echo "If you have run \"upgrade\" successfully, then POWER-CYCLE for the new sysFPGA image to load." >> $LOG_FILE
            exit 5;

        elif [ $current_fpga_version -eq 1 ]; then
            #echo "[SUCCESS]	BES / FS8-18 sysFPGA upgraded successfully." ; 
            echo "BES / FS8-18 sysFPGA version is upgraded" ; 
            echo >> $LOG_FILE
            echo "BES / FS8-18 sysFPGA version is already upgraded" >> $LOG_FILE
            echo >> $LOG_FILE
            exit 0;
        else
            echo "BES / FS8-18 sysFPGA version is unknown. Probably you should not hit this message."
            # No need to exit 5 here. Because it is just upgrade verification
        fi
        echo "Verification of upgrade completed" >> $LOG_FILE	

        echo; 
        echo "[INFO]	Verification of upgrade completed"; 
        echo;

    elif [ $SYSFPGA_TASK = "prepare" ]; then
        echo >> $LOG_FILE
        echo "Inside prepare" >> $LOG_FILE
        echo >> $LOG_FILE		
        echo "[INFO]	Preparing the BES / FS8-18 for sysFPGA upgrade."; 
        get_switch_current_state				# Get the current switch state.
        current_switch_state=$?

        if [ $current_switch_state -ne 0 ]; then
            echo "BES / FS8-18 is not in an expected state for upgrade. Cannot proceed for upgrade." >> $LOG_FILE
            echo "[FAIL]	BES / FS8-18 is not in an expected state for upgrade. ";
            exit 6;
        fi

        check_jtagpgm_is_available_in_bp			# Check if the build has jtagpgm utility for fpga upgrade
        check_fpga_upgrade_image_is_available_in_bp	# Check if the build has fpga upgrade image.

        find_current_fpga_version				# This finds current FPGA Version		
        check_current_fpga_version				# Get 0 if old version, 1 if new and 2 if anything else
        current_fpga_version=$?
        if [ $current_fpga_version = 0 ]; then
            echo "[INFO]	Current sysFPGA version is old. This qualifies for upgrade. Proceeding..."
            if [ $BES_BLADE_PL_CHECK_RV -eq 0 ]; then
                echo "[INFO]	Note: If you have done \"upgrade\" already, it will reflect only after POWER-CYCLE"; 
            fi
        elif [ $current_fpga_version -eq 1 ]; then
            echo "[INFO]	BES / FS8-18 sysFPGA version is already upgraded."
            exit 0;
        else
            echo "[FATAL]	BES / FS8-18 sysFPGA version is unknown. "
            echo "[FATAL]	Probably you should not hit this error message."
            exit 5;
        fi

        check_preempt_fos_file_available_in_bp		# CHeck if the preemptfos file is in BP
        create_preempt_fos_file_in_bp				# Create the preemptfos file in BP
        sleep 5;
        check_content_of_preempt_fos_file_in_bp		# Check the contents of the preemptfos file in BP
        sleep 5;

        check_bp_ip_addr_file_in_bp				    # Check if the bp_ip_addr file in is available in BP
        create_bp_ip_addr_file_in_bp				# Create the bp_ip_addr file in BP
        sleep 5;
        check_content_of_bp_ip_addr_file_in_bp		# Check the contents of the bp_ip_addr file in BP
        sleep 5;

        echo "[INFO]	Environment is set for upgrade."; 
        if [ $BES_BLADE_PL_CHECK_RV -eq 0 ]; then
            echo "[INFO]	After the reboot of BES, it takes 10 minutes"
            echo "	for the BES to come to expected state (Faulty). You should"
            echo "	re-run this utility with \"upgrade\" option after the BES is Faulted.";
        else
            echo "[INFO]	After the power-cycle of FS8-18, it takes 10 minutes"
            echo "	for the FS8-18 to come to expected state (Faulty). You should"
            echo "	re-run this utility with upgrade option after the FS8-18 is Faulted.";
        fi
        echo "Environment is set for upgrade." >> $LOG_FILE
        echo "After the forthcoming reboot, run the upgrade directly." >> $LOG_FILE

        reboot_blade						# Reboot AP Blade in DCX or BES box

        get_switch_current_state				# Get the current switch state.
        current_switch_state=$?
        if [ $current_switch_state -eq 1 ]; then
            echo "[INFO]	BES / FS8-18 is in expected state for upgrade. Proceeding..." ;
        else
            echo "[FATAL]	BES / FS8-18 is not in expected state for upgrade."
            echo "[INFO]	Wait for 10 minutes after reboot and then run \"upgrade\".";
            exit 6;
        fi
        echo "[INFO]	Preparation completed. Please re-run the script with \"upgrade\" option to upgrade."; echo;
    elif [ $SYSFPGA_TASK = "upgrade" ]; then
        echo >> $LOG_FILE
        echo "Inside upgrade" >> $LOG_FILE
        echo >> $LOG_FILE
        echo "[INFO]	Starting the sysFPGA upgrade."; 

        get_switch_current_state				# Get the current switch state.
        current_switch_state=$?
        if [ $current_switch_state -eq 1 ]; then
            echo "[INFO]	BES / FS8-18 is in expected state for upgrade. Proceeding..." ; 
        else
            echo "[FATAL]	BES / FS8-18 is not in expected state for upgrade."
            echo "[INFO]	Wait for 10 minutes after reboot and then re-run upgrade.";
            exit 6;
        fi

        check_jtagpgm_is_available_in_bp			# Check if the build has jtagpgm utility for fpga upgrade

        check_fpga_upgrade_image_is_available_in_bp	# Check if the build has fpga upgrade image.

        find_current_fpga_version					# This finds current FPGA Version
        check_current_fpga_version					# Get 0 if old version, 1 if new and 2 if anything else
        current_fpga_version=$?

        if [ $current_fpga_version = 0 ]; then
            echo "[INFO]	Current sysFPGA version is old. This qualifies for upgrade."
            if [ $BES_BLADE_PL_CHECK_RV -eq 0 ]; then
                echo "	About to upgrade sysFPGA. After upgrade is"
                echo "	successful, please POWER-CYCLE the BES."
            fi
        elif [ $current_fpga_version -eq 1 ]; then
            echo "[INFO]	BES / FS8-18 sysFPGA version is already upgraded to the latest version."
            echo "[SUCCESS]	Upgrade already completed successfully."
            exit 0;
        else
            echo "[FATAL]	BES / FS8-18 sysFPGA version is unknown. If you hit this message, please contact Brocade Support."
            exit 5;
        fi

        upgrade_fpga							# Upgrade FPGA
        validate_fpga_upgrade_message			# Validate the fpga upgrade messages.
        rv_of_fpga_upgrade=$?
        if [ $rv_of_fpga_upgrade -ne 0 ]; then
            echo "[FATAL]	Upgrade failed - Return value: $rv_of_fpga_upgrade."
            echo "	Please go through the logs manually."
            echo "	If this is due to network outage, try it again."
            echo "Upgrade failed - Return value: $rv_of_fpga_upgrade. Please go through the logs manually." >> $LOG_FILE
            echo "If this is due to network outage, try it again." >> $LOG_FILE
            exit 5;
        fi

        if [ $BES_BLADE_PL_CHECK_RV -eq 0 ]; then 
            echo "[INFO]	POWER-CYCLE the BES in order to load the new sysFPGA image."
            echo "	Wait for 10 minutes to let the BES come up and run \"verify_upgrade\"";
            echo "POWER-CYCLE the BES / FS8-18 in order to load the new sysFPGA image. " >> $LOG_FILE
            echo "Let the BES / FS8-18 come up and run \"verify_upgrade\"." >> $LOG_FILE
            exit 10;
        else
            echo "[INFO]	After this message the FS8-18 blade in Slot No $AP_BLADE_NO will be POWER-CYCLED."
            reboot_blade							# Reboot AP Blade in DCX or the whole pizza box
        fi

        get_switch_current_state				# Get the current switch state.
        current_switch_state=$?
        if [ $current_switch_state -ne 1 ]; then
            echo "[INFO]	BES / FS8-18 is in expected state. Proceeding..."
        else
            echo "[FATAL]	BES / FS8-18 is not in expected state. Check the logs."
            exit 6;
        fi

        find_current_fpga_version				# This finds current FPGA Version
        check_current_fpga_version				# Get 0 if old version, 1 if new and 2 if anything else
        current_fpga_version=$?

        if [ $current_fpga_version = 0 ]; then
            echo "[FATAL]	Current sysFPGA version is Old. Some problem in upgrade. Check the logs."
            exit 5;
        elif [ $current_fpga_version -eq 1 ]; then
            echo "[SUCCESS]	Upgrade successfully done."
            exit 0;
        else
            echo "[FAIL]	BES / FS8-18 sysFPGA version is unknown. If you hit this message, please contact Brocade Support."
            exit 5;
        fi
        echo "[INFO]	sysFPGA upgrade is completed. Please verify the upgrade by running \"verify_upgrade\"."; 
        echo ;
    elif [ $SYSFPGA_TASK = "clear_prepare" ]; then
        echo >> $LOG_FILE
        echo "Inside clear_prepare" >> $LOG_FILE
        echo >> $LOG_FILE
        echo "[INFO]	Reverting back the environmental changes for sysFPGA upgrade."

        delete_preempt_fos_file_in_bp				# Delete the preemptfos file in BP
        delete_bp_ip_addr_file_in_bp				# Delete the bp_ip_addr file in BP

        sleep $PT_TOTAL_POLL_TIME;					# Sleep for 10 seconds to have the files registered

        find_current_fpga_version				# This finds current FPGA Version
        check_current_fpga_version				# Get 0 if old version, 1 if new and 2 if anything else
        current_fpga_version=$?

        if [ $current_fpga_version = 0 ]; then
            echo "[INFO]	Current sysFPGA version is old."
        elif [ $current_fpga_version -eq 1 ]; then
            if [ $BES_BLADE_PL_CHECK_RV -eq 0 ]; then
                echo "[SUCCESS]	BES is already upgraded. Can't \"clear_prepare\". "
            else
                echo "[SUCCESS]	FS8-18 in Slot No $AP_BLADE_NO is already upgraded. Can't \"clear_prepare\". "
            fi
            exit 0;
        else
            echo "[FAIL]	BES / FS8-18 sysFPGA version is unknown. If you hit this message, please contact Brocade Support"
            exit 5;
        fi

        if [ $BES_BLADE_PL_CHECK_RV = 0 ]; then
            echo "[INFO]	After this message, BES will get rebooted."
        else
            echo "[INFO]	After this message, blade FS8-18 will get POWER-CYCLED."
        fi
        # Rebooting will bring the BES to correct state.
        reboot_blade							# Reboot AP Blade in DCX or the whole pizza box

        get_switch_current_state				# Get the current switch state.
        current_switch_state=$?
        if [ $current_switch_state -eq 0 ]; then
            echo "[INFO]	BES / FS8-18 is back Online."
        elif [ $current_switch_state -eq 1 ]; then
            echo "[FAIL]	BES / FS8-18 is in unexpected state after \"clear_prepare\"."; 
            exit 6;
        fi
        echo "[INFO]	\"prepare\" step has been undone successfully. BES / FS8-18 is back to normal." ; 
    elif [ $SYSFPGA_TASK = 'downgrade' ]; then
        #warning								# Warn the user and confirm user action.
        downgrade_fpga
        exit 0;
    elif [ $SYSFPGA_TASK = 'downgrade_prepare' ]; then
        warning		       			 # Warn the user and confirm user action.
        create_preempt_fos_file_in_bp            # Create the preemptfos file in BP
        sleep 5;
        create_bp_ip_addr_file_in_bp 
        sleep 5
        echo "Rebooting."
        reboot_blade 
        exit 0;
    else
        echo :
        echo "Not a supported ARG. Please consult Usage Instructions."
        echo >> $LOG_FILE
        echo "Not a supported ARG. Please consult Usage Instructions." >> $LOG_FILE
        usage
        echo ""
        exit 1
    fi
fi

