#!/bin/sh
#
# This script, usually run once during system startup from /etc/init.d/fabos
# Checks for an existane of a panicdump in devices such as MTD.
# If found, and if newer than the one found under /core_files/panic/
# then it copies the panicdump from MTD to destination directory.
#

BIN=/fabos/cliexec/
PD_SUCCESS=0

# core files and now pd location. This is a directory name
CORE_FILES_DIR=/core_files	# The place where we store panicdump
				# which is same for core files
				# see savecore also, where this is used

PD_DIR=$CORE_FILES_DIR/panic

# pd Control file. Will contain all the control date related to pd.
# It contains the time-stamp of pd.
# Note: this time-stamp is the time retrived from MTD and NOT the
# current time, when we saved this
PD_CTRL_FILE=/etc/.pd_data	# We will NOT store this in $CORE_FILES
				# This is because the existance is NOT
				# gauranteed and an user may choose to
				# delete $CORE_FILES_DIR. But we need
				# this time stamp
PD_TEMP_ERRLOG_FILE=/tmp/pd_temp_errlog

MISC_FILE=/core_files/panic/system_sym  # if you change this, also change
                                # corresponding entry in pdul_misc_pre.sh and
                                # pdul_misc_post.sh and pdcheck.sh

MAX_PD_FILES=9	# At this time we will store up to 10 panic dumps

#
# Log a message both to stdout and to the PD_TEMP_ERRLOG_FILE
# All the facilities of printf may be used
#

log_msg(){
	printf "$@"
	printf "$@" >> $PD_TEMP_ERRLOG_FILE
}

#
# function used to count the number of files in panic 
# # Check if there are too many dump files are around
# if more than 9 are found, get rid of the oldest ones

checkIfDumpExceeded() {
	checkIfDumpExceededHelper ${PD_DIR}/core.*
	checkIfDumpExceededHelper ${PD_DIR}/misc.*
}

#
#   Helper function for checkIfDumpExceeded
#   Actually does the work of removing excess files
#

checkIfDumpExceededHelper() {
	while [ $# -gt $MAX_PD_FILES ] ; do
		/bin/rm "$1"
		shift
	done
}

#
#	Function used to check for a presence of a dumpfile
#	if new pd found, record the date in $mtd_ts and return 1
#	returns 0 otherwise
#
checkForNewDump() {
	# just checking if pdcp works
	$BIN/pdcp $DUMPFILE 1>/dev/null 2>/dev/null
	rv=$?
	if [ $rv -ne $PD_SUCCESS ]
	then
		# well, our MAGIC# is not found
		# we will excuse if the dump file is of "\000" NULLs for 10K.
		# we resorted to 10k limit because of time limit to write NULLs

		nonNulls=`/bin/dd if=$DUMPFILE bs=1024 count=10 2>/dev/null | /usr/bin/tr -d "\000" | wc -c`
		if [ "$nonNulls" -eq "0" ]
		then
			# this is a "clean" flash, so let us just return
			return 0
		fi

		# Some error condition, like,
		#	either no valid pd exists in MTD
		#	or     there was some error condition in MTD
		#   This is usually the case when we run the new pd code
		#   on a cp where pd has not been implemented, such as
		#   upgrade from 4.0.0 to 4.1
		# Anyways, we will assume NO new pd and clean up pd partition

		/bin/dd if=/dev/zero of=$DUMPFILE bs=1024 count=$DUMPSIZE 2>/dev/null
		log_msg "Info: panic dump has been initialized!\n"
		return 0
	fi
	# now that there is some valid pd in MTD
	# so, let us check if we have seen this already...
	# by comparing it with time stamp in CF
	mtd_ts_full=`$BIN/pdcp $DUMPFILE 2>/dev/null`
	mtd_ts=`echo $mtd_ts_full | /usr/bin/cut -f1 -d' '`
	[ -z "$mtd_ts" ] && mtd_ts=mtd_ts

	# We will remove comments along with resulting blank lines
	cf_ts_full=`/bin/sed -e \
		's/#.*//;/^[ \t]*$/d' \
		$PD_CTRL_FILE 2>/dev/null`
	cf_ts=`echo $cf_ts_full | /usr/bin/cut -f1 -d' '`
	[ -z "$cf_ts" ] && cf_ts=cf_ts

	if [ "$mtd_ts" -le "$cf_ts" ]
	then
		# We have already seen this...
		return 0
	fi
	# We found a different pd in MTD!
	log_msg "pdcheck: info: found new pd: mtd_ts=%s, cf_ts=%s\n" \
		"$mtd_ts_full" "$cf_ts_full"
	return 1
}

# function used to adjust the size of the dump file just in case
# we decide to change it from one build to another
# and map the file to the ATA dumper

adjustSizeAndMap() {
	# on platforms with a dump partition, nothing to do
	if [ -h $DUMPLINK ]
	then
		return 0
	fi

	# get the size in bytes of the dump file
	set -- $( ls -al $DUMPFILE )
	CURRENTSIZE=${5:-0}
	# adjust it to be represented in KB
	let CURRENTSIZE=CURRENTSIZE/1024

	# if the dump file is not the right size
	if [ $CURRENTSIZE -ne $DUMPSIZE ]
	then
		# recreate it with the right size
		printf "setting $DUMPFILE to $DUMPSIZE KB\n"
		/bin/dd if=/dev/zero of=$DUMPFILE bs=1024 count=$DUMPSIZE 2>/dev/null
		/bin/sync
	fi

	# map the dump file to the ATA dumper
	$BIN/pdmap $DUMPFILE
	return 0
}

#
# main starts here
# let us see if panicdump is chkconfig'd off
chkconfig panicdump
if [ $? -eq 1 ]
then
        # 
        # we will turn off the panic dump
        # Please see panicdump.h and panicdump.c for more details on
        # these strings used (like ioctl)
        #
        printf "PD_IOCTL1" > /proc/panicdump
fi

#
# Let us first calculate the date -- This will be seconds like time(NULL)
#

DATE=`/bin/date +"%s"`

#
# Check that there is some space, if not remove some old core files
#

if [ -x /sbin/core_tidy.sh ] ; then
	(
		set -e
		cd "$CORE_FILES_DIR"
		/sbin/core_tidy.sh
	)
fi

# Let us create the panic directory, if this does not exists already
if [ ! -d $PD_DIR ]
then
	# delet anything that is not a directoy...
	rm -f $PD_DIR 1>/dev/null 2>/dev/null
	# and create a diretory
	mkdir -p $PD_DIR
fi

# Let us create the date file, if this does not exists
if [ ! -f $PD_CTRL_FILE ]
then
	# The file does not exists so, create it!
	printf "0\n" > $PD_CTRL_FILE
	log_msg "pdcheck: info: Created New pd control file\n"
fi

# this link should exist on platforms with a separate panic dump partition
# and not on those which do not like stealth
DUMPLINK=/dev/dump0

# if a link named /dev/dump0 exists
if [ -h $DUMPLINK ]
then
	# look for a new dump in the panic dump partiion
	DUMPFILE=/dev/panicdump
	# we use DUMPSIZE in checkForNewDump on some error conditions
	DUMPSIZE=10
else
	# look for a new dump in /var/panicdump
	DUMPFILE=/var/panicdump
	# on platforms with no panic dump partition, we use a regular 512KB file
	DUMPSIZE=512
	PLATFORM=$[`sin 2>/dev/null | awk '/SWBD/ { print $2 }' | sed 's/[^0-9]//g'`]
	if [[ ($PLATFORM -eq 148) || ($PLATFORM -eq 156) || ($PLATFORM -eq 157) || ($PLATFORM -eq 158) ]]; then
		DUMPSIZE=1024
	fi
	if [[ ($PLATFORM -eq 161) || ($PLATFORM -eq 162) || ($PLATFORM -eq 165) || ($PLATFORM -eq 166) ]]; then
		DUMPSIZE=4096
	fi

	# make sure that DUMPFILE exists
	if [ ! -f $DUMPFILE ]
	then
		# if DUMPFILE does not exist, create it
		/bin/dd if=/dev/zero of=$DUMPFILE bs=1024 count=$DUMPSIZE 2>/dev/null
	fi
fi

checkForNewDump # Checking for new dump
if [ $? -eq 0 ]
then
	# no new dump, so let us just log this and exit
	adjustSizeAndMap
	exit 0
fi
checkIfDumpExceeded # Check if there are too many dump files are around
#
# new dump found, so let us save it and also save the time after
# successful completion of the whole dump
#
PD_DATE=`date -d "1970-01-01 $mtd_ts sec" +"%Y%m%d%H%M%S"`
$BIN/pdcp $DUMPFILE ${PD_DIR}/core.pd$PD_DATE
rv=$?
if [ $rv -ne $PD_SUCCESS ]
then
	log_msg "pdcheck: pdcp failed to copy the files (err = %d)\n" $rv
#
# We will NOT exit here but will save the time stamp so we will not
# attempt to re-red this again -- if we do, we will end up the same
# problem likd this during next boot.
#

fi
if [ -f $MISC_FILE ]
then
	/bin/cp $MISC_FILE  ${PD_DIR}/misc.pd$PD_DATE
else
	printf "'$MISC_FILE' was missing - ksyms was not saved?\n" >  ${PD_DIR}/misc.pd$DATE
fi

#
# we will nullyfy the original file now
# The actual ksyms, etc will be copied in pdul_misc_post.sh
#
# let us create this file, voiding any previous contents
[ -d "${MISC_FILE%/*}" ] && printf "\nCreated At:`/bin/date`\n" >  $MISC_FILE
# after creating such a file, we have to do sync to save the fs
/bin/sync
/bin/sync
/bin/sync # three times!? why???
# This is a bit of Unix folklore. The Unix 'sync(2)' system call just
# scheduled all the disks to be written. It was well known that in the
# time it took for you to type sync and press return 3 times, all the
# pending disk I/O would finish. Putting sync 3 times in a file is
# a waste of time, it runs too fast to be useful. The Linux 'sync(2)'
# call is different from unix, in that it waits for all the disk I/O
# to complete before it returns.

#
# Everything copied successfully! So, let us save this time stamp
# so next time we will not attempt to copy again
# NOTE:
#	This time-stamp copying should be done at last
#
printf "
# This time stamp created on `/bin/date`
# with startup time in seconds: $DATE
#
$mtd_ts
#
" > $PD_CTRL_FILE

# Let is stamp this information also on the /mnt, so when we switch
# partition, we will NOT save the same dump on /mnt
#
if [ -d /mnt ]
then
printf "
# This time stamp created on `/bin/date`
# with startup time in seconds: $DATE
#
$mtd_ts
#
" > /mnt/$PD_CTRL_FILE
fi
adjustSizeAndMap
sync
sync
sync
