/*
 *
 *  File Name:      ras_trace.h
 *
 *
 *
 *  Description:
 *
 *  This header file defines the user interface to the Trace Subsystem.
 *  Callers use the interface to create trace buffers and store
 *  program execution information into them.
 *
 *  It contains:
 *
 *  1.  Function prototypes 
 *  2.  Subsystem Identifiers assigned for use to other subsystems
 *  3.  Trace entry data type
 *  4.  Macro definitions that allow users to create Event IDs.
 *
 *
 *  This file is divided into two interfaces.  User-level software uses one
 *  and kernel-level software the other.  The __KERNEL__ flag is used to 
 *  distinguish between code compiled for user space and kernel-level code.
 *
 *  Note:  The management Interface to the Trace Subsystem is contained in 
 *         separate header file, "ras_trace_freeze.h"
 *
 *
 *  Warnings:
 *
 *  None.
 *
 *
 *
 *
 *  Copyright (C) 2003-2011 Brocade Communications
 *  All rights reserved.
 *
 *  These coded instructions and statements contain unpublished trade
 *  secrets and proprietary information.  They are protected by federal
 *  copyright law and by trade secret law, and may not be disclosed to
 *  third parties or used, copied, or duplicated in any form, in whole
 *  or in part, without the prior written consent of Brocade Communications.
 *
 *
 *
 *
 *
 *  Change History:
 *
 *  Date        Author      Description
 *
 *  8/4/03      S. Stolper  Initial Version
 *  8/12/03	S. Stolper	Add include for module IDs.
 *
 */

/*  Avoid Duplication
 */
#ifndef __RAS_TRACE_H__
#define __RAS_TRACE_H__
// #include <linux/synclink.h>
#ifdef	__KERNEL__
#include <linux/types.h>
#include <linux/module.h>
#else
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <strings.h>
#endif
#include <trace/trace_buff_cfg.h>
/*  Define Function Return Values */
#define TRACE_SUCCESS			(0)
/* Max positive int to make sure this is not a real handle */
#define	TRACE_MOD_DISABLED		(2147483647)
#define TRACE_ERROR			(-1)
#define TRACE_ERR_BAD_STATE		(-2)
#define TRACE_ERR_NOCREATE		(-3)
#define TRACE_ERR_NORESOURCES		(-4)
#define TRACE_ERR_DRIVER		(-5)
#define	TRACE_ERR_LISTFULL		(-6)
#define	TRACE_ERR_LISTEMPTY		(-7)
#define	TRACE_ERR_BADINPUT		(-8)
#define	TRACE_ERR_DUPLICATE		(-9)
#define	TRACE_ERR_BADSLOT		(-10)
#define	TRACE_ERR_BADMSGID		(-11)
#define	TRACE_ERR_NOTREADY		(-12)
#define	TRACE_ERR_CFG			(-13)
#define TRACE_ERR_SIOM			(-14)

/* Trace severity levels */
#define TRACE_MIN_LVL	(TRACE_1)
#define TRACE_1			(1)
#define TRACE_2			(2)
#define TRACE_3			(3)
#define TRACE_4			(4)
#define TRACE_5			(5)
#define TRACE_6			(6)
#define TRACE_7			(7)
#define TRACE_8			(8)
#define TRACE_9			(9)
#define TRACE_10		(10)
#define TRACE_11		(11)
#define TRACE_12		(12)
#define TRACE_MAX_LVL		(TRACE_12)
#define TRACE_DEFAULT_LVL	(TRACE_5)


#define TRACE_LVL_DBG		TRACE_4
#define TRACE_LVL_INFO		TRACE_5
#define TRACE_LVL_WARNING	TRACE_6
#define TRACE_LVL_ERROR		TRACE_7
#define TRACE_LVL_CRITICAL	TRACE_8
#define TRACE_LVL_CONSOLE	TRACE_10
#define TRACE_LVL_RASLOG	TRACE_11

/* Trace area filters */
#define TRACE_A0		0
#define TRACE_A1		1
#define TRACE_A2		2
#define TRACE_A3		3
#define TRACE_A4		4
#define TRACE_A5		5
#define TRACE_A6		6
#define TRACE_A7		7
#define TRACE_A8		8
#define TRACE_A9		9
#define TRACE_A10		10
#define TRACE_A11		11
#define TRACE_A12		12
#define TRACE_A13		13
#define TRACE_A14		14
#define TRACE_A15		15
#define TRACE_A_DFT		0
#define TRACE_A_MAX		16

#define TRACE_LVL2TRCE(lvl)	(lvl & 0xff)
#define TRACE_LVL2CONSOLE(lvl)	((lvl >> 8) & 0xff)
#define TRACE_2LVL(trce, console) ((console << 8) | trce)

#define TRACE_NO_RF		0  // rf disabled	
#define TRACE_RF1		1  // doing rf if 10% of total entry left		
#define TRACE_RF2		2  // doing rf if 20% of total entry left
#define TRACE_RF3		3  // doing rf if 30% of total entry left


/*
 * Trace allows user to pass in 3 generic bytes and a level per entry.
 * The user can decide how these values are used, and when displaying data
 * or calling trace_snapshot, can pass in a filter on these values.
 */

#define TRACE_LVL(lvl)  (lvl & 0xff)
#define TRACE_G1(vu)	((vu & 0xff) << 8)
#define TRACE_G2(vu)	((vu & 0xff) << 16)
#define TRACE_G3(vu)	((vu & 0xff) << 24)
#define TRACE_G_NONE	(0)

#define TRACE_LVL_ON (0x000000FF)
#define TRACE_G1_ON	(0x0000FF00)
#define TRACE_G2_ON	(0x00FF0000)
#define TRACE_G3_ON	(0xFF000000)
#define TRACE_G_ALL_ON	(0xFFFFFFFF)
#define TRACE_SW(vu) (TRACE_G1(vu))
#define TRACE_LVL2G1(lvl)	((lvl >> 8) & 0xff)
#define TRACE_LVL2G2(lvl)	((lvl >> 16) & 0xff)
#define TRACE_LVL2G3(lvl)	((lvl >> 24) & 0xff)

#define TRACE_8_ENTRY_SZ	(8)
#define TRACE_16_ENTRY_SZ	(16)
#define TRACE_32_ENTRY_SZ	(32)
#define TRACE_64_ENTRY_SZ	(64)
#define TRACE_128_ENTRY_SZ	(128)
#define TRACE_256_ENTRY_SZ	(256)

#define TRACE_ATTR_NONE (0)
#define TRACE_ATTR_SHARED (0x00000001)
#define TRACE_ATTR_RAW (0x00000002)
#define TRACE_ATTR_ENTRY_SZ (0x00000004)

typedef struct {
	unsigned int  td_word[2];
}  trace_8_data_t;

typedef struct {
	unsigned int  td_word[4];
}  trace_16_data_t;


typedef struct {
	unsigned int  td_word[8];
}  trace_32_data_t;

typedef struct {
	unsigned int  td_word[16];
}  trace_64_data_t;

typedef struct {
	unsigned int  td_word[32];
}  trace_128_data_t;

typedef struct {
	unsigned int  td_word[64];
}  trace_256_data_t;


/* Generic bytes filter */
#define MAX_GENERIC_ID 10
typedef struct trace_filter {
	int	bmap; /* bitmap indicating which bytes of the data field to use */
	int	data;
    unsigned int generic_id[MAX_GENERIC_ID];
    short unsigned int generic_id_index;
} trace_filter_t;

#ifndef NOS_BUILD
/* Rtlog data structure*/
typedef struct rt_data{
        unsigned long td_word[7];
}rt_data_t;
#endif

/* Limits */
#define TRACE_MAX_NUM_ENTRIES	(1024 * 64) /* 64K entries */

/*  Module IDs
 *
 *  Module IDs identify software when it creates a 
 *  trace buffer or stores trace information.  The values
 *  are centrally managed by the RAS Subsystem and 
 *  assigned to user software subsystems.  The values can 
 *  also be used by operators from the CLI to manipulate
 *  trace buffers maintained by the system.
 */
#include <ras/ras_modules.h>


/*  Trace Entry Data Type
 *
 *  This union contains the bytes to be preserved in the
 *  trace buffer.  Developers fill the union and pass its
 *  address as an argument to the trace_store() function.
 *  The trace entry is accessible as both bytes and words.
 *  The format of the data is associated with a unique 
 *  trace event ID using the TRACE_EVENT_ID macro.
 */
#define TRACE_BYTES_PER_ENTRY		(64)  // was (32)
#define TRACE_UINT16S_PER_ENTRY		(TRACE_BYTES_PER_ENTRY / sizeof(uint16_t))
#define TRACE_WORDS_PER_ENTRY		(TRACE_BYTES_PER_ENTRY / sizeof(int))

typedef union {
	unsigned char td_byte[TRACE_BYTES_PER_ENTRY];
	uint16_t  td_uint16[TRACE_UINT16S_PER_ENTRY];
	unsigned int  td_word[TRACE_WORDS_PER_ENTRY];
}  trace_data_t;

/*
 * buffer entry sizes
 */
#define TRACE_DEF_ENTRY_SZ	(32)
#define TRACE_MIN_ENTRY_SZ	(8)
#define TRACE_MAX_ENTRY_SZ	(256)


typedef union {
                unsigned char td_byte[TRACE_MAX_ENTRY_SZ];
		uint16_t  td_uint16[TRACE_MAX_ENTRY_SZ / sizeof(uint16_t)];
                unsigned int  td_word[TRACE_MAX_ENTRY_SZ / sizeof(int)];
}  trace_long_data_t;



/*  Macros to define EVENT IDs
 *
 *  Users invoke this macro to create an event ID to be 
 *  used in calls to trace_store().
 *
 *  When compiled as part of an analytical tool, the macro
 *  creates an entry that can be used as part of an array
 *  to decode the message produced by the switch.
 *
 *  TRACE_EVENT_ID(event_ID,  constant_int,  fmt_string)
 *
 *  event_ID - Mnemonic used to identify trace entry format
 *  constant_int - constant value of mnemonic
 *  fmt_string - Optional printf-like formatting string.
 *  
 *  If supplied, the format string is compiled into a
 *  utility that post-processes trace dumps.  If it is 
 *  NULL, the entry is displayed as raw hexadecimal.
 *  The special characters in the format string must be in
 *  the same order as they appear in the trace entry.  
 *  Allowable values are:
 *  
 *  %d = display 4 bytes as integer value
 *  %x = display 4 bytes as hexadecimal value
 *  %c = display 1 byte as hexadecimal value 
 *  Note that this differs from how the printf() function 
 *  works.
 *
 *  EXAMPLE:
 *  
 *  TRACE_EVENT_ID(FW_TEMP_TRACE, 1, "tempID= %d, val= %x")
 *
 *  Callers place the IDs into a header file 
 *  "xxx_trace_id.h", where 'xxx' is the name of the 
 *  Subsystem Identifier assigned by the Trace Subsystem.  
 *  Developers include these files in their 'C' source 
 *  files and use the event IDs when storing trace 
 *  information.
 */
#ifdef __ANALYSIS_TOOL__

/*
 * the first module id is used as the entry identify
 */
#define TRACE_EVENT_ID(trace_event_ID, constant_int, optional_fmt_string) \
	{MODULE_ID, #trace_event_ID, constant_int, optional_fmt_string},

#else
#define TRACE_EVENT_ID(trace_event_ID, constant_int, optional_fmt_string) \
	enum { trace_event_ID = constant_int }; \
//        static char *##trace_event_ID##_FMT __attribute__ ((unused)) = {optional_fmt_string};
#endif


/*
 * we use the following format to get finer
 * granularity of buffer indexing inside one module id
 * 
 *	Without taking the page size in the offset
 *	Bits		Operation
 *	0 - 11	--> Main buffer Module ID
 *	12 - 15	--> Control ID (Will be effective only in remmaping)
 *	16 - 31	--> Sub buffer ID
 * 
 */
#define TRACE_BUFF_ID(modid, subid)	(((subid) << 16) + (modid))
#define TRACE_BUFF_MODID(id)		((id) & 0xFFFF)
#define TRACE_BUFF_SUBID(id)		(((id) >> 16) & 0xFFFF)
#define TRACE_BUFF_SUBID_MASK(id)	(id & 0x000F0000)
#define TRACE_BUFF_SUBID_MAX		7  // bit 0, 1, 2
#define TRACE_BUFF_MAPID_MAX		0x00040000  // bit 0, 1, 2
#define TRACE_BUFF_REMAP_IDX		1
#define TRACE_BUFF_LOW2BYTES        	TRACE_BUFF_MODID
#define TRACE_BUFF_HIGH2BYTES    	TRACE_BUFF_SUBID


#define INVALID_TRACE_HNDL		-1

/*
 *  * OTF logging related macros and functions
 *   */
#define IS_OTF_DICTIONARY(mid)  ((mid == RAS_OTF_DICTIONARY_NS_ID) || \
                                 (mid == RAS_OTF_DICTIONARY_FCR_ID))
#define IS_OTF_USER(mid)        ((mid == RAS_NS_TRACE_ID) || \
                                 (mid == RAS_NS_TRACE_ID_NSZONE) || \
                                 (mid == RAS_NS_TRACE_ID_IPC) || \
                                 (mid == RAS_NS_TRACE_ID_AD) || \
                                 (mid == RAS_FCR_ID) || \
                                 (mid == RAS_FCRM_ID))
#define IS_OTF(mid)             (IS_OTF_DICTIONARY(mid) || \
                                 IS_OTF_USER(mid))
extern  inline  int trace_get_otf_dict_mid(int mid);

extern inline int
trace_get_otf_dict_mid(int mid)
{
    int otf_dict_mid = RAS_UNKN_ID;

    if (mid == RAS_NS_TRACE_ID ||
        mid == RAS_NS_TRACE_ID_NSZONE ||
        mid == RAS_NS_TRACE_ID_IPC ||
        mid == RAS_NS_TRACE_ID_AD) {
        otf_dict_mid = RAS_OTF_DICTIONARY_NS_ID;
	} else if (mid == RAS_FCR_ID ||
		mid == RAS_FCRM_ID) {
		otf_dict_mid = RAS_OTF_DICTIONARY_FCR_ID;
	} 

    return (otf_dict_mid);
}

/*
 * trace buffer handle
 */
typedef int trace_hndl_t; 

/*  Select interface by determining if the code being 
 *  compiled runs in kernel or user space.
 */
#ifdef __KERNEL__

int trace_core_notify(char *address, u_int size, int id);

/*
 * platform-specific dump funtions
 */
typedef struct trace_dump_func {
#if defined CONFIG_BLK_DEV_ATA_DUMPER
	int (*open)(unsigned int);
	int (*close)(int);
	int (*write)(int, char *buf, int size);
#elif defined CONFIG_WM_DUMPER
	int (*open)(int);
	int (*close)(int , int);
	int (*write)(int , int, char *buf, int size);
#else
    int (*open)(unsigned int);
    int (*close)(int);
    int (*write)(int, char *buf, int size);
#endif
} trace_dump_func_t;

/*  ***** Kernel-level interface *****
 */


/*
 *
 *
 *  Function Name:  trace_init
 *
 *
 *  Inputs:	index        	-  Each module must identify itself when
 *					creating a trace buffer.  
 *		number_entries	-  Indicates how many entries stored in 
 *					buffer before wrapping (power of 2).
 *		module_string_p	-  Eight character string (not including 
 *					NULL) or NULL.
 *		prop		- trace buffer property
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	Returns zero if successful, non-zero if fails (See below)
 *
 *
 *  Description:
 *
 *  This function creates a trace buffer in kernel memory and alerts the 
 *  Trace Subsystem that the new buffer exists.  It also records the module ID
 *  in the buffer so that the trace information is associated with the new 
 *  module.
 *
 *  Each kernel level module creates a trace buffer by calling this function
 *  from its module initialization function.  Modules identify themselves by
 *  supplying a index preprocessor definition from "ras_trace.h".  The 
 *  module uses the same ID when tracing information into the buffer.  
 *  If supplied, the module character string is preserved in the trace buffer 
 *  for identification purposes.  Callers may use the string to identify a 
 *  kernel module.
 *  
 *  If this function is called more than once with the same module ID, it
 *  reports an error.  If it is unable to create the trace buffer, it LOGs
 *  an error and return TRACE_ERR_NOCREATE.  Subsequent calls to trace 
 *  information do not succeed, but the calling software continues to run
 *  normally.  If this function fails, it may be retried at a later time.
 *
 *
 *  NOTE: 
 *
 *  "number_entries" must be a power of two (2).  If not, the function 
 *  returns TRACE_ERR_BADPARAM and LOG a message.
 *
 */
int trace_init(int index, unsigned int number_entries, char *module_string_p, void *prop); 
int trace_init_new(int index, char *module_string_p);


/*
 *
 *
 *  Function Name:  trace_thread_identify
 *
 *
 *  Inputs:	thread_string_p  -  Eight character string (not including
 *					NULL)
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function should only be called from kernel threads.  It accepts a 
 *  string that is used to identify all trace entries that are generated 
 *  by the calling thread context.
 *
 *  Developers may choose to identify trace entries created by a thread with a 
 *  string to assist post-mortem debugging.  Each kernel module trace buffer 
 *  supports up to 16 thread identification entries.  If there are already 
 *  sixteen threads identified in the buffer, calling trace_thread_identify() 
 *  generates a LOG message and modifies the buffer metrics.
 *
 *  When a kernel thread exits normally, it should remove its identification 
 *  information from the trace buffer by calling 
 *  trace_thread_remove_identity().
 *
 *  If this function is called before calling trace_init(), it logs a message 
 *  to indicate a bug in the code.  It does not ASSERT() like the user-level 
 *  version of this function.  If it is called after a failed call to 
 *  trace_init(), or called after trace_cleanup(), it returns without 
 *  performing an operation.
 *
 *
 *  NOTE: 
 *
 *  This function cannot be called from interrupt level. It generates a 
 *  LOG message if called from an ISR or tasklet.
 *
 */
void
trace_thread_identify(char *thread_string_p);




/*
 *
 *
 *  Function Name:  trace_thread_remove_identity
 *
 *
 *  Inputs:	None.
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function should only be called from kernel threads.  It removes the
 *  association between PID and 8-character string created by 
 *  trace_thread_identify().  
 *
 *  Threads that terminate abnormally should not call this function.  This way,
 *  the buffer entries left behind by the thread are associated with the 
 *  "thread string" it placed into the buffer prior to making the entries.
 *
 *  If this function is called before calling trace_init(), it logs a message 
 *  to indicate a bug in the code.  It does not call ASSERT() like the user-
 *  space version of the function.  If it is called after a failed call to 
 *  trace_init(), or called after trace_cleanup(), it returns without 
 *  performing an operation.  If it is called when there is no 
 *  entry for the thread in the buffer, it returns without performing an 
 *  operation.
 *
 *
 *  NOTE: 
 *
 *  This function cannot be called from interrupt level. It generates a 
 *  LOG message if called from an ISR or tasklet.
 *
 */
void
trace_thread_remove_identity(void);




/*
 *
 *
 *  Function Name:  trace_store
 *
 *
 *  Inputs:	index		-  Identifies software performing the trace.
 *		event_ID	-  Identifies format of the trace entry.
 *		data_p		-  Address of a trace_data_t structure.  
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function allows software to store execution information into a trace 
 *  buffer.  Callers place the trace data into a trace_data_t structure 
 *  without regard to endianness or word alignment.  The function places the
 *  contents of the structure into the trace buffer indicated by the index.
 *  The event_ID indicates the format of the trace entry such that it can be 
 *  decoded and analyzed.  All entries with the same trace ID share the same 
 *  formatting.  
 *
 *  If the trace_store() function is called before trace_init(), it LOGs an
 *  error message, but does not ASSERT() like the user-level version.  If the 
 *  trace_init() function fails to create the trace buffer, this function does 
 *  not store information and but does not produce an error. 
 *
 *
 */
void trace_store(int module_ID, unsigned short int event_ID, void* data_p);
void trace_store_lvl(int module_ID, unsigned short int event_ID, int lvl, void* data_p);



/*
 *
 *
 *  Function Name:  trace_cleanup
 *
 *
 *  Inputs:	index  -  Identifies buffer to remove.
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function removes the process trace buffer and notifies the Trace 
 *  Subsystem that the application is exiting. When a module is unloaded, or 
 *  terminates normally, it must call this function.  Modules should place a 
 *  call to trace_cleanup() into their un-initialization module.  
 *
 *  The index must be the same one used to create the trace buffer the 
 *  caller was using.
 *
 *  If the application calls the trace_store() function after it invokes this 
 *  one, it does not place data into the buffer.  However, program execution 
 *  is not affected and the function does not generate an error.  This 
 *  handles a potential race condition where one thread calls trace_cleanup() 
 *  and another calls trace_store() before the first thread can terminate the 
 *  process. 
 *
 *  NOTE: 
 *
 *  This function cannot be called from interrupt level. It generates a 
 *  LOG message if called from an ISR or tasklet.
 *
 *
 */
void
trace_cleanup(int index);



/*
 *
 *
 *  Function Name:  trace_dump
 *
 *
 *  Inputs:	dump cause.
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	0 if success, negative if failed.
 *
 *
 *  Description:
 *
 *	This function triggers the trace device doing trace
 *	dump using the ATA polling driver, it is a
 *	non-recoverable event
 *
 */
int
trace_dump(int);

/*
 *
 *
 *  Function Name:  trace_dump_register
 *
 *
 *  Inputs:	func	- dump function block.
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *	This function provides the interface for platform
 *	software to register the platform specific dump functions
 *	such as ATA on CP, flash on BP
 *
 */
void
trace_dump_register(trace_dump_func_t *);


#else
/*  ***** User-level interface *****
 */
#include <sys/time.h>

/*
 * exposed trace entry data to user application
 */
typedef struct trace_entry {
	uint16_t	mod_id;
	uint16_t	evt_id;
	uint16_t	generic_id;
	int			generic;
	struct timeval	time;
	trace_long_data_t	data;
} trace_entry_t;


/*
 * trace callback function
 */
typedef void (*trace_cb_func_t)(trace_entry_t *entry_p);

/*
 * default trace buffer handle
 */
#define	TRACE_DEF_HANDLE	0x1010
#define	TRACE_FIRST_PID		0

/*
 *
 *
 *  Function Name:  trace_process_init
 *
 *
 *  Inputs:	module_ID	-  Each application must identify itself 
 *					when creating a trace buffer.  
 *		number_entries	-  Indicates how many entries stored in 
 *					buffer before wrapping (power of 2).
 *		module_string_p	-  Eight character string (not including 
 *					NULL) or NULL.
 *		prop		-  trace buffer property
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	Returns zero if successful, non-zero if fails (See below)
 *
 *
 *  Description:
 *
 *  This function creates a trace buffer in shared memory and alerts the 
 *  Trace Subsystem that the new buffer exists.  It also records the 
 *  application's module ID in the buffer so that the trace information is 
 *  associated with the new application process.
 *
 *  When a new user-level process begins execution it must call this function 
 *  before it can begin tracing information.  Every thread in the process 
 *  traces information into the buffer.  So, the size of the process's buffer 
 *  must account for the total number of threads the process expects to have.
 *  
 *  If this function is called more than once from the same application, it 
 *  reports an error, but returns normally.  If it is unable to create the trace 
 *  buffer, it LOGs an error and returns TRACE_ERR_NOCREATE.  Subsequent 
 *  calls to trace information do not succeed, but the calling software 
 *  continues to run normally.  If this function fails, it may be retried at a 
 *  later time.
 *
 *
 *  NOTE: 
 *
 *  "number_entries" must be a power of two (2).  If not, the function 
 *  ASSERT()s to indicate that ther is a bug in the software.
 *
 */
int trace_process_init(unsigned int module_ID, 
	unsigned int number_entries, char * module_string_p,
	void *prop);
int trace_process_init_new(unsigned short int module_ID, char * module_string_p);

/*
 *
 *
 *  Function Name:  trace_module_init
 *
 *
 *  Inputs:	hndl		-  trace buffer handle
 *		module_ID	-  Each application must identify itself
 *					when creating a trace buffer.
 *		number_entries	-  Indicates how many entries stored in
 *					buffer before wrapping (power of 2).
 *		module_string_p	-  Eight character string (not including
 *					NULL) or NULL.
 *		prop		-  trace buffer property
 *
 *
 *  Outputs:    None.
 *
 *
 *  Returns:    Returns zero if successful, non-zero if fails (See below)
 *
 *
 *  Description:
 *
 *  This function nitializes the Library Object.  In the future, we may decide
 *  to have multiple Library Objects for a single process.  In this case, only
 *  this function needs to change.
 *
 *  When a new user-level process begins execution it must call this function
 *  before it can begin tracing information.  Every thread in the process
 *  traces information into the buffer.  So, the size of the process's buffer
 *  must account for the total number of threads the process expects to have.
 *
 *  If this function is called more than once from the same application, it
 *  reports an error, but returns normally.  If it is unable to create the
 *  trace buffer, it LOGs an error and returns TRACE_ERR_NOCREATE.  Subsequent
 *  calls to trace information do not succeed, but the calling software
 *  continues to run normally.  If this function fails, it may be retried at a
 *  later time.
 *
 *
 *  NOTE:
 *
 *  "number_entries" must be a power of two (2).  If not, the function
 *  ASSERT()s to indicate that there is a bug in the software.
 *
 */
int trace_module_init(trace_hndl_t *hndl, unsigned int module_ID, 
		unsigned int number_entries, char * module_string_p,
		void *prop);

int trace_module_init_new(trace_hndl_t *hndl, unsigned short int module_ID, 
	char * module_string_p);


/*
 *
 *
 *  Function Name:  trace_data_init
 *
 *
 *  Inputs:	hndl		-  trace buffer handle
 *		module_ID	-  Each application must identify itself
 *					when creating a trace buffer.
 *		number_entries	-  Indicates how many entries stored in
 *					buffer before wrapping (power of 2).
 *		module_string_p	-  Eight character string (not including
 *					NULL) or NULL.
 *		prop		-  trace buffer property
 *
 *
 *  Outputs:    None.
 *
 *
 *  Returns:    Returns zero if successful, non-zero if fails (See below)
 *
 *
 *  Description:
 *
 *  This function nitializes the Library Object.  In the future, we may decide
 *  to have multiple Library Objects for a single process.  In this case, only
 *  this function needs to change.
 *
 *  When a new user-level process begins execution it must call this function
 *  before it can begin tracing information.  Every thread in the process
 *  traces information into the buffer.  So, the size of the process's buffer
 *  must account for the total number of threads the process expects to have.
 *
 *  If this function is called more than once from the same application, it
 *  reports an error, but returns normally.  If it is unable to create the
 *  trace buffer, it LOGs an error and returns TRACE_ERR_NOCREATE.  Subsequent
 *  calls to trace information do not succeed, but the calling software
 *  continues to run normally.  If this function fails, it may be retried at a
 *  later time.
 *
 *
 *  NOTE:
 *
 *  "number_entries" must be a power of two (2).  If not, the function
 *  ASSERT()s to indicate that there is a bug in the software.
 *
 */
int
trace_data_init(trace_hndl_t *hndl, unsigned int module_ID,
	unsigned int size, char *module_string_p,
		int trace_sec, void **buf);


/*
 *
 *
 *  Function Name:  trace_thread_identify
 *
 *
 *  Inputs:	thread_string_p  -  Eight character string (not including
 *					NULL)
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function is called by threads in an application.  It accepts a 
 *  string that is used to identify all trace entries that are generated 
 *  by the calling thread context.
 *
 *  Every entry in the trace buffer contains the OS process ID of the thread 
 *  that created it.  The process ID (of the thread) uniquely identifies each 
 *  thread in a process.  Developers may choose to identify trace entries 
 *  created by a thread with a string to assist post-mortem debugging.  
 *  The application trace buffer supports up to 16 thread identification 
 *  entries.  If there are already sixteen threads identified in the buffer, 
 *  calling trace_thread_identify() generates a LOG message and modifies the 
 *  buffer metrics.
 *
 *  When an application thread exits normally, it should remove its 
 *  identification information from the trace buffer by calling 
 *  trace_thread_remove_identity().
 *
 *  If this function is called before calling trace_process_init(), it logs a
 *  message and ASSERT()s to indicate a bug in the code.  If it is called 
 *  after a failed call to trace_process_init(), or called after 
 *  trace_process_cleanup(), it returns without performing an operation.
 *
 *
 */
void
trace_thread_identify(char *thread_string_p);




/*
 *
 *
 *  Function Name:  trace_thread_remove_identity
 *
 *
 *  Inputs:	None.
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function is called by threads in an application.  It removes the
 *  association between PID and 8-character string created by 
 *  trace_thread_identify().  
 *
 *  Threads that terminate abnormally should not call this function.  This way,
 *  the buffer entries left behind by the thread are associated with the 
 *  "thread string" it placed into the buffer prior to making the entries.
 *
 *  If this function is called before calling trace_process_init(), it logs a 
 *  message and ASSERT()s to indicate a bug in the code.  If it is called after
 *  a failed call to trace_process_init(), or called after 
 *  trace_process_cleanup(), it returns without performing an operation. 
 *  If it is called when there is no entry for the thread in the 
 *  buffer, it logs a message and ASSERT()s to indicate a bug in the code.
 *
 *
 */
void
trace_thread_remove_identity(void);




/*
 *
 *
 *  Function Name:  trace_process_store
 *
 *
 *  Inputs:	module_ID  -  Identifies software performing the trace.
 *		event_ID   -  Identifies format of the trace entry.
 *		data_p     -  Address of a trace_data_t structure.  
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function allows software to store execution information into a trace 
 *  buffer.  Callers place the trace data into a trace_data_t structure 
 *  without regard to endianness or word alignment.  The function places the
 *  contents of the structure into the trace buffer that was created by the 
 *  application process context.  The event_ID indicates the format of the 
 *  trace entry such that it can be decoded and analyzed.  All entries with 
 *  the same trace ID share the same formatting.  
 *
 *  If the trace_store() function is called before trace_process_init(), it 
 *  LOGs an error message and ASSERT()s to indicate a bug in the code.  If the
 *  trace_process_init() function fails to create the trace buffer, this function 
 *  does not store information but does not produce an error. 
 *
 *
 */
void
trace_process_store(unsigned short int module_ID,
	unsigned short int event_ID, void* data_p);
void
trace_process_store_lvl(unsigned short int module_ID,
	unsigned short int event_ID, int lvl, void* data_p);

/*
 *
 *
 *  Function Name:  trace_module_store
 *
 *
 *  Inputs:	hndl       -  trace buffer handle
 *		module_ID  -  Identifies software performing the trace.
 *		event_ID   -  Identifies format of the trace entry.
 *		data_p     -  Address of a trace_data_t structure.
 *
 *
 *  Outputs:    None.
 *
 *
 *  Returns:    None.
 *
 *
 *  Description:
 *
 *  This function allows software to store execution information into a trace
 *  buffer.  Callers place the trace data into a trace_data_t structure
 *  without regard to endianness or word alignment.  The function places the
 *  contents of the structure into the trace buffer that was created by the
 *  application process context.  The event_ID indicates the format of the
 *  trace entry such that it can be decoded and analyzed.  All entries with
 *  the same trace ID share the same formatting.
 *
 *  If the trace_store() function is called before trace_process_init(), it
 *  LOGs an error message and ASSERT()s to indicate a bug in the code.  If the
 *  trace_process_init() function fails to create the trace buffer,
 *  this function does not store information but does not produce an error.
 *
 *
 */
#define trace_module_store(hndl, module_ID, event_ID, data_p) \
	trace_module_store_lvl(hndl, module_ID, event_ID, TRACE_DEFAULT_LVL, data_p)
void
trace_module_store_lvl(trace_hndl_t hndl, unsigned short int module_ID,
	unsigned short int event_ID, int lvl, void* data_p);

#define G4_DEFAULT_VALUE 0
void
trace_module_g4_store_lvl(trace_hndl_t hndl, unsigned short int module_ID, 
unsigned short int g4_value /* default value = 0 */, unsigned short int event_ID, int lvl, void* data_p); 

/*
 *
 *
 *  Function Name:  trace_process_cleanup
 *
 *
 *  Inputs:	None.
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function removes the process trace buffer and notifies the Trace 
 *  Subsystem that the application is exiting. When a process terminates 
 *  normally, it must call trace_process_cleanup(). 
 *
 *  If the application calls the trace_store() function after it invokes this 
 *  one, it does not place data into the buffer.  However, program execution 
 *  is not affected and the function does not generate an error.  This 
 *  handles a potential race condition where one thread calls 
 *  trace_process_cleanup() and another calls trace_store() before the first 
 *  thread can terminate the process. 
 *
 *
 */
void
trace_process_cleanup(void);

/*
 *
 *
 *  Function Name:  trace_module_cleanup
 *
 *
 *  Inputs:	hndl	- trace buffer handle.
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function removes the process trace buffer and notifies the Trace
 *  Subsystem that the application is exiting. When a process terminates
 *  normally, it must call trace_process_cleanup().
 *
 *  If the application calls the trace_store() function after it invokes this
 *  one, it does not place data into the buffer.  However, program execution
 *  is not affected and the function does not generate an error.  This
 *  handles a potential race condition where one thread calls
 *  trace_process_cleanup() and another calls trace_store() before the first
 *  thread can terminate the process.
 *
 *
 */
void
trace_module_cleanup(trace_hndl_t hndl);



/*
 * trace_hndl_t trace_attach(int mod_id, int pid, int number_entries)
 *
 * Attach to an existing trace buffer
 *
 * Inputs:
 *	mod_id	- module id of the buffer
 *	pid	- pid of the buffer
 *	number_entries - number of trace entries
 *
 * Outputs:
 *	N/A
 *
 * Returns:
 *	trace buffer handle, -1 if fail
 *
 */
trace_hndl_t
trace_attach(int mod_id, int pid, int number_entries);

/*
 * int trace_traverse(trace_hndl_t hndl, int back, trace_cb_func_t func)
 *
 * Traverse the buffer buffer backward or forward
 *
 * Inputs:
 *	hndl	- trace buffer handle
 *	back	- traverse forward (from earlier to latest) or backwards
 *	func	- call back function to process the entry
 *
 * Outputs:
 *	N/A
 *
 * Returns:
 *	TRACE_SUCCESS if succeed, otherwise TRACE_ERROR
 *
 */
int
trace_traverse(trace_hndl_t hndl, int back, trace_cb_func_t func);

/*
 * int trace_snapshot(trace_hndl_t hndl, int lock,
 *                      trace_entry_t **entry_list_p, int *num_p)
 *
 * Take a snapshot of the whole buffer
 *
 * Inputs:
 *	hndl	- trace buffer handle
 *	lock	- lock the buffer during taking snapshot
 *
 * Outputs:
 *	entry_list_p	- entry list (the caller needs to free the list)
 *	num_p		- number of entries
 *
 * Returns:
 *	TRACE_SUCCESS if succeed, otherwise TRACE_ERROR
 *
 */
int
trace_snapshot(trace_hndl_t hndl, int lock,
	trace_entry_t **entry_list_p, int *num_p);

int
trace_snapshot_filter(trace_hndl_t hndl, int lock,
	trace_entry_t **entry_list_p, int *num_p, trace_filter_t filter);

/*
 * int trace_resize(trace_hndl_t hndl, int new_num)
 *
 * Traverse the buffer buffer backward or forward
 *
 * Inputs:
 *	hndl	- trace buffer handle
 *	new_num	- new number of entries
 *
 * Outputs:
 *	N/A
 *
 * Returns:
 *	TRACE_SUCCESS if succeed, otherwise TRACE_ERROR
 *
 */
int
trace_resize(trace_hndl_t hndl, int new_num);


/*
 * int trace_reset(trace_hndl_t hndl)
 *
 * Clears trace buffer
 *
 * Inputs:
 *  hndl    - trace buffer handle
 *
 * Outputs:
 *  N/A
 *
 * Returns:
 *  TRACE_SUCCESS if succeed, otherwise TRACE_ERROR
 *
 */

int
trace_reset(trace_hndl_t hndl);


/*
 * int trace_clear(trace_hndl_t hndl)
 *
 * Clears trace buffer
 *
 * Inputs:
 *  hndl    - trace buffer handle
 *
 * Outputs:
 *  N/A
 *
 * Returns:
 *  TRACE_SUCCESS if succeed, otherwise TRACE_ERROR
 *
 */

int
trace_clear(trace_hndl_t hndl);


/*
 * int trace_get_buf_size(int mod_id)
 * Returns the max buffer size configured for the
 * module id passed as argument
 * Inputs:
 * mod_id - module ID
 *
 * Outputs:
 *	N/A
 *
 * Returns:
 * the max buffer size configured 
 */

int
trace_get_buf_size(int mod_id);
#endif   // end user space API

extern trace_hndl_t	process_hndl;

// define Default trace handle to be process_hndl
#define	DHNDL	process_hndl

static inline void
trace_cat2str(int len, char *dest, char *str1)
{
	*dest++ = ':'; 
	len--;
	while ((1 < len--) && str1 && *str1) {
		*dest++ = *str1++;
	}
	*dest = '\0';
}

static inline void
trace_cat3str(int len, char *dest, char *str1, char *str2)
{
    *dest++ = ':'; 
    len--;
    while ((1 < len--) && str1 && *str1) {
	*dest++ = *str1++;
    }
    *dest++ = ':';
    while ((1 < len--) && str2 && *str2) {
	*dest++ = *str2++;
    }
    *dest = '\0';
}

static inline void
trace_cat4str(int len, char *dest, char *str1, char *str2, char *str3)
{
    *dest++ = ':'; 
    len--;
    while ((1 < len--) && str1 && *str1) {
	*dest++ = *str1++;
    }
    *dest++ = ':';
    while ((1 < len--) && str2 && *str2) {
	*dest++ = *str2++;
    }
    *dest++ = ':';
    while ((1 < len--) && str3 && *str3) {
	*dest++ = *str3++;
    }
    *dest = '\0';
}

/* following section could be done by direct include /include/trace/gtrace.h file
 * #include <trace/gtrace.h>  
*/

/*
 * This file has been auto-generated by gtrace.pl to be include only in ras_trace.h,
 * find Makefile that builds corresponding code.
 * (Name of #ifndef below should be a hint.)
 *
 * Copyright (c) 2011 Brocade Communications Systems, Inc.
 */
#ifndef __GTRACE_H__
#define __GTRACE_H__
#ifndef __DCM_COMPILATION__
/* ***********************************************************************************
 *   GTRACE_HAL scession
 ************************************************************************************/
//  static int dbg = 20;
// specific trace handle, area and level with 0 integer and 0 string argument
static inline void
GTRACE_HAL(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);   

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;
    /* string always at last arg */

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
		printk(KERN_CRIT "%s %s\n", e_ids, e_fmt);
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt);
	printf("\n");
    }
#endif
}

// default trace handle, area and level with 0 integer and 0 string argument
#define GTRACE(MID, EID, EFMT)             \
	GTRACE_HAL(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT)

// trace handle, default area and level with 0 integer and 0 string argument
#define GTRACE_H(MID, HNDL, EID, EFMT)             \
	GTRACE_HAL(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT)

// default trace handle, area and spdcific level with 0 integer and 0 string argument
#define GTRACE_L(MID, LEVEL, EID, EFMT)             \
	GTRACE_HAL(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT)

// default trace handle, specific area and default level with 0 integer and 0 string argument
#define GTRACE_A(MID, AREA, EID, EFMT)             \
	GTRACE_HAL(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT)

// trace handl, default area and specific level with 0 integer and 0 string argument
#define GTRACE_HL(MID, HNDL, LEVEL, EID, EFMT)             \
	GTRACE_HAL(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT)

// trace handl, specific area and default level with 0 integer and 0 string argument
#define GTRACE_HA(MID, HNDL, AREA, EID, EFMT)             \
	GTRACE_HAL(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT)

// trace handl, specific area and level with 0 integer and 0 string argument
#define GTRACE_AL(MID, AREA, LEVEL, EID, EFMT)             \
	GTRACE_HAL(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT)

/* ***********************************************************************************
 *   GTRACE_HAL_I1 scession
 ************************************************************************************/

// specific trace handle, area and level with 1 integer and 0 string argument
static inline void
GTRACE_HAL_I1(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
, int i1)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;
    data.td_word[0] = i1;

    /* string always at last arg */

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
         printk(KERN_CRIT "%s,", e_ids); printk(e_fmt, i1);
	printk(KERN_CRIT "\n");
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt, i1);
	printf("\n");
    }
#endif
}

// default trace handle, area and level with 1 integer and 0 string argument
#define GTRACE_I1(MID, EID, EFMT, I1)             \
	GTRACE_HAL_I1(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1)

// trace handle, default area and level with 1 integer and 0 string argument
#define GTRACE_H_I1(MID, HNDL, EID, EFMT, I1)             \
	GTRACE_HAL_I1(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1)

// default trace handle, area and spdcific level with 1 integer and 0 string argument
#define GTRACE_L_I1(MID, LEVEL, EID, EFMT, I1)             \
	GTRACE_HAL_I1(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT, I1)

// default trace handle, specific area and default level with 1 integer and 0 string argument
#define GTRACE_A_I1(MID, AREA, EID, EFMT, I1)             \
	GTRACE_HAL_I1(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT, I1)

// trace handl, default area and specific level with 1 integer and 0 string argument
#define GTRACE_HL_I1(MID, HNDL, LEVEL, EID, EFMT, I1)             \
	GTRACE_HAL_I1(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT, I1)

// trace handl, specific area and default level with 1 integer and 0 string argument
#define GTRACE_HA_I1(MID, HNDL, AREA, EID, EFMT, I1)             \
	GTRACE_HAL_I1(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT, I1)

// trace handl, specific area and level with 1 integer and 0 string argument
#define GTRACE_AL_I1(MID, AREA, LEVEL, EID, EFMT, I1)             \
	GTRACE_HAL_I1(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT, I1)

/* ***********************************************************************************
 *   GTRACE_HAL_I2 scession
 ************************************************************************************/

// specific trace handle, area and level with 2 integer and 0 string argument
static inline void
GTRACE_HAL_I2(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
, int i1, int i2)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;

    data.td_word[0] = i1;
    data.td_word[1] = i2;

    /* string always at last arg */

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
         printk(KERN_CRIT "%s,", e_ids); printk(e_fmt, i1, i2);
	printk(KERN_CRIT "\n");
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt, i1, i2);
	printf("\n");
    }
#endif
}

// default trace handle, area and level with 2 integer and 0 string argument
#define GTRACE_I2(MID, EID, EFMT, I1, I2)             \
	GTRACE_HAL_I2(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2)

// trace handle, default area and level with 2 integer and 0 string argument
#define GTRACE_H_I2(MID, HNDL, EID, EFMT, I1, I2)             \
	GTRACE_HAL_I2(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2)

// default trace handle, area and spdcific level with 2 integer and 0 string argument
#define GTRACE_L_I2(MID, LEVEL, EID, EFMT, I1, I2)             \
	GTRACE_HAL_I2(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT, I1, I2)

// default trace handle, specific area and default level with 2 integer and 0 string argument
#define GTRACE_A_I2(MID, AREA, EID, EFMT, I1, I2)             \
	GTRACE_HAL_I2(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT, I1, I2)

// trace handl, default area and specific level with 2 integer and 0 string argument
#define GTRACE_HL_I2(MID, HNDL, LEVEL, EID, EFMT, I1, I2)             \
	GTRACE_HAL_I2(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT, I1, I2)

// trace handl, specific area and default level with 2 integer and 0 string argument
#define GTRACE_HA_I2(MID, HNDL, AREA, EID, EFMT, I1, I2)             \
	GTRACE_HAL_I2(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT, I1, I2)

// trace handl, specific area and level with 2 integer and 0 string argument
#define GTRACE_AL_I2(MID, AREA, LEVEL, EID, EFMT, I1, I2)             \
	GTRACE_HAL_I2(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT, I1, I2)

/* ***********************************************************************************
 *   GTRACE_HAL_I3 scession
 ************************************************************************************/

// specific trace handle, area and level with 3 integer and 0 string argument
static inline void
GTRACE_HAL_I3(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
, int i1, int i2, int i3)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;

    data.td_word[0] = i1;
    data.td_word[1] = i2;
    data.td_word[2] = i3;

    /* string always at last arg */

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
         printk(KERN_CRIT "%s,", e_ids); printk(e_fmt, i1, i2, i3);
	printk(KERN_CRIT "\n");
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt, i1, i2, i3);
	printf("\n");
    }
#endif
}

// default trace handle, area and level with 3 integer and 0 string argument
#define GTRACE_I3(MID, EID, EFMT, I1, I2, I3)             \
	GTRACE_HAL_I3(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3)

// trace handle, default area and level with 3 integer and 0 string argument
#define GTRACE_H_I3(MID, HNDL, EID, EFMT, I1, I2, I3)             \
	GTRACE_HAL_I3(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3)

// default trace handle, area and spdcific level with 3 integer and 0 string argument
#define GTRACE_L_I3(MID, LEVEL, EID, EFMT, I1, I2, I3)             \
	GTRACE_HAL_I3(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT, I1, I2, I3)

// default trace handle, specific area and default level with 3 integer and 0 string argument
#define GTRACE_A_I3(MID, AREA, EID, EFMT, I1, I2, I3)             \
	GTRACE_HAL_I3(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT, I1, I2, I3)

// trace handl, default area and specific level with 3 integer and 0 string argument
#define GTRACE_HL_I3(MID, HNDL, LEVEL, EID, EFMT, I1, I2, I3)             \
	GTRACE_HAL_I3(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT, I1, I2, I3)

// trace handl, specific area and default level with 3 integer and 0 string argument
#define GTRACE_HA_I3(MID, HNDL, AREA, EID, EFMT, I1, I2, I3)             \
	GTRACE_HAL_I3(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT, I1, I2, I3)

// trace handl, specific area and level with 3 integer and 0 string argument
#define GTRACE_AL_I3(MID, AREA, LEVEL, EID, EFMT, I1, I2, I3)             \
	GTRACE_HAL_I3(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT, I1, I2, I3)

/* ***********************************************************************************
 *   GTRACE_HAL_I4 scession
 ************************************************************************************/

// specific trace handle, area and level with 4 integer and 0 string argument
static inline void
GTRACE_HAL_I4(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
, int i1, int i2, int i3, int i4)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;

    data.td_word[0] = i1;
    data.td_word[1] = i2;
    data.td_word[2] = i3;
    data.td_word[3] = i4;

    /* string always at last arg */

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
         printk(KERN_CRIT "%s,", e_ids); printk(e_fmt, i1, i2, i3, i4);
	printk(KERN_CRIT "\n");
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt, i1, i2, i3, i4);
	printf("\n");
    }
#endif
}

// default trace handle, area and level with 4 integer and 0 string argument
#define GTRACE_I4(MID, EID, EFMT, I1, I2, I3, I4)             \
	GTRACE_HAL_I4(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4)

// trace handle, default area and level with 4 integer and 0 string argument
#define GTRACE_H_I4(MID, HNDL, EID, EFMT, I1, I2, I3, I4)             \
	GTRACE_HAL_I4(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4)

// default trace handle, area and spdcific level with 4 integer and 0 string argument
#define GTRACE_L_I4(MID, LEVEL, EID, EFMT, I1, I2, I3, I4)             \
	GTRACE_HAL_I4(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4)

// default trace handle, specific area and default level with 4 integer and 0 string argument
#define GTRACE_A_I4(MID, AREA, EID, EFMT, I1, I2, I3, I4)             \
	GTRACE_HAL_I4(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT, I1, I2, I3, I4)

// trace handl, default area and specific level with 4 integer and 0 string argument
#define GTRACE_HL_I4(MID, HNDL, LEVEL, EID, EFMT, I1, I2, I3, I4)             \
	GTRACE_HAL_I4(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT, I1, I2, I3, I4)

// trace handl, specific area and default level with 4 integer and 0 string argument
#define GTRACE_HA_I4(MID, HNDL, AREA, EID, EFMT, I1, I2, I3, I4)             \
	GTRACE_HAL_I4(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT, I1, I2, I3, I4)

// trace handl, specific area and level with 4 integer and 0 string argument
#define GTRACE_AL_I4(MID, AREA, LEVEL, EID, EFMT, I1, I2, I3, I4)             \
	GTRACE_HAL_I4(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT, I1, I2, I3, I4)

/* ***********************************************************************************
 *   GTRACE_HAL_I5 scession
 ************************************************************************************/

// specific trace handle, area and level with 5 integer and 0 string argument
static inline void
GTRACE_HAL_I5(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
, int i1, int i2, int i3, int i4, int i5)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;

    data.td_word[0] = i1;
    data.td_word[1] = i2;
    data.td_word[2] = i3;
    data.td_word[3] = i4;
    data.td_word[4] = i5;

    /* string always at last arg */

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
         printk(KERN_CRIT "%s,", e_ids); printk(e_fmt, i1, i2, i3, i4, i5);
	printk(KERN_CRIT "\n");
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt, i1, i2, i3, i4, i5);
	printf("\n");
    }
#endif
}

// default trace handle, area and level with 5 integer and 0 string argument
#define GTRACE_I5(MID, EID, EFMT, I1, I2, I3, I4, I5)             \
	GTRACE_HAL_I5(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5)

// trace handle, default area and level with 5 integer and 0 string argument
#define GTRACE_H_I5(MID, HNDL, EID, EFMT, I1, I2, I3, I4, I5)             \
	GTRACE_HAL_I5(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5)

// default trace handle, area and spdcific level with 5 integer and 0 string argument
#define GTRACE_L_I5(MID, LEVEL, EID, EFMT, I1, I2, I3, I4, I5)             \
	GTRACE_HAL_I5(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5)

// default trace handle, specific area and default level with 5 integer and 0 string argument
#define GTRACE_A_I5(MID, AREA, EID, EFMT, I1, I2, I3, I4, I5)             \
	GTRACE_HAL_I5(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5)

// trace handl, default area and specific level with 5 integer and 0 string argument
#define GTRACE_HL_I5(MID, HNDL, LEVEL, EID, EFMT, I1, I2, I3, I4, I5)             \
	GTRACE_HAL_I5(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5)

// trace handl, specific area and default level with 5 integer and 0 string argument
#define GTRACE_HA_I5(MID, HNDL, AREA, EID, EFMT, I1, I2, I3, I4, I5)             \
	GTRACE_HAL_I5(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5)

// trace handl, specific area and level with 5 integer and 0 string argument
#define GTRACE_AL_I5(MID, AREA, LEVEL, EID, EFMT, I1, I2, I3, I4, I5)             \
	GTRACE_HAL_I5(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5)

/* ***********************************************************************************
 *   GTRACE_HAL_I6 scession
 ************************************************************************************/

// specific trace handle, area and level with 6 integer and 0 string argument
static inline void
GTRACE_HAL_I6(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
, int i1, int i2, int i3, int i4, int i5, int i6)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;

    data.td_word[0] = i1;
    data.td_word[1] = i2;
    data.td_word[2] = i3;
    data.td_word[3] = i4;
    data.td_word[4] = i5;
    data.td_word[5] = i6;

    /* string always at last arg */

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
         printk(KERN_CRIT "%s,", e_ids); printk(e_fmt, i1, i2, i3, i4, i5, i6);
	printk(KERN_CRIT "\n");
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt, i1, i2, i3, i4, i5, i6);
	printf("\n");
    }
#endif
}

// default trace handle, area and level with 6 integer and 0 string argument
#define GTRACE_I6(MID, EID, EFMT, I1, I2, I3, I4, I5, I6)             \
	GTRACE_HAL_I6(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6)

// trace handle, default area and level with 6 integer and 0 string argument
#define GTRACE_H_I6(MID, HNDL, EID, EFMT, I1, I2, I3, I4, I5, I6)             \
	GTRACE_HAL_I6(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6)

// default trace handle, area and spdcific level with 6 integer and 0 string argument
#define GTRACE_L_I6(MID, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6)             \
	GTRACE_HAL_I6(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6)

// default trace handle, specific area and default level with 6 integer and 0 string argument
#define GTRACE_A_I6(MID, AREA, EID, EFMT, I1, I2, I3, I4, I5, I6)             \
	GTRACE_HAL_I6(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6)

// trace handl, default area and specific level with 6 integer and 0 string argument
#define GTRACE_HL_I6(MID, HNDL, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6)             \
	GTRACE_HAL_I6(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6)

// trace handl, specific area and default level with 6 integer and 0 string argument
#define GTRACE_HA_I6(MID, HNDL, AREA, EID, EFMT, I1, I2, I3, I4, I5, I6)             \
	GTRACE_HAL_I6(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6)

// trace handl, specific area and level with 6 integer and 0 string argument
#define GTRACE_AL_I6(MID, AREA, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6)             \
	GTRACE_HAL_I6(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6)

/* ***********************************************************************************
 *   GTRACE_HAL_I7 scession
 ************************************************************************************/

// specific trace handle, area and level with 7 integer and 0 string argument
static inline void
GTRACE_HAL_I7(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
, int i1, int i2, int i3, int i4, int i5, int i6, int i7)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;

    data.td_word[0] = i1;
    data.td_word[1] = i2;
    data.td_word[2] = i3;
    data.td_word[3] = i4;
    data.td_word[4] = i5;
    data.td_word[5] = i6;
    data.td_word[6] = i7;

    /* string always at last arg */

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
         printk(KERN_CRIT "%s,", e_ids); printk(e_fmt, i1, i2, i3, i4, i5, i6, i7);
	printk(KERN_CRIT "\n");
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt, i1, i2, i3, i4, i5, i6, i7);
	printf("\n");
    }
#endif
}

// default trace handle, area and level with 7 integer and 0 string argument
#define GTRACE_I7(MID, EID, EFMT, I1, I2, I3, I4, I5, I6, I7)             \
	GTRACE_HAL_I7(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7)

// trace handle, default area and level with 7 integer and 0 string argument
#define GTRACE_H_I7(MID, HNDL, EID, EFMT, I1, I2, I3, I4, I5, I6, I7)             \
	GTRACE_HAL_I7(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7)

// default trace handle, area and spdcific level with 7 integer and 0 string argument
#define GTRACE_L_I7(MID, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6, I7)             \
	GTRACE_HAL_I7(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7)

// default trace handle, specific area and default level with 7 integer and 0 string argument
#define GTRACE_A_I7(MID, AREA, EID, EFMT, I1, I2, I3, I4, I5, I6, I7)             \
	GTRACE_HAL_I7(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7)

// trace handl, default area and specific level with 7 integer and 0 string argument
#define GTRACE_HL_I7(MID, HNDL, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6, I7)             \
	GTRACE_HAL_I7(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7)

// trace handl, specific area and default level with 7 integer and 0 string argument
#define GTRACE_HA_I7(MID, HNDL, AREA, EID, EFMT, I1, I2, I3, I4, I5, I6, I7)             \
	GTRACE_HAL_I7(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7)

// trace handl, specific area and level with 7 integer and 0 string argument
#define GTRACE_AL_I7(MID, AREA, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6, I7)             \
	GTRACE_HAL_I7(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7)

/* ***********************************************************************************
 *   GTRACE_HAL_I8 scession
 ************************************************************************************/

// specific trace handle, area and level with 8 integer and 0 string argument
static inline void
GTRACE_HAL_I8(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
, int i1, int i2, int i3, int i4, int i5, int i6, int i7, int i8)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;

    data.td_word[0] = i1;
    data.td_word[1] = i2;
    data.td_word[2] = i3;
    data.td_word[3] = i4;
    data.td_word[4] = i5;
    data.td_word[5] = i6;
    data.td_word[6] = i7;
    data.td_word[7] = i8;

    /* string always at last arg */

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
         printk(KERN_CRIT "%s,", e_ids); printk(e_fmt, i1, i2, i3, i4, i5, i6, i7, i8);
	printk(KERN_CRIT "\n");
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt, i1, i2, i3, i4, i5, i6, i7, i8);
	printf("\n");
    }
#endif
}

// default trace handle, area and level with 8 integer and 0 string argument
#define GTRACE_I8(MID, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8)             \
	GTRACE_HAL_I8(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8)

// trace handle, default area and level with 8 integer and 0 string argument
#define GTRACE_H_I8(MID, HNDL, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8)             \
	GTRACE_HAL_I8(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8)

// default trace handle, area and spdcific level with 8 integer and 0 string argument
#define GTRACE_L_I8(MID, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8)             \
	GTRACE_HAL_I8(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8)

// default trace handle, specific area and default level with 8 integer and 0 string argument
#define GTRACE_A_I8(MID, AREA, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8)             \
	GTRACE_HAL_I8(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8)

// trace handl, default area and specific level with 8 integer and 0 string argument
#define GTRACE_HL_I8(MID, HNDL, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8)             \
	GTRACE_HAL_I8(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8)

// trace handl, specific area and default level with 8 integer and 0 string argument
#define GTRACE_HA_I8(MID, HNDL, AREA, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8)             \
	GTRACE_HAL_I8(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8)

// trace handl, specific area and level with 8 integer and 0 string argument
#define GTRACE_AL_I8(MID, AREA, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8)             \
	GTRACE_HAL_I8(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8)

/* ***********************************************************************************
 *   GTRACE_HAL_S scession
 ************************************************************************************/

// specific trace handle, area and level with 0 integer and 1 string argument
static inline void
GTRACE_HAL_S(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
, char *str)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;


    /* string always at last arg */
    if (cfg->entry_size > 0)
        trace_cat2str(cfg->entry_size - 0, (char *)&data.td_word[0], (str) ? (str) : "null");

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
         printk(KERN_CRIT "%s,", e_ids); printk(e_fmt, (str) ? (str) : "null");
	printk(KERN_CRIT "\n");
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt, (str) ? (str) : "null");
	printf("\n");
    }
#endif
}

// default trace handle, area and level with 0 integer and 1 string argument
#define GTRACE_S(MID, EID, EFMT, S)             \
	GTRACE_HAL_S(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, S)

// trace handle, default area and level with 0 integer and 1 string argument
#define GTRACE_H_S(MID, HNDL, EID, EFMT, S)             \
	GTRACE_HAL_S(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, S)

// default trace handle, area and spdcific level with 0 integer and 1 string argument
#define GTRACE_L_S(MID, LEVEL, EID, EFMT, S)             \
	GTRACE_HAL_S(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT, S)

// default trace handle, specific area and default level with 0 integer and 1 string argument
#define GTRACE_A_S(MID, AREA, EID, EFMT, S)             \
	GTRACE_HAL_S(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT, S)

// trace handl, default area and specific level with 0 integer and 1 string argument
#define GTRACE_HL_S(MID, HNDL, LEVEL, EID, EFMT, S)             \
	GTRACE_HAL_S(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT, S)

// trace handl, specific area and default level with 0 integer and 1 string argument
#define GTRACE_HA_S(MID, HNDL, AREA, EID, EFMT, S)             \
	GTRACE_HAL_S(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT, S)

// trace handl, specific area and level with 0 integer and 1 string argument
#define GTRACE_AL_S(MID, AREA, LEVEL, EID, EFMT, S)             \
	GTRACE_HAL_S(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT, S)

/* ***********************************************************************************
 *   GTRACE_HAL_I1S scession
 ************************************************************************************/

// specific trace handle, area and level with 1 integer and 1 string argument
static inline void
GTRACE_HAL_I1S(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
, int i1, char *str)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;

    data.td_word[0] = i1;

    /* string always at last arg */
    if (cfg->entry_size > 4)
        trace_cat2str(cfg->entry_size - 4, (char *)&data.td_word[1], (str) ? (str) : "null");

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
         printk(KERN_CRIT "%s,", e_ids); printk(e_fmt, i1, (str) ? (str) : "null");
	printk(KERN_CRIT "\n");
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt, i1, (str) ? (str) : "null");
	printf("\n");
    }
#endif
}

// default trace handle, area and level with 1 integer and 1 string argument
#define GTRACE_I1S(MID, EID, EFMT, I1, S)             \
	GTRACE_HAL_I1S(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, S)

// trace handle, default area and level with 1 integer and 1 string argument
#define GTRACE_H_I1S(MID, HNDL, EID, EFMT, I1, S)             \
	GTRACE_HAL_I1S(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, S)

// default trace handle, area and spdcific level with 1 integer and 1 string argument
#define GTRACE_L_I1S(MID, LEVEL, EID, EFMT, I1, S)             \
	GTRACE_HAL_I1S(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT, I1, S)

// default trace handle, specific area and default level with 1 integer and 1 string argument
#define GTRACE_A_I1S(MID, AREA, EID, EFMT, I1, S)             \
	GTRACE_HAL_I1S(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT, I1, S)

// trace handl, default area and specific level with 1 integer and 1 string argument
#define GTRACE_HL_I1S(MID, HNDL, LEVEL, EID, EFMT, I1, S)             \
	GTRACE_HAL_I1S(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT, I1, S)

// trace handl, specific area and default level with 1 integer and 1 string argument
#define GTRACE_HA_I1S(MID, HNDL, AREA, EID, EFMT, I1, S)             \
	GTRACE_HAL_I1S(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT, I1, S)

// trace handl, specific area and level with 1 integer and 1 string argument
#define GTRACE_AL_I1S(MID, AREA, LEVEL, EID, EFMT, I1, S)             \
	GTRACE_HAL_I1S(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT, I1, S)

/* ***********************************************************************************
 *   GTRACE_HAL_I2S scession
 ************************************************************************************/

// specific trace handle, area and level with 2 integer and 1 string argument
static inline void
GTRACE_HAL_I2S(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
, int i1, int i2, char *str)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;

    data.td_word[0] = i1;
    data.td_word[1] = i2;

    /* string always at last arg */
    if (cfg->entry_size > 8)
        trace_cat2str(cfg->entry_size - 8, (char *)&data.td_word[2], (str) ? (str) : "null");

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
         printk(KERN_CRIT "%s,", e_ids); printk(e_fmt, i1, i2, (str) ? (str) : "null");
	printk(KERN_CRIT "\n");
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt, i1, i2, (str) ? (str) : "null");
	printf("\n");
    }
#endif
}

// default trace handle, area and level with 2 integer and 1 string argument
#define GTRACE_I2S(MID, EID, EFMT, I1, I2, S)             \
	GTRACE_HAL_I2S(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, S)

// trace handle, default area and level with 2 integer and 1 string argument
#define GTRACE_H_I2S(MID, HNDL, EID, EFMT, I1, I2, S)             \
	GTRACE_HAL_I2S(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, S)

// default trace handle, area and spdcific level with 2 integer and 1 string argument
#define GTRACE_L_I2S(MID, LEVEL, EID, EFMT, I1, I2, S)             \
	GTRACE_HAL_I2S(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT, I1, I2, S)

// default trace handle, specific area and default level with 2 integer and 1 string argument
#define GTRACE_A_I2S(MID, AREA, EID, EFMT, I1, I2, S)             \
	GTRACE_HAL_I2S(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT, I1, I2, S)

// trace handl, default area and specific level with 2 integer and 1 string argument
#define GTRACE_HL_I2S(MID, HNDL, LEVEL, EID, EFMT, I1, I2, S)             \
	GTRACE_HAL_I2S(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT, I1, I2, S)

// trace handl, specific area and default level with 2 integer and 1 string argument
#define GTRACE_HA_I2S(MID, HNDL, AREA, EID, EFMT, I1, I2, S)             \
	GTRACE_HAL_I2S(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT, I1, I2, S)

// trace handl, specific area and level with 2 integer and 1 string argument
#define GTRACE_AL_I2S(MID, AREA, LEVEL, EID, EFMT, I1, I2, S)             \
	GTRACE_HAL_I2S(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT, I1, I2, S)

/* ***********************************************************************************
 *   GTRACE_HAL_I3S scession
 ************************************************************************************/

// specific trace handle, area and level with 3 integer and 1 string argument
static inline void
GTRACE_HAL_I3S(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
, int i1, int i2, int i3, char *str)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;

    data.td_word[0] = i1;
    data.td_word[1] = i2;
    data.td_word[2] = i3;

    /* string always at last arg */
    if (cfg->entry_size > 12)
        trace_cat2str(cfg->entry_size - 12, (char *)&data.td_word[3], (str) ? (str) : "null");

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
         printk(KERN_CRIT "%s,", e_ids); printk(e_fmt, i1, i2, i3, (str) ? (str) : "null");
	printk(KERN_CRIT "\n");
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt, i1, i2, i3, (str) ? (str) : "null");
	printf("\n");
    }
#endif
}

// default trace handle, area and level with 3 integer and 1 string argument
#define GTRACE_I3S(MID, EID, EFMT, I1, I2, I3, S)             \
	GTRACE_HAL_I3S(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, S)

// trace handle, default area and level with 3 integer and 1 string argument
#define GTRACE_H_I3S(MID, HNDL, EID, EFMT, I1, I2, I3, S)             \
	GTRACE_HAL_I3S(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, S)

// default trace handle, area and spdcific level with 3 integer and 1 string argument
#define GTRACE_L_I3S(MID, LEVEL, EID, EFMT, I1, I2, I3, S)             \
	GTRACE_HAL_I3S(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT, I1, I2, I3, S)

// default trace handle, specific area and default level with 3 integer and 1 string argument
#define GTRACE_A_I3S(MID, AREA, EID, EFMT, I1, I2, I3, S)             \
	GTRACE_HAL_I3S(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT, I1, I2, I3, S)

// trace handl, default area and specific level with 3 integer and 1 string argument
#define GTRACE_HL_I3S(MID, HNDL, LEVEL, EID, EFMT, I1, I2, I3, S)             \
	GTRACE_HAL_I3S(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT, I1, I2, I3, S)

// trace handl, specific area and default level with 3 integer and 1 string argument
#define GTRACE_HA_I3S(MID, HNDL, AREA, EID, EFMT, I1, I2, I3, S)             \
	GTRACE_HAL_I3S(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT, I1, I2, I3, S)

// trace handl, specific area and level with 3 integer and 1 string argument
#define GTRACE_AL_I3S(MID, AREA, LEVEL, EID, EFMT, I1, I2, I3, S)             \
	GTRACE_HAL_I3S(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT, I1, I2, I3, S)

/* ***********************************************************************************
 *   GTRACE_HAL_I4S scession
 ************************************************************************************/

// specific trace handle, area and level with 4 integer and 1 string argument
static inline void
GTRACE_HAL_I4S(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
, int i1, int i2, int i3, int i4, char *str)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;

    data.td_word[0] = i1;
    data.td_word[1] = i2;
    data.td_word[2] = i3;
    data.td_word[3] = i4;

    /* string always at last arg */
    if (cfg->entry_size > 16)
        trace_cat2str(cfg->entry_size - 16, (char *)&data.td_word[4], (str) ? (str) : "null");

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
         printk(KERN_CRIT "%s,", e_ids); printk(e_fmt, i1, i2, i3, i4, (str) ? (str) : "null");
	printk(KERN_CRIT "\n");
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt, i1, i2, i3, i4, (str) ? (str) : "null");	
	printf("\n");
    }
#endif
}

// default trace handle, area and level with 4 integer and 1 string argument
#define GTRACE_I4S(MID, EID, EFMT, I1, I2, I3, I4, S)             \
	GTRACE_HAL_I4S(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, S)

// trace handle, default area and level with 4 integer and 1 string argument
#define GTRACE_H_I4S(MID, HNDL, EID, EFMT, I1, I2, I3, I4, S)             \
	GTRACE_HAL_I4S(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, S)

// default trace handle, area and spdcific level with 4 integer and 1 string argument
#define GTRACE_L_I4S(MID, LEVEL, EID, EFMT, I1, I2, I3, I4, S)             \
	GTRACE_HAL_I4S(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, S)

// default trace handle, specific area and default level with 4 integer and 1 string argument
#define GTRACE_A_I4S(MID, AREA, EID, EFMT, I1, I2, I3, I4, S)             \
	GTRACE_HAL_I4S(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT, I1, I2, I3, I4, S)

// trace handl, default area and specific level with 4 integer and 1 string argument
#define GTRACE_HL_I4S(MID, HNDL, LEVEL, EID, EFMT, I1, I2, I3, I4, S)             \
	GTRACE_HAL_I4S(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT, I1, I2, I3, I4, S)

// trace handl, specific area and default level with 4 integer and 1 string argument
#define GTRACE_HA_I4S(MID, HNDL, AREA, EID, EFMT, I1, I2, I3, I4, S)             \
	GTRACE_HAL_I4S(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT, I1, I2, I3, I4, S)

// trace handl, specific area and level with 4 integer and 1 string argument
#define GTRACE_AL_I4S(MID, AREA, LEVEL, EID, EFMT, I1, I2, I3, I4, S)             \
	GTRACE_HAL_I4S(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT, I1, I2, I3, I4, S)

/* ***********************************************************************************
 *   GTRACE_HAL_I5S scession
 ************************************************************************************/

// specific trace handle, area and level with 5 integer and 1 string argument
static inline void
GTRACE_HAL_I5S(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
, int i1, int i2, int i3, int i4, int i5, char *str)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;

    data.td_word[0] = i1;
    data.td_word[1] = i2;
    data.td_word[2] = i3;
    data.td_word[3] = i4;
    data.td_word[4] = i5;

    /* string always at last arg */
    if (cfg->entry_size > 20)
        trace_cat2str(cfg->entry_size - 20, (char *)&data.td_word[5], (str) ? (str) : "null");

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
         printk(KERN_CRIT "%s,", e_ids); printk(e_fmt, i1, i2, i3, i4, i5, (str) ? (str) : "null");
	printk(KERN_CRIT "\n");
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt, i1, i2, i3, i4, i5, (str) ? (str) : "null");	
	printf("\n");
    }
#endif
}

// default trace handle, area and level with 5 integer and 1 string argument
#define GTRACE_I5S(MID, EID, EFMT, I1, I2, I3, I4, I5, S)             \
	GTRACE_HAL_I5S(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, S)

// trace handle, default area and level with 5 integer and 1 string argument
#define GTRACE_H_I5S(MID, HNDL, EID, EFMT, I1, I2, I3, I4, I5, S)             \
	GTRACE_HAL_I5S(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, S)

// default trace handle, area and spdcific level with 5 integer and 1 string argument
#define GTRACE_L_I5S(MID, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, S)             \
	GTRACE_HAL_I5S(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, S)

// default trace handle, specific area and default level with 5 integer and 1 string argument
#define GTRACE_A_I5S(MID, AREA, EID, EFMT, I1, I2, I3, I4, I5, S)             \
	GTRACE_HAL_I5S(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, S)

// trace handl, default area and specific level with 5 integer and 1 string argument
#define GTRACE_HL_I5S(MID, HNDL, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, S)             \
	GTRACE_HAL_I5S(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, S)

// trace handl, specific area and default level with 5 integer and 1 string argument
#define GTRACE_HA_I5S(MID, HNDL, AREA, EID, EFMT, I1, I2, I3, I4, I5, S)             \
	GTRACE_HAL_I5S(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, S)

// trace handl, specific area and level with 5 integer and 1 string argument
#define GTRACE_AL_I5S(MID, AREA, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, S)             \
	GTRACE_HAL_I5S(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, S)

/* ***********************************************************************************
 *   GTRACE_HAL_I6S scession
 ************************************************************************************/

// specific trace handle, area and level with 6 integer and 1 string argument
static inline void
GTRACE_HAL_I6S(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
, int i1, int i2, int i3, int i4, int i5, int i6, char *str)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;

    data.td_word[0] = i1;
    data.td_word[1] = i2;
    data.td_word[2] = i3;
    data.td_word[3] = i4;
    data.td_word[4] = i5;
    data.td_word[5] = i6;

    /* string always at last arg */
    if (cfg->entry_size > 24)
        trace_cat2str(cfg->entry_size - 24, (char *)&data.td_word[6], (str) ? (str) : "null");

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
         printk(KERN_CRIT "%s,", e_ids); printk(e_fmt, i1, i2, i3, i4, i5, i6, (str) ? (str) : "null");
	printk(KERN_CRIT "\n");
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt, i1, i2, i3, i4, i5, i6, (str) ? (str) : "null");
	printf("\n");
    }
#endif
}

// default trace handle, area and level with 6 integer and 1 string argument
#define GTRACE_I6S(MID, EID, EFMT, I1, I2, I3, I4, I5, I6, S)             \
	GTRACE_HAL_I6S(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, S)

// trace handle, default area and level with 6 integer and 1 string argument
#define GTRACE_H_I6S(MID, HNDL, EID, EFMT, I1, I2, I3, I4, I5, I6, S)             \
	GTRACE_HAL_I6S(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, S)

// default trace handle, area and spdcific level with 6 integer and 1 string argument
#define GTRACE_L_I6S(MID, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6, S)             \
	GTRACE_HAL_I6S(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, S)

// default trace handle, specific area and default level with 6 integer and 1 string argument
#define GTRACE_A_I6S(MID, AREA, EID, EFMT, I1, I2, I3, I4, I5, I6, S)             \
	GTRACE_HAL_I6S(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, S)

// trace handl, default area and specific level with 6 integer and 1 string argument
#define GTRACE_HL_I6S(MID, HNDL, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6, S)             \
	GTRACE_HAL_I6S(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, S)

// trace handl, specific area and default level with 6 integer and 1 string argument
#define GTRACE_HA_I6S(MID, HNDL, AREA, EID, EFMT, I1, I2, I3, I4, I5, I6, S)             \
	GTRACE_HAL_I6S(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, S)

// trace handl, specific area and level with 6 integer and 1 string argument
#define GTRACE_AL_I6S(MID, AREA, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6, S)             \
	GTRACE_HAL_I6S(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, S)

/* ***********************************************************************************
 *   GTRACE_HAL_I7S scession
 ************************************************************************************/

// specific trace handle, area and level with 7 integer and 1 string argument
static inline void
GTRACE_HAL_I7S(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
, int i1, int i2, int i3, int i4, int i5, int i6, int i7, char *str)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;

    data.td_word[0] = i1;
    data.td_word[1] = i2;
    data.td_word[2] = i3;
    data.td_word[3] = i4;
    data.td_word[4] = i5;
    data.td_word[5] = i6;
    data.td_word[6] = i7;

    /* string always at last arg */
    if (cfg->entry_size > 28)
        trace_cat2str(cfg->entry_size - 28, (char *)&data.td_word[7], (str) ? (str) : "null");

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
         printk(KERN_CRIT "%s,", e_ids); printk(e_fmt, i1, i2, i3, i4, i5, i6, i7, (str) ? (str) : "null");
	printk(KERN_CRIT "\n");
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt, i1, i2, i3, i4, i5, i6, i7, (str) ? (str) : "null");
	printf("\n");
    }
#endif
}

// default trace handle, area and level with 7 integer and 1 string argument
#define GTRACE_I7S(MID, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, S)             \
	GTRACE_HAL_I7S(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, S)

// trace handle, default area and level with 7 integer and 1 string argument
#define GTRACE_H_I7S(MID, HNDL, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, S)             \
	GTRACE_HAL_I7S(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, S)

// default trace handle, area and spdcific level with 7 integer and 1 string argument
#define GTRACE_L_I7S(MID, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, S)             \
	GTRACE_HAL_I7S(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, S)

// default trace handle, specific area and default level with 7 integer and 1 string argument
#define GTRACE_A_I7S(MID, AREA, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, S)             \
	GTRACE_HAL_I7S(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, S)

// trace handl, default area and specific level with 7 integer and 1 string argument
#define GTRACE_HL_I7S(MID, HNDL, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, S)             \
	GTRACE_HAL_I7S(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, S)

// trace handl, specific area and default level with 7 integer and 1 string argument
#define GTRACE_HA_I7S(MID, HNDL, AREA, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, S)             \
	GTRACE_HAL_I7S(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, S)

// trace handl, specific area and level with 7 integer and 1 string argument
#define GTRACE_AL_I7S(MID, AREA, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, S)             \
	GTRACE_HAL_I7S(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, S)

/* ***********************************************************************************
 *   GTRACE_HAL_I8S scession
 ************************************************************************************/

// specific trace handle, area and level with 8 integer and 1 string argument
static inline void
GTRACE_HAL_I8S(int mid, int hndl, int area, int level, int e_id, char *e_ids, char *e_fmt 
, int i1, int i2, int i3, int i4, int i5, int i6, int i7, int i8, char *str)
{
    trace_data_t data;
    tbc_cfg_t *cfg = tbc_get_cfg(mid);

    if (level < TRACE_LVL2TRCE(cfg->area_level[area]))
	return;

    data.td_word[0] = i1;
    data.td_word[1] = i2;
    data.td_word[2] = i3;
    data.td_word[3] = i4;
    data.td_word[4] = i5;
    data.td_word[5] = i6;
    data.td_word[6] = i7;
    data.td_word[7] = i8;

    /* string always at last arg */
    if (cfg->entry_size > 32)
        trace_cat2str(cfg->entry_size - 32, (char *)&data.td_word[8], (str) ? (str) : "null");

#ifdef	__KERNEL__
   trace_store_lvl(mid, e_id, level, &data);

    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
         printk(KERN_CRIT "%s,", e_ids); printk(e_fmt, i1, i2, i3, i4, i5, i6, i7, i8, (str) ? (str) : "null");
	printk(KERN_CRIT "\n");
    }
#else
    trace_module_store_lvl(hndl, mid, e_id, level, &data);
    
    if (level >= TRACE_LVL2CONSOLE(cfg->area_level[area])) {
        printf("%s,", e_ids); printf(e_fmt, i1, i2, i3, i4, i5, i6, i7, i8, (str) ? (str) : "null");
	printf("\n");
    }
#endif
}


// default trace handle, area and level with 8 integer and 1 string argument
#define GTRACE_I8S(MID, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8, S)             \
	GTRACE_HAL_I8S(MID, DHNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8, S)

// trace handle, default area and level with 8 integer and 1 string argument
#define GTRACE_H_I8S(MID, HNDL, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8, S)             \
	GTRACE_HAL_I8S(MID, HNDL, TRACE_A_DFT, TRACE_DEFAULT_LVL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8, S)

// default trace handle, area and spdcific level with 8 integer and 1 string argument
#define GTRACE_L_I8S(MID, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8, S)             \
	GTRACE_HAL_I8S(MID, DHNDL, TRACE_A_DFT, LEVEL,	\
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8, S)

// default trace handle, specific area and default level with 8 integer and 1 string argument
#define GTRACE_A_I8S(MID, AREA, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8, S)             \
	GTRACE_HAL_I8S(MID, DHNDL, AREA, TRACE_DEFAULT_LVL, \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8, S)

// trace handl, default area and specific level with 8 integer and 1 string argument
#define GTRACE_HL_I8S(MID, HNDL, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8, S)             \
	GTRACE_HAL_I8S(MID, HNDL, TRACE_A_DFT, LEVEL,       \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8, S)

// trace handl, specific area and default level with 8 integer and 1 string argument
#define GTRACE_HA_I8S(MID, HNDL, AREA, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8, S)             \
	GTRACE_HAL_I8S(MID, HNDL, AREA, TRACE_DEFAULT_LVL,    \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8, S)

// trace handl, specific area and level with 8 integer and 1 string argument
#define GTRACE_AL_I8S(MID, AREA, LEVEL, EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8, S)             \
	GTRACE_HAL_I8S(MID, DHNDL, AREA, LEVEL,	         \
		 EID, #EID, EFMT, I1, I2, I3, I4, I5, I6, I7, I8, S)

#endif
#endif


#endif  /*  __RAS_TRACE_H__ */
