/*
 * Copyright (c) 2005 Brocade Communications Systems, Inc.
 * All rights reserved.
 *
 * Description:
 *     Internal Message Passing Interface (MPI) defines related to
 *		Shared Memory (SM) size.
 *
 *		Many of these definitions refer to internal MPI types which are
 *		exposed here for in support of the macro that returns the overall
 *		MPI shared memory requirements and should not be referenced by
 *		client software.
 */

#ifndef __MPI_MEM_SIZE_H__
#define __MPI_MEM_SIZE_H__

#include <mpi/mpi_priv.h>

#ifdef __cplusplus
extern "C" {
#endif

/*
 * The MPI_MEM_SZ() macro provides a means to statically determine the
 * MPI shared memory requirements based on the following parameters:
 *
 * 		num_cq		Number of CQ Rx/Tx pairs
 * 		cq_sz		Number of entries in each CQ
 * 		pg_sz		Page size used for message payload storage
 * 		num_pgs		Number of pages in free pool
 *
 * The resulting macro expansion gives the total shared memory (in bytes)
 * required to instantiate MPI with the specified CQ parameters.  The macro
 * equation is:
 *
 * MPI_MEM_SZ =
 *
 * 		num_cq * sizeof(consumer/producer index) * (number of ci/pi indices) +
 *
 *		sizeof(cq entry) * num_cq * cq_sz * 2 (for Rx & Tx) +
 *
 *		(number of internal mem blks) * sizeof(internal mem blk) +
 *
 *		(number of internal mem blks) * sizeof(internal mem blk mgmt struct) +
 *
 *		num_pgs * page_sz +
 *
 *		num_pgs * sizeof(internal page mgmt struct) +
 *
 *		sizeof(mpi statistics) +
 *
 *		MPI_CQ_ALIGN_SZ (to account for unaligned base shared mem addr)
 *
 * Note that each sub block of MPI shared memory is aligned to a double word
 * boundary to facilitate potential use of HW assist in data transfers.
 */

#define MPI_MEM_SZ(num_cq, cq_sz, pg_sz, num_pgs) ( \
	MPI_CQ_ALIGNED((num_cq) * MPI_CPI_SZ * MPI_CPI_NUM) + \
	MPI_CQ_ALIGNED(MPI_CQ_ENTRY_SZ * (num_cq) * (cq_sz) * MPI_RX_TX_PAIR) + \
	MPI_CQ_ALIGNED( \
		MPI_CQ_ALIGNED(MPI_MBLKS((num_cq), (cq_sz))) * MPI_MBLK_SZ) + \
	MPI_CQ_ALIGNED(MPI_CQ_ALIGNED( \
			MPI_MBLKS((num_cq), (cq_sz))) * MPI_MBLK_MGMT_ENTRY_SZ) + \
	MPI_CQ_ALIGNED(MPI_CQ_ALIGNED((num_pgs)) * MPI_CQ_ALIGNED((pg_sz))) + \
	MPI_CQ_ALIGNED(MPI_CQ_ALIGNED((num_pgs)) * MPI_PAGE_MGMT_ENTRY_SZ) + \
	MPI_CQ_ALIGNED(sizeof (mpi_stats_t)) +  \
	MPI_CQ_ALIGN_SZ)


#ifdef __cplusplus
}
#endif

#endif
