#!/bin/sh

#################################################################
#
# Copyright (c) 1996-2007 by Brocade Communications Systems, Inc.
# All rights reserved.
#
# Description:
#   Script to test correctness of the following  encryption /
#   decryption algorithms in the OB1 blocks of Brocade Encryption 
#   Switches / FS8-18 encryption blades.
#    1) XTS - AES
#
#################################################################

# Variables used during testing of XTS-AES algorithm
KEY_W0=0x27182818 

KEY_W1=0x28459045 

KEY_W2=0x23536028 

KEY_W3=0x74713526 

KEY_W4=0x62497757 

KEY_W5=0x24709369 

KEY_W6=0x99595749 

KEY_W7=0x66967627 

TWK_KEY_W0=0x31415926 

TWK_KEY_W1=0x53589793 

TWK_KEY_W2=0x23846264 

TWK_KEY_W3=0x33832795 

TWK_KEY_W4=0x02884197 

TWK_KEY_W5=0x16939937 

TWK_KEY_W6=0x51058209 

TWK_KEY_W7=0x74944592 

LBA_W0=0x00000000 

LBA_W1=0x000000ff 

DATA_LEN=0x00000008

PTX_DATA0_W0=0x00010203 

PTX_DATA0_W1=0x04050607 

PTX_DATA0_W2=0x08090a0b 

PTX_DATA0_W3=0x0c0d0e0f 

PTX_DATA1_W0=0x10111213 

PTX_DATA1_W1=0x14151617 

PTX_DATA1_W2=0x18191a1b 

PTX_DATA1_W3=0x1c1d1e1f 

CTX_DATA0_W0=0x1c3b3a10 

CTX_DATA0_W1=0x2f770386 

CTX_DATA0_W2=0xe4836c99 

CTX_DATA0_W3=0xe370cf9b 

CTX_DATA1_W0=0xea00803f 

CTX_DATA1_W1=0x5e482357 

CTX_DATA1_W2=0xa4ae12d4 

CTX_DATA1_W3=0x14a3e63b 

TWK0_W0=0x1a267577 

TWK0_W1=0xa90caad6 

TWK0_W2=0xae988e22 

TWK0_W3=0x714a2b8b 

TWK1_W0=0xb34ceaee 

TWK1_W1=0x521954ad 

TWK1_W2=0x5d311d45 

TWK1_W3=0xe2945616

NEW_OB1_REV_ID=0x12073001

#################################################################
# function rd_reg
#
# Description:
#   Read the current value in a given register in an OB1 block.
#
# Input:
#   $1 - Register address
#################################################################
function rd_reg
{
    echo $1 > /proc/vdrob1/rw/regval
    cat /proc/vdrob1/rw/regval | grep "vader\|ob1" |  cut -d '=' -f 2 | cut -d ' ' -f 2
}

#################################################################
# function wr_reg
#
# Description:
#   Writes a given value into a given register in the OB1
#   block.
#
# Input:
#   $1 - Register address
#   $2 - Value to be written
#################################################################
function wr_reg
{
    echo $1 $2 > /proc/vdrob1/rw/regval
}

#################################################################
# function cmp_eq
#
# Description:
#   Compares two registers in an OB1 block. As most of the tests
#   expect an equality during the comparison, if inequality is
#   detected, this function exits from the script with a message.
#################################################################
function cmp_eq
{
    if [ $1 != $2 ]
        then
        echo -en "\n" $1 "!=" $2
        echo -en "\nTest failed as an expected equality was not satisfied. Script exiting...\n"
        exit 1;
    fi
}
#################################################################
# function reg_address
#
# Description:
#   Calculate the adress of registers in agiven OB1 Block.
#
# Input:
#   $1 - OB1 index
#
# Output:
#   None
#################################################################

function reg_address
{
# calculate ob1 register addresses from OB1 id input

    EDB_TWF_HIGH=`printf "0x%08x" $((0x00838130 + $(($1 * 0x00080000))))`

    EDB_TWF_LOW=`printf "0x%08x" $((0x0083812c + $(($1 * 0x00080000))))` 

    EDB_MISC_TEST_DATA=`printf "0x%08x" $((0x0083810c + $(($1 * 0x00080000))))`

    EDB_TEST_DATA_HIGH=`printf "0x%08x" $((0x00838108 + $(($1 * 0x00080000))))`

    EDB_TEST_DATA_LOW=`printf "0x%08x" $((0x00838104 + $(($1 * 0x00080000))))`

    EDB_IDF_CONTROL=`printf "0x%08x" $((0x00838144 + $(($1 * 0x00080000))))`

    EDB_TEST_CONTROL_1=`printf "0x%08x" $((0x00838100 + $(($1 * 0x00080000))))`

    EDB_TEST_CONTROL_2=`printf "0x%08x" $((0x00838110 + $(($1 * 0x00080000))))`

    EDB_IDF_TAG=`printf "0x%08x" $((0x00838148 + $(($1 * 0x00080000))))`

    EDB_PCF_W1_OUTPUT_HIGH=`printf "0x%08x" $((0x0083811c + $(($1 * 0x00080000))))`

    EDB_PCF_W1_OUTPUT_LOW=`printf "0x%08x" $((0x00838118 + $(($1 * 0x00080000))))`

    EDB_PCF_W2_OUTPUT_HIGH=`printf "0x%08x" $((0x00838124 + $(($1 * 0x00080000))))`

    EDB_PCF_W2_OUTPUT_LOW=`printf "0x%08x" $((0x00838120 + $(($1 * 0x00080000))))`

    OB1_DEBUG_CONTROL=`printf "0x%08x" $((0x00800100 + $(($1 * 0x00080000))))`
    
    OB1_FPGA_REV_ID=`printf "0x%08x" $((0x00800104 + $(($1 * 0x00080000))))`
    
    EDB_INTERRUPT_ENABLE_REGISTER=`printf "0x%08x" $((0x00838004 + $(($1 * 0x00080000))))`

    EDB_AUT_HIGH=`printf "0x%08x" $((0x0083813C + $(($1 * 0x00080000))))`

    EDB_AUT_LOW=`printf "0x%08x" $((0x00838138 + $(($1 * 0x00080000))))`

    EDB_AUTH_RST=`printf "0x%08x" $((0x0083814C + $(($1 * 0x00080000))))`
   
}
#################################################################
# function test_xts_aes
#
# Description:
#   Test the XTS-AES algorithm on a given OB1 block.
#
# Input:
#   $1 - OB1 index
#
# Output:
#   None
#################################################################
function test_xts_aes
{
    reg_address $1

    REVISION=`rd_reg $OB1_FPGA_REV_ID`

    wr_reg  $EDB_INTERRUPT_ENABLE_REGISTER  0x7ffff 
   
    wr_reg  $EDB_TEST_CONTROL_1 0x1
    if [ $REVISION == $NEW_OB1_REV_ID ]
    then
     wr_reg  $OB1_DEBUG_CONTROL 0x601b0256
    fi

#############################################################
#
# GENERATE SUBKEYS FOR ENCRYPTION
#
#############################################################

# the first 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00008000 # encrypt the key index = 0x00

    wr_reg $EDB_TEST_DATA_HIGH $KEY_W0 # the high part of the First 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $KEY_W1 # the low part of the First 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000020 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

# the second 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00008100 # encrypt the key index = 0x00 the second 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $KEY_W2 # the high part of the second 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $KEY_W3 # the low part of the second 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000020 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

# the third 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00008200 # encrypt the key index = 0x00 the third 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $KEY_W4 # the high part of the third 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $KEY_W5 # the low part of the third 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000020 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

# the fourth 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00008300 # encrypt the key index = 0x00 the fourth 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $KEY_W6 # the high part of the fourth 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $KEY_W7 # the low part of the fourth 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000020 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

# the placeholder of the key crc10

    wr_reg $EDB_MISC_TEST_DATA 0x00008400 # encrypt the key index = 0x00 the fourth 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH 0x00000000 # the high part of the fourth 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  0x00000000 # the low part of the fourth 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000020 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

# Start the key generation

    wr_reg $EDB_TEST_CONTROL_2 0x00000001 # write 1 TO the test_edb_keygen_start Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_keygen_start Field of the $EDB_TEST_CONTROL_2 reg


#############################################################
#
# GENERATE TWEAKS FOR VERIFICATION
#
#############################################################

# testing the tweak generator

# the 64-bit CMD1 field

    wr_reg $EDB_MISC_TEST_DATA 0x00000000 # encrypt the key index = 0x00 the first 64 bits of CMD1

    wr_reg $EDB_TEST_DATA_HIGH 0x80000000 # the high part of the First 64 bits of CMD1

    wr_reg $EDB_TEST_DATA_LOW  $DATA_LEN # the low part of the First 64 bits of CMD1

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# the first 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00010000 # encrypt the key index = 0x00 the second 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $TWK_KEY_W0 # the high part of the first 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $TWK_KEY_W1 # the low part of the first 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# the second 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00020000 # encrypt the key index = 0x00 the second 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $TWK_KEY_W2 # the high part of the second 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $TWK_KEY_W3 # the low part of the second 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# the third 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00030000 # encrypt the key index = 0x00 the third 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $TWK_KEY_W4 # the high part of the third 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $TWK_KEY_W5 # the low part of the third 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# the fourth 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00040000 # encrypt the key index = 0x00 the fourth 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $TWK_KEY_W6 # the high part of the fourth 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $TWK_KEY_W7 # the low part of the fourth 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# the LBA field

    wr_reg $EDB_MISC_TEST_DATA 0x00050000 # encrypt the key index = 0x00 the fourth 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $LBA_W0 # the high part of the LBA

    wr_reg $EDB_TEST_DATA_LOW  $LBA_W1 # the low part of the LBA

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# Start the tweak generation

    wr_reg $EDB_TEST_CONTROL_2 0x00000002 # write 1 TO the test_edb_tweak_start Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_start Field of the $EDB_TEST_CONTROL_2 reg

#delay_cycles400

# verifying the tweak results

# Read Tweak0 (first 64-bits)

    VALUE_EDB_TWF_W1_HIGH=`rd_reg $EDB_TWF_HIGH`

    VALUE_EDB_TWF_W1_LOW=`rd_reg $EDB_TWF_LOW`

    cmp_eq $VALUE_EDB_TWF_W1_HIGH $TWK0_W0 

    cmp_eq $VALUE_EDB_TWF_W1_LOW $TWK0_W1 

    wr_reg $EDB_TEST_CONTROL_2 0x00000004 # write 4 TO pop the tweak fifo $EDB_TEST_CONTROL_2 reg

# Read Tweak1 (second 64-bits)

    VALUE_EDB_TWF_W2_HIGH=`rd_reg $EDB_TWF_HIGH`

    VALUE_EDB_TWF_W2_LOW=`rd_reg $EDB_TWF_LOW`

    cmp_eq $VALUE_EDB_TWF_W2_HIGH $TWK0_W2 

    cmp_eq $VALUE_EDB_TWF_W2_LOW $TWK0_W3 

    wr_reg $EDB_TEST_CONTROL_2 0x00000004 # write 4 TO pop the tweak fifo $EDB_TEST_CONTROL_2 reg

# Read Tweak2 (third 64-bits)

    VALUE_EDB_TWF_HIGH=`rd_reg $EDB_TWF_HIGH`

    VALUE_EDB_TWF_LOW=`rd_reg $EDB_TWF_LOW`

    cmp_eq $VALUE_EDB_TWF_HIGH $TWK1_W0 

    cmp_eq $VALUE_EDB_TWF_LOW $TWK1_W1 

    wr_reg $EDB_TEST_CONTROL_2 0x00000004 # write 4 TO pop the tweak fifo $EDB_TEST_CONTROL_2 reg

# Read Tweak3 (fourth 64-bits)

    VALUE_EDB_TWF_HIGH=`rd_reg $EDB_TWF_HIGH`

    VALUE_EDB_TWF_LOW=`rd_reg $EDB_TWF_LOW`

    cmp_eq $VALUE_EDB_TWF_HIGH $TWK1_W2 

    cmp_eq $VALUE_EDB_TWF_LOW $TWK1_W3 

    wr_reg $EDB_TEST_CONTROL_2 0x00000004 # write 4 TO pop the tweak fifo $EDB_TEST_CONTROL_2 reg


#############################################################
#
# GENERATE TWEAKS FOR ENCRYPTION
#
#############################################################

# Since the above pops all tweaks from the fifo we need to generate Tweaks again to verify datapath encryption

#    echo -en "\nEncryption Test Starts"

# tweak generator

# the 64-bit CMD1 field

    wr_reg $EDB_MISC_TEST_DATA 0x00000000 # encrypt the key index = 0x00 the first 64 bits of CMD1

    wr_reg $EDB_TEST_DATA_HIGH 0x80000000 # the high part of the First 64 bits of CMD1

    wr_reg $EDB_TEST_DATA_LOW  $DATA_LEN # the low part of the First 64 bits of CMD1

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# the first 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00010000 # encrypt the key index = 0x00 the second 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $TWK_KEY_W0 # the high part of the first 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $TWK_KEY_W1 # the low part of the first 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# the second 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00020000 # encrypt the key index = 0x00 the second 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $TWK_KEY_W2 # the high part of the second 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $TWK_KEY_W3 # the low part of the second 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# the third 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00030000 # encrypt the key index = 0x00 the third 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $TWK_KEY_W4 # the high part of the third 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $TWK_KEY_W5 # the low part of the third 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# the fourth 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00040000 # encrypt the key index = 0x00 the fourth 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $TWK_KEY_W6 # the high part of the fourth 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $TWK_KEY_W7 # the low part of the fourth 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# the LBA field

    wr_reg $EDB_MISC_TEST_DATA 0x00050000 # encrypt the key index = 0x00 the fourth 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $LBA_W0 # the high part of the LBA

    wr_reg $EDB_TEST_DATA_LOW  $LBA_W1 # the low part of the LBA

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# Start the tweak generation

    wr_reg $EDB_TEST_CONTROL_2 0x00000002 # write 1 TO the test_edb_tweak_start Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_start Field of the $EDB_TEST_CONTROL_2 reg


#delay_cycles400

#############################################################
#
# INPUT DATA FOR ENCRYPTION
#
#############################################################

#testing the AES datapath encryption (first 64 bits)

# writing the input data words TO the test data registers First 64 bits

    wr_reg $EDB_TEST_DATA_HIGH $PTX_DATA0_W0 # the high part of the 64 bits of the input data

    wr_reg $EDB_TEST_DATA_LOW $PTX_DATA0_W1 # the low part of the 64 bits of the input data

    wr_reg $EDB_IDF_CONTROL 0x80000001 # test_idf_enc_msw=1 test_idf_key_index=0x0 test_idf_cmd=1

    wr_reg $EDB_IDF_TAG 0x6  #12.10.2.3.d ends

    wr_reg $EDB_TEST_CONTROL_2 0x00000010 # write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2 12.10.2.3.e

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 0 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2 12.10.2.3.e


###################################################


#testing the AES datapath encryption (second 64 bits)

# writing the input data words TO the test data registers second 64 bits

    wr_reg $EDB_TEST_DATA_HIGH $PTX_DATA0_W2 # the high part of the 64 bits of the input data

    wr_reg $EDB_TEST_DATA_LOW $PTX_DATA0_W3 # the low part of the 64 bits of the input data

    wr_reg $EDB_IDF_CONTROL 0x00000001 # test_idf_enc_msw=1 test_idf_key_index=0x20 test_idf_cmd=1

    wr_reg $EDB_IDF_TAG 0x6  #12.10.2.3.d ends

    wr_reg $EDB_TEST_CONTROL_2 0x00000008 # write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2 12.10.2.3.g

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 0 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2 12.10.2.3.g

#delay_cycles10

# verifying encrypted data0 (first and second 64-bits)

#delay_cycles10

    VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg $EDB_PCF_W1_OUTPUT_HIGH`

    VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg $EDB_PCF_W1_OUTPUT_LOW`

    VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg $EDB_PCF_W2_OUTPUT_HIGH`

    VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg $EDB_PCF_W2_OUTPUT_LOW`

    VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg $EDB_PCF_W2_OUTPUT_LOW`

    cmp_eq $VALUE_EDB_PCF_W1_OUTPUT_HIGH $CTX_DATA0_W0 

    cmp_eq $VALUE_EDB_PCF_W1_OUTPUT_LOW $CTX_DATA0_W1 

    cmp_eq $VALUE_EDB_PCF_W2_OUTPUT_HIGH $CTX_DATA0_W2 

    cmp_eq $VALUE_EDB_PCF_W2_OUTPUT_LOW $CTX_DATA0_W3 


###################################################

#testing the AES datapath encryption (third 64 bits)

# writing the input data words TO the test data registers Third 64 bits

    wr_reg $EDB_TEST_DATA_HIGH $PTX_DATA1_W0 # the high part of the 64 bits of the input data

    wr_reg $EDB_TEST_DATA_LOW $PTX_DATA1_W1 # the low part of the 64 bits of the input data

    wr_reg $EDB_IDF_CONTROL 0x80000001 # test_idf_enc_msw=1 test_idf_key_index=0x0 test_idf_cmd=1

    wr_reg $EDB_IDF_TAG 0x6  #12.10.2.3.d ends

    wr_reg $EDB_TEST_CONTROL_2 0x00000010 # write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2 12.10.2.3.e

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 0 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2 12.10.2.3.e

#testing the AES datapath encryption (fourth 64 bits)

# writing the input data words TO the test data registers Fourth 64 bits

    wr_reg $EDB_TEST_DATA_HIGH $PTX_DATA1_W2 # the high part of the 64 bits of the input data

    wr_reg $EDB_TEST_DATA_LOW $PTX_DATA1_W3 # the low part of the 64 bits of the input data

    wr_reg $EDB_IDF_CONTROL 0x00000001 # test_idf_enc_msw=1 test_idf_key_index=0x0 test_idf_cmd=1

    wr_reg $EDB_IDF_TAG 0x6  #12.10.2.3.d ends

    wr_reg $EDB_TEST_CONTROL_2 0x00000008 # write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2 12.10.2.3.e

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 0 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2 12.10.2.3.e

# verifying encrypted data1 (third and fourth 64-bits)

#delay_cycles10

    VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg $EDB_PCF_W1_OUTPUT_HIGH`

    VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg $EDB_PCF_W1_OUTPUT_LOW`

    VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg $EDB_PCF_W2_OUTPUT_HIGH`

    VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg $EDB_PCF_W2_OUTPUT_LOW`

    VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg $EDB_PCF_W2_OUTPUT_LOW`

    cmp_eq $VALUE_EDB_PCF_W1_OUTPUT_HIGH $CTX_DATA1_W0 

    cmp_eq $VALUE_EDB_PCF_W1_OUTPUT_LOW $CTX_DATA1_W1 

    cmp_eq $VALUE_EDB_PCF_W2_OUTPUT_HIGH $CTX_DATA1_W2 

    cmp_eq $VALUE_EDB_PCF_W2_OUTPUT_LOW $CTX_DATA1_W3 


#    echo -en "\nEncryption Test Ends"


####################################################
#
#
# DECRYPTION
#
#    echo -en "\nDecryption Test  Starts"
####################################################



#############################################################
#
# GENERATE SUBKEYS FOR DECRYPTION
#
#############################################################

# the first 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00000000 # decrypt the key index = 0x00 the First 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $KEY_W0 # the high part of the First 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $KEY_W1 # the low part of the First 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000020 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

# the second 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00000100 # decrypt the key index = 0x00 the second 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $KEY_W2 # the high part of the second 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $KEY_W3 # the low part of the second 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000020 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

# the third 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00000200 # decrypt the key index = 0x00 the third 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $KEY_W4 # the high part of the third 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $KEY_W5 # the low part of the third 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000020 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

# the fourth 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00000300 # decrypt the key index = 0x00 the fourth 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $KEY_W6 # the high part of the fourth 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $KEY_W7 # the low part of the fourth 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000020 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

# the placeholder of the key crc10

    wr_reg $EDB_MISC_TEST_DATA 0x00000400 # decrypt the key index = 0x00 the fourth 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH 0x00000000 # the high part of the fourth 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  0x00000000 # the low part of the fourth 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000020 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

# Start the key generation

    wr_reg $EDB_TEST_CONTROL_2 0x00000001 # write 1 TO the test_edb_keygen_start Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_keygen_start Field of the $EDB_TEST_CONTROL_2 reg


#############################################################
#
# GENERATE TWEAKS FOR DECRYPTION
#
#############################################################

# generate Tweaks to verify datapath decryption

# tweak generator

# the 64-bit CMD1 field

    wr_reg $EDB_MISC_TEST_DATA 0x00000000 # decrypt the key index = 0x00 the first 64 bits of CMD1

    wr_reg $EDB_TEST_DATA_HIGH 0x80000000 # the high part of the First 64 bits of CMD1

    wr_reg $EDB_TEST_DATA_LOW  $DATA_LEN # the low part of the First 64 bits of CMD1

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# the first 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00010000 # decrypt the key index = 0x00 the second 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $TWK_KEY_W0 # the high part of the first 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $TWK_KEY_W1 # the low part of the first 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# the second 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00020000 # decrypt the key index = 0x00 the second 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $TWK_KEY_W2 # the high part of the second 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $TWK_KEY_W3 # the low part of the second 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# the third 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00030000 # decrypt the key index = 0x00 the third 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $TWK_KEY_W4 # the high part of the third 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $TWK_KEY_W5 # the low part of the third 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# the fourth 64 bits of the key

    wr_reg $EDB_MISC_TEST_DATA 0x00040000 # decrypt the key index = 0x00 the fourth 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $TWK_KEY_W6 # the high part of the fourth 64 bits of the key

    wr_reg $EDB_TEST_DATA_LOW  $TWK_KEY_W7 # the low part of the fourth 64 bits of the key

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# the LBA field

    wr_reg $EDB_MISC_TEST_DATA 0x00050000 # decrypt the key index = 0x00 the fourth 64 bits of the key

    wr_reg $EDB_TEST_DATA_HIGH $LBA_W0 # the high part of the LBA

    wr_reg $EDB_TEST_DATA_LOW  $LBA_W1 # the low part of the LBA

    wr_reg $EDB_TEST_CONTROL_2 0x00000040 # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

# Start the tweak generation

    wr_reg $EDB_TEST_CONTROL_2 0x00000002 # write 1 TO the test_edb_tweak_start Field of the $EDB_TEST_CONTROL_2 reg

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_tweak_start Field of the $EDB_TEST_CONTROL_2 reg

#delay_cycles400

#############################################################
#
# INPUT DATA FOR DECRYPTION
#
#############################################################


#testing the AES datapath decryption (First 64 bits)

# writing the input data words TO the test data registers First 64 bits

    wr_reg $EDB_TEST_DATA_HIGH $CTX_DATA0_W0 # the high part of the 64 bits of the input data

    wr_reg $EDB_TEST_DATA_LOW $CTX_DATA0_W1 # the low part of the 64 bits of the input data

    wr_reg $EDB_IDF_CONTROL 0x80000001 # test_idf_enc_msw=1 test_idf_key_index=0x0 test_idf_cmd=1

    wr_reg $EDB_IDF_TAG 0x6  #12.10.2.3.d ends

    wr_reg $EDB_TEST_CONTROL_2 0x00000010 # write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2 12.10.2.3.e

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 0 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2 12.10.2.3.e


###################################################

#testing the AES datapath decryption (second 64 bits)

# writing the input data words TO the test data registers second 64 bits

    wr_reg $EDB_TEST_DATA_HIGH $CTX_DATA0_W2 # the high part of the 64 bits of the input data

    wr_reg $EDB_TEST_DATA_LOW $CTX_DATA0_W3 # the low part of the 64 bits of the input data

    wr_reg $EDB_IDF_CONTROL 0x00000001 # test_idf_enc_msw=1 test_idf_key_index=0x20 test_idf_cmd=1

    wr_reg $EDB_IDF_TAG 0x6  #12.10.2.3.d ends

    wr_reg $EDB_TEST_CONTROL_2 0x00000008 # write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2 12.10.2.3.g

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 0 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2 12.10.2.3.g


#delay_cycles10

# verifying decrypted data0 (first and second 64-bits)

#delay_cycles10

    VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg $EDB_PCF_W1_OUTPUT_HIGH`

    VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg $EDB_PCF_W1_OUTPUT_LOW`

    VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg $EDB_PCF_W2_OUTPUT_HIGH`

    VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg $EDB_PCF_W2_OUTPUT_LOW`

    VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg $EDB_PCF_W2_OUTPUT_LOW`

    cmp_eq $VALUE_EDB_PCF_W1_OUTPUT_HIGH $PTX_DATA0_W0 

    cmp_eq $VALUE_EDB_PCF_W1_OUTPUT_LOW $PTX_DATA0_W1 

    cmp_eq $VALUE_EDB_PCF_W2_OUTPUT_HIGH $PTX_DATA0_W2 

    cmp_eq $VALUE_EDB_PCF_W2_OUTPUT_LOW $PTX_DATA0_W3 

###################################################

#testing the AES datapath decryption (third 64 bits)

# writing the input data words TO the test data registers Third 64 bits

    wr_reg $EDB_TEST_DATA_HIGH $CTX_DATA1_W0 # the high part of the 64 bits of the input data

    wr_reg $EDB_TEST_DATA_LOW $CTX_DATA1_W1 # the low part of the 64 bits of the input data

    wr_reg $EDB_IDF_CONTROL 0x80000001 # test_idf_enc_msw=1 test_idf_key_index=0x0 test_idf_cmd=1

    wr_reg $EDB_IDF_TAG 0x6  #12.10.2.3.d ends

    wr_reg $EDB_TEST_CONTROL_2 0x00000010 # write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2 12.10.2.3.e

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 0 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2 12.10.2.3.e

#testing the AES datapath decryption (fourth 64 bits)

# writing the input data words TO the test data registers Fourth 64 bits

    wr_reg $EDB_TEST_DATA_HIGH $CTX_DATA1_W2 # the high part of the 64 bits of the input data

    wr_reg $EDB_TEST_DATA_LOW $CTX_DATA1_W3 # the low part of the 64 bits of the input data

    wr_reg $EDB_IDF_CONTROL 0x00000001 # test_idf_enc_msw=1 test_idf_key_index=0x0 test_idf_cmd=1

    wr_reg $EDB_IDF_TAG 0x6  #12.10.2.3.d ends

    wr_reg $EDB_TEST_CONTROL_2 0x00000008 # write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2 12.10.2.3.e

    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 0 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2 12.10.2.3.e

# verifying decrypted data1 (third and fourth 64-bits)

#delay_cycles10

    VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg $EDB_PCF_W1_OUTPUT_HIGH`

    VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg $EDB_PCF_W1_OUTPUT_LOW`

    VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg $EDB_PCF_W2_OUTPUT_HIGH`

    VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg $EDB_PCF_W2_OUTPUT_LOW`

    VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg $EDB_PCF_W2_OUTPUT_LOW`

    cmp_eq $VALUE_EDB_PCF_W1_OUTPUT_HIGH $PTX_DATA1_W0 

    cmp_eq $VALUE_EDB_PCF_W1_OUTPUT_LOW $PTX_DATA1_W1 

    cmp_eq $VALUE_EDB_PCF_W2_OUTPUT_HIGH $PTX_DATA1_W2 

    cmp_eq $VALUE_EDB_PCF_W2_OUTPUT_LOW $PTX_DATA1_W3 

    wr_reg  $EDB_TEST_CONTROL_1 0x0
    if [ $REVISION == $NEW_OB1_REV_ID ]
    then
    wr_reg  $OB1_DEBUG_CONTROL 0x0
    fi
#delay_cycles4
} #end of function test_xts_aes

function test_gcm_aes
{
###################################
# cpuTesting EDB GCM testcase  
# Tiger Guo
###################################

# IEEE P1619.1/D20  June 2007
# Copyright  2007 IEEE. All rights reserved.
# This is an unapproved IEEE Standards Draft  subject to change.
# D.3.7 GCM test vector 7
# KEY 404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f
# IV 101112131415161718191a1b
# HDR 202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f
# PTX 000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f
# PTX 202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f
# PTX 404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f
# PTX 606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f
# PTX 808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9f
# PTX a0a1a2a3a4a5a6a7a8a9aaabacadaeafb0b1b2b3b4b5b6b7b8b9babbbcbdbebf
# PTX c0c1c2c3c4c5c6c7c8c9cacbcccdcecfd0d1d2d3d4d5d6d7d8d9dadbdcdddedf
# PTX e0e1e2e3e4e5e6e7e8e9eaebecedeeeff0f1f2f3f4f5f6f7f8f9fafbfcfdfeff
# CTX 793b3fd252941224a6afdc5be7f501b9150696da12045c1c6077d3cac774accf
# CTX c3d530d848d665d81a49cbb500b88bbb624ae61d1667229c302dc6ff0bb4d70b
# CTX dbbc8566d6f5b158da99a2ff2e01dda629b89c34ad1e5feba70e7aae4328289c
# CTX 3629b0588350581ca8b97ccf1258fa3bbe2c5026047ba72648969cff8ba10ae3
# CTX 0e05935df0c693741892b76faf67133abd2cf2031121bd8bb38127a4d2eedeea
# CTX 13276494f402cd7c107fb3ec3b24784834338e55436287092ac4a26f5ea7ea4a
# CTX d68d73151639b05b24e68b9816d1398376d8e4138594758db9ad3b409259b26d
# CTX cfc06e722be987b3767f70a7b856b774b1ba2685b368091429fccb8dcdde09e4
# TAG 87ec837abf532855b2cea169d6943fcd
    
    reg_address $1

	wr_reg  $EDB_INTERRUPT_ENABLE_REGISTER  0x7ffff 

    # Enable CPU testing
	wr_reg  $EDB_TEST_CONTROL_1  0x1 
    if [ $REVISION == $NEW_OB1_REV_ID ]
    then
	wr_reg  $OB1_DEBUG_CONTROL  0x601b0256 
    fi

	wr_reg $EDB_AUTH_RST 0x0000000F #reset all the engines
    wr_reg $EDB_AUTH_RST 0x00000000 #Out of reset all the engines

    # generating subkeys @ key index 0x40, 
    # the key IS 0x404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f
    # the first 64 bits of the key
	wr_reg $EDB_MISC_TEST_DATA 0x00008040 # encrypt, the key index = 0x40, the First 64 bits of the key
	wr_reg $EDB_TEST_DATA_HIGH 0x40414243 # the high part of the First 64 bits of the key
	wr_reg $EDB_TEST_DATA_LOW  0x44454647 # the low part of the First 64 bits of the key
	wr_reg $EDB_TEST_CONTROL_2 0x00000020 # write 1 TO the test_edb_keygen_wren Field of the EDB_TEST_CONTROL_2 reg
	wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 1 TO the test_edb_keygen_wren Field of the EDB_TEST_CONTROL_2 reg
	
    # the second 64 bits of the key
	wr_reg $EDB_MISC_TEST_DATA 0x00008140
	wr_reg $EDB_TEST_DATA_HIGH 0x48494a4b
	wr_reg $EDB_TEST_DATA_LOW  0x4c4d4e4f 
	wr_reg $EDB_TEST_CONTROL_2 0x00000020 
	wr_reg $EDB_TEST_CONTROL_2 0x00000000 

	# the third 64 bits of the key
	wr_reg $EDB_MISC_TEST_DATA 0x00008240
	wr_reg $EDB_TEST_DATA_HIGH 0x50515253
	wr_reg $EDB_TEST_DATA_LOW  0x54555657
	wr_reg $EDB_TEST_CONTROL_2 0x00000020
	wr_reg $EDB_TEST_CONTROL_2 0x00000000

	# the fourth 64 bits of the key
	wr_reg $EDB_MISC_TEST_DATA 0x00008340
	wr_reg $EDB_TEST_DATA_HIGH 0x58595a5b
	wr_reg $EDB_TEST_DATA_LOW  0x5c5d5e5f
	wr_reg $EDB_TEST_CONTROL_2 0x00000020
	wr_reg $EDB_TEST_CONTROL_2 0x00000000
	# the placeholder of the key crc10
	wr_reg $EDB_MISC_TEST_DATA 0x00008440
	wr_reg $EDB_TEST_DATA_HIGH 0x00000000
	wr_reg $EDB_TEST_DATA_LOW  0x00000000
	wr_reg $EDB_TEST_CONTROL_2 0x00000020
	wr_reg $EDB_TEST_CONTROL_2 0x00000000
  
    # Start the key generation
    wr_reg $EDB_TEST_CONTROL_2 0x00000001 # write 1 TO the test_edb_keygen_start Field of the EDB_TEST_CONTROL_2 reg
    wr_reg $EDB_TEST_CONTROL_2 0x00000000 # write 0 TO the test_edb_keygen_start Field of the EDB_TEST_CONTROL_2 reg
    # the tweak generation
    # the 64-bit CMD1 field
	wr_reg  $EDB_MISC_TEST_DATA  0x00000040  # encrypt  the key index = 0x40  the first 64 bits of CMD1
	wr_reg  $EDB_TEST_DATA_HIGH  0x20000010  # the high part of the First 64 bits of CMD1
	wr_reg  $EDB_TEST_DATA_LOW   0x00000040  # the low part of the First 64 bits of CMD1  w:0  is LENGTH
	wr_reg  $EDB_TEST_CONTROL_2  0x00000040  # write 1 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  # write 0 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

    # the first 64 bits of the IV
	wr_reg  $EDB_MISC_TEST_DATA  0x00010000  # the second 64 bits of the IV
	wr_reg  $EDB_TEST_DATA_HIGH  0x10111213  # the high part of the first 64 bits of the IV
	wr_reg  $EDB_TEST_DATA_LOW   0x14151617  # the low part of the first 64 bits of the IV
	wr_reg  $EDB_TEST_CONTROL_2  0x00000040  # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  # write 0 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

    # the second 64 bits of the IV
	wr_reg  $EDB_MISC_TEST_DATA  0x00020000  # the second 64 bits of the IV
	wr_reg  $EDB_TEST_DATA_HIGH  0x18191a1b  # the high part of the second 64 bits of the IV
	wr_reg  $EDB_TEST_DATA_LOW   0x0         # zeros
	wr_reg  $EDB_TEST_CONTROL_2  0x00000040  # write 1 TO the test_edb_keygen_wren Field of the $EDB_TEST_CONTROL_2 reg
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  # write 0 TO the test_edb_tweak_wren Field of the $EDB_TEST_CONTROL_2 reg

    # Start the tweak generation
	wr_reg  $EDB_TEST_CONTROL_2  0x00000002  # write 1 TO the test_edb_tweak_start Field of the $EDB_TEST_CONTROL_2 reg
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  # write 0 TO the test_edb_tweak_start Field of the $EDB_TEST_CONTROL_2 reg
   
    #12.10.2.4 table
    # cmd bits  w:0   opcode bits[6:4   
    # row 0  
	wr_reg  $EDB_IDF_CONTROL  0x80000040     #CMD bits 3b'000  opcode 0x4  msw = 1b'0 (w1)
	wr_reg  $EDB_IDF_TAG  0x0                #test_idf_tag = 0
	wr_reg  $EDB_TEST_DATA_HIGH  0x00000000  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x00000000  # session id = 0x1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
    # row 1  
	wr_reg  $EDB_IDF_CONTROL  0x00000000     #CMD bits 3b'000  opcode 0x0  msw = 1b'0 (w2)
	wr_reg  $EDB_TEST_DATA_HIGH  0x00000000  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x00000000  # 
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
    # row 2  
	wr_reg  $EDB_IDF_CONTROL  0x80400044     #CMD bits 3b'100  opcode 0x4  msw = 1b'0 (w1)
	wr_reg  $EDB_TEST_DATA_HIGH  0x00000000  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x00000000  # 
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
    # row 3  
	wr_reg  $EDB_IDF_CONTROL  0x00400044     #CMD bits 3b'100  opcode 0x4  msw = 1b'0 (w2)
	wr_reg  $EDB_TEST_DATA_HIGH  0x00000000  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x00000000  # 
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
    # row 4  
	wr_reg  $EDB_IDF_CONTROL  0x80400044     #CMD bits 3b'100  opcode 0x4  msw = 1b'0 (w1)
	wr_reg  $EDB_TEST_DATA_HIGH  0x10111213  # iv w5:32  = 0
	wr_reg  $EDB_TEST_DATA_LOW   0x14151617  # 
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
    # row 5  
	wr_reg  $EDB_IDF_CONTROL  0x00400044     #CMD bits 3b'100  opcode 0x4  msw = 1b'0 (w2)
	wr_reg  $EDB_TEST_DATA_HIGH  0x18191a1b  # iv[31:0  = 0
	wr_reg  $EDB_TEST_DATA_LOW   0x00000001  # 0x1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
    # row 6  
	wr_reg  $EDB_IDF_CONTROL  0x80000040     #CMD bits 3b'000  opcode 0x4  msw = 1b'0 (w1)
	wr_reg  $EDB_TEST_DATA_HIGH  0x00000000  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x00000100  # LenA IN bits (Length of AAD)
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
    # row 7  
	wr_reg  $EDB_IDF_CONTROL  0x00000040     #CMD bits 3b'000  opcode 0x4  msw = 1b'0 (w1)
	wr_reg  $EDB_TEST_DATA_HIGH  0x00000000  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x00000800  # LenC IN bits (Length of data)
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 8  
	wr_reg  $EDB_IDF_CONTROL  0x80000010     #CMD bits 3b'000  opcode 0x1  msw = 1b'0 (w1)
	wr_reg  $EDB_TEST_DATA_HIGH  0x20212223  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x24252627  # AAD0
	wr_reg  $EDB_IDF_TAG  0x6                #test_idf_tag = 6
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 9  
	wr_reg  $EDB_IDF_CONTROL  0x00000010     #CMD bits 3b'000  opcode 0x4  msw = 1b'0 (w1)
	wr_reg  $EDB_TEST_DATA_HIGH  0x28292a2b  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x2c2d2e2f  # AAD1
	wr_reg  $EDB_IDF_TAG  0x6                #test_idf_tag = 6
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

	#row 10  
	wr_reg  $EDB_IDF_CONTROL  0x80000010     #CMD bits 3b'000  opcode 0x1  msw = 1b'0 (w1)
	wr_reg  $EDB_TEST_DATA_HIGH  0x30313233  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x34353637  # AAD2
	wr_reg  $EDB_IDF_TAG  0x6                #test_idf_tag = 6
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 11  
	wr_reg  $EDB_IDF_CONTROL  0x00000010     #CMD bits 3b'000  opcode 0x4  msw = 1b'0 (w1)
	wr_reg  $EDB_TEST_DATA_HIGH  0x38393a3b  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x3c3d3e3f  # AAD3
	wr_reg  $EDB_IDF_TAG  0x6                #test_idf_tag = 6
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

    # ********* the 1st 128-bit input set
	#row 12  
	wr_reg  $EDB_IDF_CONTROL  0x80400012     #CMD bits 3b'010  opcode 0x2  msw = 1b'1 (w1)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x00010203  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x04050607  # payload0
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

	#row 13  
	wr_reg  $EDB_IDF_CONTROL  0x00400012     #CMD bits 3b'010  opcode 0x2  msw = 1b'0 (w2)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x008090a0b # 
	wr_reg  $EDB_TEST_DATA_LOW   0x0c0d0e0f  # payload1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

    # reading back the CTX values
    #_ACTION#=[delay_cycles 10 
	VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W1_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg  $EDB_PCF_W1_OUTPUT_LOW` 
	VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W2_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg  $EDB_PCF_W2_OUTPUT_LOW` 

    #compare the CTX values with the expected ones
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_HIGH  0x793b3fd2
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_LOW   0x52941224
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_HIGH  0xa6afdc5b
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_LOW   0xe7f501b9

    # ********* the 2nd 128-bit input set
	#row 12  
	wr_reg  $EDB_IDF_CONTROL     0x80400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'1 (w1)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x10111213  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x14151617  # payload0
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 13  
	wr_reg  $EDB_IDF_CONTROL     0x00400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'0 (w2)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x18191a1b  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x1c1d1e1f  # payload1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# reading back the CTX values
#_ACTION#=[delay_cycles 10 
	VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W1_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg  $EDB_PCF_W1_OUTPUT_LOW` 
	VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W2_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg  $EDB_PCF_W2_OUTPUT_LOW` 
	
#compare the CTX values with the expected ones
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_HIGH  0x150696da
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_LOW   0x12045c1c
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_HIGH  0x6077d3ca
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_LOW   0xc774accf

# ********* the 3th 128-bit input set
	#row 12  
	wr_reg  $EDB_IDF_CONTROL     0x80400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'1 (w1)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x20212223  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x24252627  # payload0
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 13  
	wr_reg  $EDB_IDF_CONTROL     0x00400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'0 (w2)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x28292a2b  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x2c2d2e2f  # payload1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# reading back the CTX values
#_ACTION#=[delay_cycles 10 
	VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W1_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg  $EDB_PCF_W1_OUTPUT_LOW` 
	VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W2_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg  $EDB_PCF_W2_OUTPUT_LOW` 

#compare the CTX values with the expected ones
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_HIGH  0xc3d530d8
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_LOW   0x48d665d8
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_HIGH  0x1a49cbb5
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_LOW   0x00b88bbb
# ****************************

# ********* the 4th 128-bit input set
	#row 12  
	wr_reg  $EDB_IDF_CONTROL     0x80400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'1 (w1)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x30313233  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x34353637  # payload0
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 13  
	wr_reg  $EDB_IDF_CONTROL     0x00400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'0 (w2)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x38393a3b  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x3c3d3e3f  # payload1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# reading back the CTX values
#_ACTION#=[delay_cycles 10 
	VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W1_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg  $EDB_PCF_W1_OUTPUT_LOW` 
	VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W2_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg  $EDB_PCF_W2_OUTPUT_LOW` 

#compare the CTX values with the expected ones
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_HIGH  0x624ae61d
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_LOW   0x1667229c
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_HIGH  0x302dc6ff
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_LOW   0x0bb4d70b

# ********* the 5th 128-bit input set
	#row 12  
	wr_reg  $EDB_IDF_CONTROL     0x80400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'1 (w1)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x40414243  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x44454647  # payload0
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 13  
	wr_reg  $EDB_IDF_CONTROL     0x00400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'0 (w2)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x48494a4b  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x4c4d4e4f  # payload1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# reading back the CTX values
#_ACTION#=[delay_cycles 10 
	VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W1_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg  $EDB_PCF_W1_OUTPUT_LOW` 
	VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W2_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg  $EDB_PCF_W2_OUTPUT_LOW` 

#compare the CTX values with the expected ones
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_HIGH  0xdbbc8566
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_LOW   0xd6f5b158
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_HIGH  0xda99a2ff
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_LOW   0x2e01dda6

# ********* the 6th 128-bit input set
	#row 12  
	wr_reg  $EDB_IDF_CONTROL     0x80400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'1 (w1)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x50515253  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x54555657  # payload0
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 13  
	wr_reg  $EDB_IDF_CONTROL     0x00400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'0 (w2)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x58595a5b  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x5c5d5e5f  # payload1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# reading back the CTX values
#_ACTION#=[delay_cycles 10 
	VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W1_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg  $EDB_PCF_W1_OUTPUT_LOW` 
	VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W2_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg  $EDB_PCF_W2_OUTPUT_LOW` 

#compare the CTX values with the expected ones
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_HIGH  0x29b89c34
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_LOW   0xad1e5feb
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_HIGH  0xa70e7aae
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_LOW   0x4328289c
# ****************************

# ********* the 7th 128-bit input set
	#row 12  
	wr_reg  $EDB_IDF_CONTROL     0x80400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'1 (w1)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x60616263  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x64656667  # payload0
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 13  
	wr_reg  $EDB_IDF_CONTROL     0x00400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'0 (w2)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x68696a6b  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x6c6d6e6f  # payload1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# reading back the CTX values
#_ACTION#=[delay_cycles 10 
	VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W1_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg  $EDB_PCF_W1_OUTPUT_LOW` 
	VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W2_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg  $EDB_PCF_W2_OUTPUT_LOW` 

#compare the CTX values with the expected ones
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_HIGH  0x3629b058
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_LOW   0x8350581c
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_HIGH  0xa8b97ccf
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_LOW   0x1258fa3b

# ********* the 8th 128-bit input set
	#row 12  
	wr_reg  $EDB_IDF_CONTROL     0x80400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'1 (w1)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x70717273  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x74757677  # payload0
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 13  
	wr_reg  $EDB_IDF_CONTROL     0x00400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'0 (w2)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x78797a7b  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x7c7d7e7f  # payload1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# reading back the CTX values
#_ACTION#=[delay_cycles 10 
	VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W1_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg  $EDB_PCF_W1_OUTPUT_LOW` 
	VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W2_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg  $EDB_PCF_W2_OUTPUT_LOW` 

#compare the CTX values with the expected ones
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_HIGH  0xbe2c5026
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_LOW   0x047ba726
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_HIGH  0x48969cff
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_LOW   0x8ba10ae3

# ********* the 9th 128-bit input set
	#row 12  
	wr_reg  $EDB_IDF_CONTROL     0x80400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'1 (w1)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x80818283  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x84858687  # payload0
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 13  
	wr_reg  $EDB_IDF_CONTROL     0x00400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'0 (w2)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x88898a8b  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x8c8d8e8f  # payload1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# reading back the CTX values
#_ACTION#= delay_cycles 10 
	VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W1_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg  $EDB_PCF_W1_OUTPUT_LOW` 
	VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W2_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg  $EDB_PCF_W2_OUTPUT_LOW` 

#compare the CTX values with the expected ones
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_HIGH  0x0e05935d
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_LOW   0xf0c69374
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_HIGH  0x1892b76f
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_LOW   0xaf67133a

# ********* the 10th 128-bit input set
	#row 12  
	wr_reg  $EDB_IDF_CONTROL     0x80400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'1 (w1)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x90919293  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x94959697  # payload0
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 13  
	wr_reg  $EDB_IDF_CONTROL     0x00400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'0 (w2)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0x98999a9b  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x9c9d9e9f  # payload1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# reading back the CTX values
#_ACTION#= delay_cycles 10 
	VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W1_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg  $EDB_PCF_W1_OUTPUT_LOW` 
	VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W2_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg  $EDB_PCF_W2_OUTPUT_LOW` 

#compare the CTX values with the expected ones
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_HIGH  0xbd2cf203
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_LOW   0x1121bd8b
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_HIGH  0xb38127a4
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_LOW   0xd2eedeea
# ********* the 11th 128-bit input set
	#row 12  
	wr_reg  $EDB_IDF_CONTROL     0x80400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'1 (w1)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0xa0a1a2a3  # 
	wr_reg  $EDB_TEST_DATA_LOW   0xa4a5a6a7  # payload0
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 13  
	wr_reg  $EDB_IDF_CONTROL     0x00400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'0 (w2)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0xa8a9aaab  # 
	wr_reg  $EDB_TEST_DATA_LOW   0xacadaeaf  # payload1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# reading back the CTX values
#_ACTION#=[delay_cycles 10 
	VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W1_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg  $EDB_PCF_W1_OUTPUT_LOW` 
	VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W2_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg  $EDB_PCF_W2_OUTPUT_LOW` 

#compare the CTX values with the expected ones
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_HIGH  0x13276494
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_LOW   0xf402cd7c
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_HIGH  0x107fb3ec
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_LOW   0x3b247848

# ********* the 12th 128-bit input set
	#row 12  
	wr_reg  $EDB_IDF_CONTROL     0x80400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'1 (w1)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0xb0b1b2b3  # 
	wr_reg  $EDB_TEST_DATA_LOW   0xb4b5b6b7  # payload0
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 13  
	wr_reg  $EDB_IDF_CONTROL     0x00400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'0 (w2)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0xb8b9babb  # 
	wr_reg  $EDB_TEST_DATA_LOW   0xbcbdbebf  # payload1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# reading back the CTX values
#_ACTION#= delay_cycles 10 
	VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W1_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg  $EDB_PCF_W1_OUTPUT_LOW` 
	VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W2_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg  $EDB_PCF_W2_OUTPUT_LOW` 

#compare the CTX values with the expected ones
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_HIGH  0x34338e55
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_LOW   0x43628709
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_HIGH  0x2ac4a26f
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_LOW   0x5ea7ea4a

# ********* the 13th 128-bit input set
	#row 12  
	wr_reg  $EDB_IDF_CONTROL     0x80400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'1 (w1)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0xc0c1c2c3  # 
	wr_reg  $EDB_TEST_DATA_LOW   0xc4c5c6c7  # payload0
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 13  
	wr_reg  $EDB_IDF_CONTROL     0x00400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'0 (w2)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0xc8c9cacb  # 
	wr_reg  $EDB_TEST_DATA_LOW   0xcccdcecf  # payload1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# reading back the CTX values
#_ACTION#= delay_cycles 10 
	VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W1_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg  $EDB_PCF_W1_OUTPUT_LOW` 
	VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W2_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg  $EDB_PCF_W2_OUTPUT_LOW` 

#compare the CTX values with the expected ones
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_HIGH  0xd68d7315
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_LOW   0x1639b05b
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_HIGH  0x24e68b98
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_LOW   0x16d13983

# ********* the 14th 128-bit input set
	#row 12  
	wr_reg  $EDB_IDF_CONTROL     0x80400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'1 (w1)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0xd0d1d2d3  # 
	wr_reg  $EDB_TEST_DATA_LOW   0xd4d5d6d7  # payload0
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 13  
	wr_reg  $EDB_IDF_CONTROL     0x00400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'0 (w2)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0xd8d9dadb  # 
	wr_reg  $EDB_TEST_DATA_LOW   0xdcdddedf  # payload1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# reading back the CTX values
#_ACTION#= delay_cycles 10 
	VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W1_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg  $EDB_PCF_W1_OUTPUT_LOW` 
	VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W2_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg  $EDB_PCF_W2_OUTPUT_LOW` 

#compare the CTX values with the expected ones
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_HIGH  0x76d8e413
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_LOW   0x8594758d
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_HIGH  0xb9ad3b40
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_LOW   0x9259b26d

# ********* the 15th 128-bit input set
	#row 12  
	wr_reg  $EDB_IDF_CONTROL     0x80400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'1 (w1)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0xe0e1e2e3  # 
	wr_reg  $EDB_TEST_DATA_LOW   0xe4e5e6e7  # payload0
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 13  
	wr_reg  $EDB_IDF_CONTROL     0x00400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'0 (w2)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0xe8e9eaeb  # 
	wr_reg  $EDB_TEST_DATA_LOW   0xecedeeef  # payload1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# reading back the CTX values
#_ACTION#= delay_cycles 10 
	VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W1_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg  $EDB_PCF_W1_OUTPUT_LOW` 
	VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W2_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg  $EDB_PCF_W2_OUTPUT_LOW` 

#compare the CTX values with the expected ones
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_HIGH  0xcfc06e72
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_LOW   0x2be987b3
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_HIGH  0x767f70a7
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_LOW   0xb856b774

# ********* the 16th 128-bit input set
	#row 12  
	wr_reg  $EDB_IDF_CONTROL     0x80400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'1 (w1)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0xf0f1f2f3  # 
	wr_reg  $EDB_TEST_DATA_LOW   0xf4f5f6f7  # payload0
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 13  
	wr_reg  $EDB_IDF_CONTROL     0x00400012  #CMD bits 3b'010  opcode 0x2  msw = 1b'0 (w2)  key_index=0x40
	wr_reg  $EDB_TEST_DATA_HIGH  0xf8f9fafb  # 
	wr_reg  $EDB_TEST_DATA_LOW   0xfcfdfeff  # payload1
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# reading back the CTX values
#_ACTION#= delay_cycles 10 
	VALUE_EDB_PCF_W1_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W1_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W1_OUTPUT_LOW=`rd_reg  $EDB_PCF_W1_OUTPUT_LOW` 
	VALUE_EDB_PCF_W2_OUTPUT_HIGH=`rd_reg  $EDB_PCF_W2_OUTPUT_HIGH` 
	VALUE_EDB_PCF_W2_OUTPUT_LOW=`rd_reg  $EDB_PCF_W2_OUTPUT_LOW` 

#compare the CTX values with the expected ones
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_HIGH  0xb1ba2685
	cmp_eq  $VALUE_EDB_PCF_W1_OUTPUT_LOW   0xb3680914
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_HIGH  0x29fccb8d
	cmp_eq  $VALUE_EDB_PCF_W2_OUTPUT_LOW   0xcdde09e4


# auth tag check
	#row 14  
	wr_reg  $EDB_IDF_CONTROL  0x80400050     #CMD bits 3b'000  opcode 0x5  msw = 1b'0 (w1)
	wr_reg  $EDB_TEST_DATA_HIGH  0x00000000  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x00000000  # 
	wr_reg  $EDB_IDF_TAG  0x6                #test_idf_tag = 6
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 15  
	wr_reg  $EDB_IDF_CONTROL  0x00400000     #CMD bits 3b'000  opcode 0x0  msw = 1b'0 (w1)
	wr_reg  $EDB_TEST_DATA_HIGH  0x00000000  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x00000000  # 
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 16  
	wr_reg  $EDB_IDF_CONTROL  0x80400020     #CMD bits 3b'000  opcode 0x2  msw = 1b'0 (w1)
	wr_reg  $EDB_TEST_DATA_HIGH  0x00000000  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x00000000  # 
	wr_reg  $EDB_TEST_CONTROL_2  0x00000010  #  write 1 TO the test_idf_w1_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #
	#row 17  
	wr_reg  $EDB_IDF_CONTROL  0x00400000     #CMD bits 3b'000  opcode 0x0  msw = 1b'0 (w1)
	wr_reg  $EDB_TEST_DATA_HIGH  0x00000000  # 
	wr_reg  $EDB_TEST_DATA_LOW   0x00000000  # 
	wr_reg  $EDB_TEST_CONTROL_2  0x00000008  #  write 1 TO the test_idf_w2_wren Field of $EDB_TEST_CONTROL_2  
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# save the low bits of the tag values Into output registers
	wr_reg  $EDB_TEST_CONTROL_2  0x00000080  # write 4'b0001 TO the test_aut_read Field of the $EDB_TEST_CONTROL_2 reg'
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# reading back the low bit tag values
	VALUE_EDB_AUT_W1_HIGH=`rd_reg  $EDB_AUT_HIGH` 
	VALUE_EDB_AUT_W1_LOW=`rd_reg  $EDB_AUT_LOW` 

# save the high bits of the tag values Into output registers
	wr_reg  $EDB_TEST_CONTROL_2  0x00000100  # write 4'b0001 TO the test_aut_read Field of the $EDB_TEST_CONTROL_2 reg'
	wr_reg  $EDB_TEST_CONTROL_2  0x00000000  #

# reading back the high bit tag values
	VALUE_EDB_AUT_W2_HIGH=`rd_reg  $EDB_AUT_HIGH` 
	VALUE_EDB_AUT_W2_LOW=`rd_reg  $EDB_AUT_LOW` 

#compare the tag values with the expected ones
	cmp_eq  $VALUE_EDB_AUT_W1_HIGH  0xb2cea169 
	cmp_eq  $VALUE_EDB_AUT_W1_LOW   0xd6943fcd 
	cmp_eq  $VALUE_EDB_AUT_W2_HIGH  0x87ec837a 
	cmp_eq  $VALUE_EDB_AUT_W2_LOW   0xbf532855 

# CPU test IS done
#_ACTION#= delay_cycles 4 
	wr_reg  $EDB_TEST_CONTROL_1  0x0 
    if [ $REVISION == $NEW_OB1_REV_ID ]
    then
	wr_reg  $OB1_DEBUG_CONTROL  0x0 
    fi

}
#################################################################
# function main
#
# Description:
#   The main function of the script. Code to initiate other
#   algorithm tests can be added here.
#
# Input:
#   None
#
# Output:
#   None
#################################################################
function main
{

	echo -ne "\nInitiating XTS-AES algorithm tests..."

# test XTS-AES on each of six OB1s.
		for ((i=0; i<6; i=i+1))
			do
				echo -ne "\n\n\tRunning XTS-AES algorithm tests on ob1_$i"
				test_xts_aes i
			done
		echo -ne "\n\nXTS-AES algorithm tests successful. "

# test GCM-AES on each of six OB1s.
		for ((i=0; i<6; i=i+1))
			do
				echo -ne "\n\n\tRunning GCM-AES algorithm tests on ob1_$i"
				test_gcm_aes i
		    done
			echo -ne "\n\nGCM-AES algorithm tests successful. "

	echo -ne "\n\nCPU testing of EDB completed successfully !\n"
}

# call main function
main
