/*
 *    Copyright (c) 2003-2004 by Brocade Communications Systems, Inc.
 *    All rights reserved.
 *
 *    Description:
 *      RAS Event Header File
 *
 */
#ifndef	__RAS_EVENT_H__
#define	__RAS_EVENT_H__

#include <ras/ras_modules.h>	/* RAS_LOG_ID */
#include <raslog/raslog_comm.h>
#include <raslog/raslog_msgdef.h>
#include <raslog/raslog_msg.h>

typedef unsigned int raslog_oid_t;

#define RASEVT_USERNAME_LEN	41	/* 40 +1 */
#define RASEVT_IPADDR_LEN	64		
#define RASEVT_STR_LEN		16
#define RASEVT_ROLE_LEN		20	
#define RASEVT_AD_LEN		65	/* 64 +1 */
#define RASEVT_EVT_CLASS_STR_LEN  33	/* 32 +1 */
#define RASEVT_VERSION_LEN 40

#define RASEVT_WD_SIZE		4

#define RASEVT_MSG_ARG_LEN    MSG_ARG_LEN
#define RASEVT_MAX_MSG_ARGS   16
#define RASEVT_MSG_INFO_LEN   64      
#define RASEVT_MAX_FILE_LEN   16
#define RASEVT_MAX_FUNC_LEN   16

/* type of messages */
#define RASEVT_MSGS_PER_TYPE		1024
#define RASEVT_NUM_TYPES 		2
#define RASEVT_ALL_MSGS 		0
#define RASEVT_INT_MSGS 		1
#define RASEVT_EXT_MSGS 		2

#define RASEVT_INTERNAL_START		1000
#define RASEVT_INTERNAL_END		4999
#define RASEVT_EXTERNAL_START		5000
#define RASEVT_EXTERNAL_END		8999

#define RASLOG_ALL_INST			(-1)

#define	RAS_HNDL_MAGIC			0x1234

/*
 * information for chassis/switch parameters for rasevent_init()
 */
#define CHASSIS_SVC_NAME	"chassis"
#define SWITCH_SVC_NAME		"fcsw"
#define ESWITCH_SVC_NAME	"ethsw"
#define CHASSIS_SVC_ID		0
#define SWITCH_SVC_ID		1
#define ESWITCH_SVC_ID		2

/*
 * flags callers can pass into rasevt()
 */
#define RASEVT_NOFLAGS	0x00000000	/* Normal usage */
#define RASEVT_FRCLOG	0x00000001	/* Force duplicate message logging */
#define RASEVT_NO_SEEPROM	0x00000002	/* Don't store in SEEPROM */
#define RASEVT_FORCE_SEEPROM	0x00000010	/* override default filter */
#define RASEVT_IGNORE_FFDC	0x00000020	/* ignore AP FFDC */
#define RASEVT_NOTRAP	0x00000004	/* Don't generate an SNMP trap */
#define RASEVT_NODIAG	0x00000008	/* Don't display while in diag mode */
#define RASEVT_NOAUDIT	0x00000040	/* Do not Audit this message */
#define RASEVT_NOBURNIN RASLOG_NOFLAGS	/* No longer supported */

#define RASEVT_OID_INVALID   0x00000000

#define	RASEVT_OK	0
#define RASEVT_ARG_OFFSET_INVALID	0xff

/* this is defined in service.h, need decide to export it */
#define MAX_SVCNAME_LEN		16
#define MAX_SVCNAME_SIZE	MAX_SVCNAME_LEN + 1 /* for NUL character */
#define MAX_SVCINST_LEN		4
#define MAX_SVCINST_SIZE	MAX_SVCINST_LEN + 1 /* for NUL character */
#define MAX_GRPNAME_LEN		MAX_SVCNAME_LEN	
#define MAX_GRPNAME_SIZE	MAX_GRPNAME_LEN + 1 /* for NUL character */
#define MAX_COMPNAME_LEN	16
#define MAX_COMPNAME_SIZE	MAX_COMPNAME_LEN + 1 /* for NUL character */

#define MAX_INSTANCE	16

/*
 * rasevt severity values
 */
typedef enum {
	RASEVT_UNKNOWN = 0,
	RASEVT_CRITICAL = 1,
	RASEVT_ERROR,
	RASEVT_WARNING,
	RASEVT_INFO,
	RASEVT_SEVERITY_MAX = RASEVT_INFO
} rasevt_severity_t;

/*
 * handle used by kernel applications to identify their context
 */
typedef struct ras_hndl {
	int			magic;
	// said_t	    	instance;
	 char	    		comp_name[MAX_COMPNAME_SIZE+1];
#define SVC_OFFSET	24
#define INST_OFFSET	16
#define SET_HNDL_ID(svc, inst) ((svc << SVC_OFFSET) | (inst << INST_OFFSET))
#define GET_HNDL_ID(hndl)	(hndl.hndl)
#define HNDL2INST(hndl)		((hndl.hndl >> INST_OFFSET) & 0xff)
#define HNDL2SVC(hndl)		((hndl.hndl >> SVC_OFFSET) & 0xff)
#define HNDL2COMP(hndl)		(hndl.comp_name)
	 int			hndl;
} ras_hndl_t;

#define rasevt_hndl_t	ras_hndl_t

typedef struct {
	char ver;
	char magic;
	short evt_len;		/* evt len */
} rasevt_hdr_t;

#define RASEVT_HDR_SET_VER(evtp, value)	((evtp)->ver = (char)value)
#define RASEVT_HDR_GET_VER(evtp)    ((evtp)->hdr.ver)
#define RASEVT_HDR_SET_MAGIC(evtp, value) ((evtp)->magic = (char)value)
#define RASEVT_HDR_GET_MAGIC(evtp)	((char)(evtp)->magic)
#define RASEVT_HDR_SET_EVT_LEN(evtp, value) ((evtp)->evt_len = (short)value)
#define RASEVT_HDR_GET_EVT_LEN(evtp)	((evtp)->evt_len)

#define RASEVT_MAGIC		0x12
// #define	RASEVT_VER		2
// #define	RASEVT_AUDIT_VER	1
// #define	RASEVT_EVT_VER	        1


/* the sequence should not be changed */
typedef enum {
	RASEVT_VER_RASLOG_MSG = 1,
	RASEVT_VER_RASEVT = 2,
	RASEVT_VER_EVT = 3,
	RASEVT_VER_AUDIT = 4,
	RASEVT_VER_STORAGE = 5,
	RASEVT_VER_AUDIT_STORAGE = 6,
	RASEVT_VER_RASLOG_ERR = 7,
} rasevt_ver_code_t;

typedef struct {
/* header section */
	rasevt_hdr_t hdr;
	char username[RASEVT_USERNAME_LEN+1];
	char ipaddr[RASEVT_IPADDR_LEN];
	char interface[RASEVT_STR_LEN+1];
	char ad_number[RASEVT_AD_LEN+1];
	char role[RASEVT_ROLE_LEN+1];
	char app_name[RASEVT_STR_LEN+1];
} rasevt_audit_t;

/* Macro for AUDIT OBJECT */
#define RASAUDIT_GET_HDR(evtp)	(&(evtp)->hdr)
#define RASAUDIT_SET_VER(evtp, value)	((evtp)->hdr.ver = (char)value)
#define RASAUDIT_GET_VER(evtp)    ((evtp)->hdr.ver)
#define RASAUDIT_SET_MAGIC(evtp, value) ((evtp)->hdr.magic = (char)value)
#define RASAUDIT_GET_MAGIC(evtp)	((char)(evtp)->hdr.magic)
#define RASAUDIT_SET_EVT_LEN(evtp, value) ((evtp)->hdr.evt_len = (short)value)
#define RASAUDIT_GET_EVT_LEN(evtp)	((evtp)->hdr.evt_len)

#define RASAUDIT_SET_USERNAME(audit, value)	\
        (strncpy((audit)->username, value, RASEVT_USERNAME_LEN))
#define RASAUDIT_GET_USERNAME(auditp)	((auditp)->username)
#define RASAUDIT_SET_IPADDRESS(auditp, value) \
        (strncpy((auditp)->ipaddr, value, RASEVT_IPADDR_LEN))
#define RASAUDIT_GET_IPADDRESS(auditp)	 ((auditp)->ipaddr)
#define RASAUDIT_SET_INTERFACE(auditp, value) \
        (strncpy((auditp)->interface, value, RASEVT_STR_LEN))
#define RASAUDIT_GET_INTERFACE(auditp)	 ((auditp)->interface)
#define RASAUDIT_SET_AD_NUMBER(auditp, value) \
         (strncpy((auditp)->ad_number, value, RASEVT_AD_LEN))
#define RASAUDIT_GET_AD_NUMBER(auditp)	 ((auditp)->ad_number)
#define RASAUDIT_SET_ROLE(auditp, value)\
        (strncpy((auditp)->role, value, RASEVT_ROLE_LEN))
#define RASAUDIT_GET_ROLE(auditp)   ((auditp)->role)
#define RASAUDIT_SET_APP_NAME(auditp, value) \
        (strncpy((auditp)->app_name, value, RASEVT_STR_LEN))
#define RASAUDIT_GET_APP_NAME(auditp)	 ((auditp)->app_name)

typedef struct {
	/* header section */
	rasevt_hdr_t hdr;

	/* source section, caller info */
	int service_id;	/* service id, setup by raslog_init */
	int instance ;	/* instance number, setup by raslog_init */
	int comp_id;	/* component id, setup by raslog_init */
	char comp_name[MAX_COMPNAME_SIZE+1];
	int flags;	/* defined in raslog.h */
	int objid;	/* object ID of event */

	/* Filename/function name/line # where event was generated */
	char file[RLMO_MAX_FILE_LEN];
	char func[RLMO_MAX_FUNC_LEN];

	unsigned int line;

	int local_time_sec;
	int local_time_usec;
	
	/* system wide event information */
	int time_sec;
	int time_usec;
	int global_seq;		/* raslog global sequence number */
	int external_seq;	/* raslog external sequence number */
	short iq_drop;	/* incoming message queue message drop count */
	short rq_drop;	/* receiver message queue message drop count */

	/* msg info */
	char arg_offset[RLMO_MAX_MSG_ARGS]; /* word offsets of arguments */
	char arg[RLMO_MSG_ARG_LEN];	    /* packed arguments */

	/* Static message definition from message catalog */
	rlmd_msgdef_t msgdef;
} rasevt_evt_t;

#define RASEVT_EVT_GET_HDR(evtp)	(&(evtp)->hdr)

#define RASEVT_EVT_CPY(evtsrc, evtdest)	(*(evtdest) = *(evtsrc))
#define RASEVT_EVT_SET_VER(evtp, value)	((evtp)->hdr.ver = (char)value)
#define RASEVT_EVT_GET_VER(evtp)    ((evtp)->hdr.ver)
#define RASEVT_EVT_SET_MAGIC(evtp, value) ((evtp)->hdr.magic = (char)value)
#define RASEVT_EVT_GET_MAGIC(evtp)	((char)(evtp)->hdr.magic)
#define RASEVT_EVT_SET_EVT_LEN(evtp, value) ((evtp)->hdr.evt_len = (short)value)
#define RASEVT_EVT_GET_EVT_LEN(evtp)	((evtp)->hdr.evt_len)
#define RASEVT_EVT_SET_SVC_ID(evtp, value) ((evtp)->service_id = value)
#define RASEVT_EVT_GET_SVC_ID(evtp)	    ((evtp)->service_id)
#define RASEVT_EVT_SET_INSTANCE(evtp, value) ((evtp)->instance = value)
#define RASEVT_EVT_GET_INSTANCE(evtp)	    ((evtp)->instance)
#define RASEVT_EVT_SET_COMP_ID(evtp, value) ((evtp)->comp_id = value)
#define RASEVT_EVT_GET_COMP_ID(evtp)        ((evtp)->comp_id)
#define RASEVT_EVT_SET_COMP_NAME(evtp, value) \
            (strncpy((evtp)->comp_name, value, MAX_COMPNAME_LEN))
#define RASEVT_EVT_GET_COMP_NAME(evtp)        ((evtp)->comp_name)
#define RASEVT_EVT_SET_FLAGS(evtp, value)   ((evtp)->flags = value)
#define RASEVT_EVT_GET_FLAGS(evtp)	    ((evtp)->flags)
#define RASEVT_EVT_SET_OBJID(evtp, value)   ((evtp)->objid = value)
#define RASEVT_EVT_GET_OBJID(evtp)	    ((evtp)->objid)
#define RASEVT_EVT_SET_LINENUM(evtp, value) ((evtp)->line = value)
#define RASEVT_EVT_GET_LINENUM(evtp)	    ((evtp)->line)
#define RASEVT_EVT_SET_FILENAME(evtp, value) do { \
         if (strlen(value) >= RASEVT_MAX_FILE_LEN) {  \
	 strncpy((evtp)->file, value, RASEVT_MAX_FILE_LEN); \
	 (evtp)->file[RASEVT_MAX_FILE_LEN - 1] = '\0'; } else \
	 strcpy((evtp)->file, value); } while (0)
#define RASEVT_EVT_GET_FILENAME(evtp) 	((evtp)->file)
#define RASEVT_EVT_SET_FUNC(evtp, value) 	\
        do { if (strlen(value) >= RASEVT_MAX_FUNC_LEN) {  \
	 strncpy((evtp)->func, value, RASEVT_MAX_FUNC_LEN); \
	 (evtp)->func[RASEVT_MAX_FUNC_LEN - 1] = '\0'; } else \
	 strcpy((evtp)->func, value); } while (0)
#define RASEVT_EVT_GET_FUNC(evtp) 	((evtp)->func)
#define RASEVT_EVT_SET_CAT_NAME(evtp, value) 	\
	 strcpy((evtp)->catalog_name, value)
#define RASEVT_EVT_GET_CAT_NAME(evtp) 	((evtp)->catalog_name)
#define RASEVT_EVT_SET_SRCTIMESEC(evtp, value) \
        ((evtp)->local_time_sec = value)
#define RASEVT_EVT_GET_SRCTIMESEC(evtp)	((evtp)->local_time_sec)
#define RASEVT_EVT_SET_SRCTIMEUSEC(evtp, value) \
        ((evtp)->local_time_usec = value)
#define RASEVT_EVT_GET_SRCTIMEUSEC(evtp) ((evtp)->local_time_usec)
#define RASEVT_EVT_SET_TIMESEC(evtp, value)	((evtp)->time_sec = value)
#define RASEVT_EVT_GET_TIMESEC(evtp)	((evtp)->time_sec)
#define RASEVT_EVT_SET_TIMEUSEC(evtp, value) ((evtp)->time_usec = value)
#define RASEVT_EVT_GET_TIMEUSEC(evtp)	((evtp)->time_usec)
#define RASEVT_EVT_SET_GSEQ(evtp, value) ((evtp)->global_seq = value)
#define RASEVT_EVT_GET_GSEQ(evtp)	((evtp)->global_seq)
#define RASEVT_EVT_SET_ESEQ(evtp, value) ((evtp)->external_seq = value)
#define RASEVT_EVT_GET_ESEQ(evtp)	((evtp)->external_seq)
#define RASEVT_EVT_SET_IQDROP(evtp, value)  ((evtp)->iq_drop = value)
#define RASEVT_EVT_ADD_IQDROP(evtp, value)  ((evtp)->iq_drop += value)
#define RASEVT_EVT_GET_IQDROP(evtp)	((evtp)->iq_drop)
#define RASEVT_EVT_SET_RQDROP(evtp, value)	((evtp)->rq_drop = value)
#define RASEVT_EVT_ADD_RQDROP(evtp, value)	((evtp)->rq_drop += value)
#define RASEVT_EVT_GET_RQDROP(evtp)	((evtp)->rq_drop)
#define RASEVT_EVT_GET_MSG_ID(evtp)   (RLMD_GET_MSG_ID(&(evtp)->msgdef))
#define RASEVT_EVT_SET_MSG_ID(evtp)   (RLMD_SET_MSG_ID(&(evtp)->msgdef))

#define RASEVT_EVT_ARG_COMP(evtsrc, evtdest) \
        (memcmp((evtsrc)->arg, (evtdest)->arg, RASEVT_MSG_ARG_LEN))
#define RASEVT_EVT_SET_ARG_OFFSET_RESET(evtp) \
        bzero((evtp)->arg_offset, RASEVT_MAX_MSG_ARGS)
#define RASEVT_EVT_SET_ARG_OFFSET(evtp, idx, value) \
        ((evtp)->arg_offset[idx] = (char)value)
#define RASEVT_EVT_GET_ARG_OFFSET(evtp, idx) \
        ((unsigned int)((evtp)->arg_offset[idx]))
#define RASEVT_EVT_SET_ARG_RESET(evtp) \
        bzero((evtp)->arg, RASEVT_MSG_ARG_LEN)
#define RASEVT_EVT_COPY_ARG_OFFSET(dest, src) \
        memcpy((dest)->arg_offset, (src)->arg_offset, RLMO_MAX_MSG_ARGS)

#define RASEVT_EVT_COPY_ARG(dest, src) \
	memcpy((dest)->arg, (src)->arg, RLMO_MSG_ARG_LEN)

#define RASEVT_EVT_SET_ARG_STR(evtp, offset, ptr, size) \
        strncpy((char *)&(evtp)->arg[offset], ptr, size)
#define RASEVT_EVT_SET_ARG_STR(evtp, offset, ptr, size) \
        strncpy((char *)&(evtp)->arg[offset], ptr, size)
#define RASEVT_EVT_SET_ARG_1(evtp, offset, value) \
        (*(char *)&(evtp)->arg[offset] = (char)value)
#define RASEVT_EVT_SET_ARG_4(evtp, offset, value) \
        (*(int *)&(evtp)->arg[offset] = value)
#define RASEVT_EVT_SET_ARG_8(evtp, offset, value) \
        (*(double *)&(evtp)->arg[offset] = value)
#define RASEVT_EVT_GET_ARG_STR(evtp, offset) \
        ((char *)&(evtp)->arg[offset])
#define RASEVT_EVT_GET_ARG_4(evtp, offset) \
        (*(int *)&(evtp)->arg[offset])
#define RASEVT_EVT_GET_ARG_8(evtp, offset) \
        (*(double *)&(evtp)->arg[offset])
#define RASEVT_EVT_SET_MSGDEF(evtp, md) (((evtp)->msgdef) = md)
#define RASEVT_EVT_GET_MSGDEF(evtp) (&(evtp)->msgdef)

/* event object structure */
typedef struct {
	/* header section */
	rasevt_hdr_t hdr;

	/* generic event info */
	rasevt_evt_t    evt;

	/* Audit specific information */
	rasevt_audit_t 	audit;
} rasevt_t;

#define RASEVT_GET_EVT(evtp)	      (&(evtp)->evt)
#define RASEVT_CPY(evtsrc, evtdest)	(*(evtdest) = *(evtsrc))

#define RASEVT_GET_HDR(evtp)	(&(evtp)->hdr)
#define RASEVT_SET_VER(evtp, value)	((evtp)->hdr.ver = (char)value)
#define RASEVT_GET_VER(evtp)    ((evtp)->hdr.ver)
#define RASEVT_SET_MAGIC(evtp, value) ((evtp)->hdr.magic = (char)value)
#define RASEVT_GET_MAGIC(evtp)	((char)(evtp)->hdr.magic)
#define RASEVT_SET_EVT_LEN(evtp, value) ((evtp)->hdr.evt_len = (short)value)
#define RASEVT_GET_EVT_LEN(evtp)	((evtp)->hdr.evt_len)
#define RASEVT_SET_SVC_ID(evtp, value)	\
        RASEVT_EVT_SET_SVC_ID(&(evtp)->evt, value)
#define RASEVT_GET_SVC_ID(evtp)	      RASEVT_EVT_GET_SVC_ID(&(evtp)->evt)
#define RASEVT_SET_INSTANCE(evtp, value) \
        RASEVT_EVT_SET_INSTANCE(&(evtp)->evt, value)
#define RASEVT_GET_INSTANCE(evtp)     RASEVT_EVT_GET_INSTANCE(&(evtp)->evt)
#define RASEVT_SET_COMP_ID(evtp, value) \
        RASEVT_EVT_SET_COMP_ID(&(evtp)->evt, value)
#define RASEVT_GET_COMP_ID(evtp)      RASEVT_EVT_GET_COMP_ID(&(evtp)->evt)
#define RASEVT_SET_COMP_NAME(evtp, value) \
        RASEVT_EVT_SET_COMP_NAME(&(evtp)->evt, value)
#define RASEVT_GET_COMP_NAME(evtp)  RASEVT_EVT_GET_COMP_NAME(&(evtp)->evt)
#define RASEVT_SET_FLAGS(evtp, value) \
        RASEVT_EVT_SET_FLAGS(&(evtp)->evt, value)
#define RASEVT_GET_FLAGS(evtp)	\
        RASEVT_EVT_GET_FLAGS(&(evtp)->evt)
#define RASEVT_SET_OBJID(evtp, value) \
        RASEVT_EVT_SET_OBJID(&(evtp)->evt, value)
#define RASEVT_GET_OBJID(evtp)	\
        RASEVT_EVT_GET_OBJID(&(evtp)->evt)
#define RASEVT_SET_LINENUM(evtp, value) \
        RASEVT_EVT_SET_LINENUM(&(evtp)->evt, value)
#define RASEVT_GET_LINENUM(evtp) \
        RASEVT_EVT_GET_LINENUM(&(evtp)->evt)
#define RASEVT_SET_FILENAME(evtp, value) \
        RASEVT_EVT_SET_FILENAME(&(evtp)->evt, value)
#define RASEVT_GET_FILENAME(evtp) \
        RASEVT_EVT_GET_FILENAME(&(evtp)->evt)
#define RASEVT_SET_FUNC(evtp, value) 	\
        RASEVT_EVT_SET_FUNC(&(evtp)->evt, value)
#define RASEVT_GET_FUNC(evtp) 	      \
        RASEVT_EVT_GET_FUNC(&(evtp)->evt)
#define RASEVT_SET_CAT_NAME(evtp, value) 	\
        RASEVT_EVT_SET_CAT_NAME(&(evtp)->evt, value)
#define RASEVT_GET_CAT_NAME(evtp) 	      \
        RASEVT_EVT_GET_CAT_NAME(&(evtp)->evt)
#define RASEVT_SET_SRCTIMESEC(evtp, value) \
        RASEVT_EVT_SET_SRCTIMESEC(&(evtp)->evt, value)
#define RASEVT_GET_SRCTIMESEC(evtp)  \
        RASEVT_EVT_GET_SRCTIMESEC(&(evtp)->evt)
#define RASEVT_SET_SRCTIMEUSEC(evtp, value) \
        RASEVT_EVT_SET_SRCTIMEUSEC(&(evtp)->evt, value)
#define RASEVT_GET_SRCTIMEUSEC(evtp)	\
        RASEVT_EVT_GET_SRCTIMEUSEC(&(evtp)->evt)
#define RASEVT_SET_TIMESEC(evtp, value) \
        RASEVT_EVT_SET_TIMESEC(&(evtp)->evt, value)
#define RASEVT_GET_TIMESEC(evtp) \
        RASEVT_EVT_GET_TIMESEC(&(evtp)->evt)
#define RASEVT_SET_TIMEUSEC(evtp, value)\
        RASEVT_EVT_SET_TIMEUSEC(&(evtp)->evt, value)
#define RASEVT_GET_TIMEUSEC(evtp) \
        RASEVT_EVT_GET_TIMEUSEC(&(evtp)->evt)
#define RASEVT_SET_GSEQ(evtp, value) \
        RASEVT_EVT_SET_GSEQ(&(evtp)->evt, value)
#define RASEVT_GET_GSEQ(evtp)\
        RASEVT_EVT_GET_GSEQ(&(evtp)->evt)
#define RASEVT_SET_ESEQ(evtp, value)	\
        RASEVT_EVT_SET_ESEQ(&(evtp)->evt, value)
#define RASEVT_GET_ESEQ(evtp)	 \
        RASEVT_EVT_GET_ESEQ(&(evtp)->evt)
#define RASEVT_SET_IQDROP(evtp, value)\
        RASEVT_EVT_SET_IQDROP(&(evtp)->evt, value)
#define RASEVT_ADD_IQDROP(evtp, value)\
        RASEVT_EVT_ADD_IQDROP(&(evtp)->evt, value)
#define RASEVT_GET_IQDROP(evtp)	       \
        RASEVT_EVT_GET_IQDROP(&(evtp)->evt)
#define RASEVT_SET_RQDROP(evtp, value) \
        RASEVT_EVT_SET_RQDROP(&(evtp)->evt, value)
#define RASEVT_ADD_RQDROP(evtp, value)\
        RASEVT_EVT_ADD_RQDROP(&(evtp)->evt, value)
#define RASEVT_GET_RQDROP(evtp)	       \
        RASEVT_EVT_GET_RQDROP(&(evtp)->evt)
#define RASEVT_GET_MSG_ID(evtp)	 \
        RASEVT_EVT_GET_MSG_ID(&(evtp)->evt)
#define RASEVT_SET_MSG_ID(evtp)	\
        RASEVT_EVT_SET_MSG_ID(&(evtp)->evt)
#define RASEVT_ARG_COMP(evtsrc, evtdest) \
        RASEVT_EVT_ARG_COMP(&(evtsrc)->evt, &(evtdest)->evt)
#define RASEVT_SET_ARG_OFFSET_RESET(evtp) \
        RASEVT_EVT_SET_ARG_OFFSET_RESET(&(evtp)->evt)
#define RASEVT_SET_ARG_OFFSET(evtp, idx, value) \
        RASEVT_EVT_SET_ARG_OFFSET(&(evtp)->evt, idx, value)
#define RASEVT_GET_ARG_OFFSET(evtp, idx) \
        RASEVT_EVT_GET_ARG_OFFSET(&(evtp)->evt, idx)
#define RASEVT_SET_ARG_RESET(evtp) \
        RASEVT_EVT_SET_ARG_RESET(&(evtp)->evt)
#define RASEVT_SET_ARG_STR(evtp, offset, ptr, size) \
        RASEVT_EVT_SET_ARG_STR(&(evtp)->evt, offset, ptr, size)
#define RASEVT_SET_ARG_1(evtp, offset, value) \
        RASEVT_EVT_SET_ARG_1(&(evtp)->evt, offset, value)
#define RASEVT_SET_ARG_4(evtp, offset, value) \
        RASEVT_EVT_SET_ARG_4(&(evtp)->evt, offset, value)
#define RASEVT_SET_ARG_8(evtp, offset, value) \
        RASEVT_EVT_SET_ARG_8(&(evtp)->evt, offset, value)
#define RASEVT_GET_ARG_STR(evtp, offset) \
        RASEVT_EVT_GET_ARG_STR(&(evtp)->evt, offset)
#define RASEVT_GET_ARG_4(evtp, offset) \
        RASEVT_EVT_GET_ARG_4(&(evtp)->evt, offset)
#define RASEVT_GET_ARG_8(evtp, offset) \
        RASEVT_EVT_GET_ARG_8(&(evtp)->evt, offset)
#define RASEVT_SET_MSGDEF(evtp, md) \
        RASEVT_EVT_SET_MSGDEF(&(evtp)->evt, md)
#define RASEVT_GET_MSGDEF(evtp) \
        RASEVT_EVT_GET_MSGDEF(&(evtp)->evt)
#define RASEVT_GET_ARG_NUM(evtp) \
        RLMD_GET_ARG_NUM(RASEVT_EVT_GET_MSGDEF(&(evtp)->evt))
#define RASEVT_GET_ARG_TYPE(evtp, idx)\
        RLMD_GET_ARG_TYPE(RASEVT_EVT_GET_MSGDEF(&(evtp)->evt), idx)
#define RASEVT_SET_CLASS(evtp, value) \
        RLMD_SET_CLASS(RASEVT_EVT_GET_MSGDEF(&(evtp)->evt), value)
#define RASEVT_GET_CLASS(evtp) \
        RLMD_GET_CLASS(RASEVT_EVT_GET_MSGDEF(&(evtp)->evt))

/* Macro for AUDIT OBJECT */
#define RASEVT_GET_AUDIT(evtp)	(&(evtp)->audit)
#define RASEVT_AUDIT_CPY(evtsrc, evtdest)  (*(evtdest) = evtsrc->audit)
#define RASEVT_SET_USERNAME(evtp, value)	\
        RASAUDIT_SET_USERNAME(&(evtp)->audit, value)
#define RASEVT_GET_USERNAME(evtp)  \
        RASAUDIT_GET_USERNAME(&(evtp)->audit)
#define RASEVT_SET_IPADDRESS(evtp, value) \
        RASAUDIT_SET_IPADDRESS(&(evtp)->audit, value)
#define RASEVT_GET_IPADDRESS(evtp)  \
        RASAUDIT_GET_IPADDRESS(&(evtp)->audit)
#define RASEVT_SET_INTERFACE(evtp, value)\
        RASAUDIT_SET_INTERFACE(&(evtp)->audit, value)
#define RASEVT_GET_INTERFACE(evtp)  \
        RASAUDIT_GET_INTERFACE(&(evtp)->audit)
#define RASEVT_SET_AD_NUMBER(evtp, value) \
        RASAUDIT_SET_AD_NUMBER(&(evtp)->audit, value)
#define RASEVT_GET_AD_NUMBER(evtp)  \
        RASAUDIT_GET_AD_NUMBER(&(evtp)->audit)
#define RASEVT_SET_ROLE(evtp, value) \
        RASAUDIT_SET_ROLE(&(evtp)->audit, value)
#define RASEVT_GET_ROLE(evtp)  \
        RASAUDIT_GET_ROLE(&(evtp)->audit)

#define RASEVT_SET_APP_NAME(evtp, value)\
        RASAUDIT_SET_APP_NAME(&(evtp)->audit, value)
#define RASEVT_GET_APP_NAME(evtp)  \
        RASAUDIT_GET_APP_NAME(&(evtp)->audit)

/*
 *  Function Name:  rasevt_gethndl
 *
 *  Inputs:
 *    svc - service name, either CHASSIS_SVC_NAME or SWITCH_SVC_NAME
 *    inst - switch instance number, or 0 if chassis-based service
 *
 *  Outputs:    None.
 *
 *  Returns:    Returns zero if successful, -1 if fails
 *
 *  Description:
 *
 *  This function returns a rasevt structure that should be used
 *  by the calling application to pass into rasevt() to identify the
 *  caller.  If the parameters are invalid, a -1 is returned.
 *
 */
extern rasevt_hndl_t rasevt_gethndl(char *svc, int inst);

extern int rasevt_log2(__const char *file, __const char *function,
        unsigned int line, char *cat_name, rasevt_hndl_t hndl,
	raslog_oid_t obj_id, int flags, raslog_msg_id_t msg_id, ...);

extern rasevt_t *rasevt_get(raslog_cat_msg_id_t  msg_id);

extern int rasevt_set_msg_block(raslog_msg_id_t  msg_id, int blk_type);

/*
 *************************************************************************
 * Kernel space rasevt calls
 *************************************************************************
 */
#ifdef	__KERNEL__
/*
 *  Function Name:  rasevt
 *
 *  Inputs:
 *    hndl - handle structure returned by rasevt_gethndl()
 *    objid - object ID
 *    flags - raslog flags or'd together (i.e. RASLOG_NOFLAGS)
 *    msg_id - message ID of the message to send (i.e. NS_1001)
 *    ... - variable agrument list for the message text
 *
 *  Outputs:    None.
 *
 *  Returns:    Returns zero if successful, -1 if fails
 *
 *  Description:
 *
 *  This function sends a raslog message to the ras infrastructure to
 *  be displayed on the console.  It will also be sent to registered
 *  applications for the message and saved to persistent storage.
 *
 */

#ifdef RASLOG_CHK
#define rasevt(hndl, objid, flags, format, args...) \
do {rasevt_hndl_t temp = rasevt_gethndl(NULL, 0); temp = hndl;} while (0); \
	printk(format, ## args)
#else

extern inline int rasevt_log(__const char *file, __const char *function,
	      unsigned int line, char *cat_name, rasevt_hndl_t hndl,
	     raslog_oid_t obj_id, int flags, raslog_msg_id_t msg_id, ...);

#define rasevt(hndl, objid, flags, msg_id...) \
  rasevt_log2(__FILE__, __FUNCTION__, __LINE__, RAS_CAT_NAME, \
		 hndl, objid, flags,  msg_id)

#endif

/*
 *  Function Name:  ras_msgcat_register()
 *
 *  Inputs:
 *    tpa_name - Third party name defined TPA_NAME in ras_modules.h
 *    msgdef -   TPA message defination.
 *
 *  Outputs:    None.
 *
 *  Returns:    Returns zero if successful, -1 if fails
 *
 *  Description:
 *
 *  This function loads tpa message catalog. 
 *  If the parameters are invalid, a -1 is returned.
 *
 */
extern int ras_msgcat_register(char *tpa_name, rlmd_msgdef_t *msgdef);

/*
 *************************************************************************
 * User space rasevt calls
 *************************************************************************
 */
#else /* __USER__ */
/*
 *  Function Name:  rasevt_init
 *
 *  Inputs:
 *    svc - service name, either CHASSIS_SVC_NAME or SWITCH_SVC_NAME
 *    inst - switch instance number, or 0 if chassis-based service
 *    comp - a string indicating the component (i.e. "Name Server")
 *
 *  Outputs:    None.
 *
 *  Returns:    Returns zero if successful, -1 if fails
 *
 *  Description:
 *
 *  This function initializes the context for the user space application
 *  All future rasevt() calls within this process space will use the same
 *  service/instance/component name to identify where the rasevt() call
 *  came from.
 *
 */
extern int rasevt_init(char *svc, int inst, char *comp);

/*
 *  Function Name:  rasevt_init
 *
 *  Inputs:
 *    svc - service name, either CHASSIS_SVC_NAME or SWITCH_SVC_NAME
 *    inst - switch instance number, or 0 if chassis-based service
 *    comp - a string indicating the component (i.e. "Name Server")
 *
 *  Outputs:    ras_hndl_t.
 *
 *  Returns:    Returns hndl if successful, NULL if fails
 *
 *  Description:
 *
 *  This function initializes the service instance context for the user 
 *  space application. All future rasevt() calls within this process space
 *  will use the same
 *  service/instance/component name to identify where the rasevt() call
 *  came from.
 *
 */
extern ras_hndl_t ras_register(char *svc, int inst, char *comp);

/*
 *  Function Name:  rasevt_exit
 *
 *  Inputs:     None.
 *
 *  Outputs:    None.
 *
 *  Returns:    None.
 *
 *  Description:
 *
 *  This function cleans up all resources used by rasevt.  It should be
 *  called before a process exits.
 *
 */
extern void rasevt_exit(void);

/*
 *  Function Name:  rasevt_init_check
 *
 *  Inputs:     None.
 *
 *  Outputs:    None.
 *
 *  Returns:    int.
 *
 *  Description:
 *
 *  This function checks if rasevt_init was previously done.
 *
 */
extern int rasevt_init_check(void);


/*
 *  Function Name:  rasevt
 *
 *  Inputs:
 *    objid - object ID
 *    flags - raslog flags or'd together (i.e. RASLOG_NOFLAGS)
 *    msg_id - message ID of the message to send (i.e. NS_1001)
 *    ... - variable agrument list for the message text
 *
 *  Outputs:    None.
 *
 *  Returns:    Returns zero if successful, -1 if fails
 *
 *  Description:
 *
 *  This function sends a rasevt message to the ras infrastructure to
 *  be displayed on the console.  It will also be sent to registered
 *  applications for the message and saved to persistent storage.
 *
 */
#ifdef RASLOG_CHK
#define rasevt(objid, flags, format, args...) \
	printf(format, ## args)
#else
extern ras_hndl_t rasevt_gethndl_internal(void);

#define rasevt(objid, flags, msg_id...) \
	rasevt_log2(__FILE__, __FUNCTION__, __LINE__, \
		RAS_CAT_NAME, rasevt_gethndl_internal(), \
                objid, flags, msg_id)
#endif


#ifdef RASLOG_CHK
#define rasevt2(objid, flags, hndl, format, args...) \
	printf(format, ## args)
#else
#define rasevt2(hndl, objid, flags, msg_id...) \
	rasevt_log2(__FILE__, __FUNCTION__, __LINE__, \
		RAS_CAT_NAME, hndl, objid, flags, msg_id)
#endif

#endif /* __USER__ */

extern int rasevt_send_evt(__const char *file, __const char *function,
	    unsigned int line, rasevt_t *evt, ...);

#define rasevt_send(evt...) \
           rasevt_send_evt(__FILE__, __FUNCTION__, __LINE__, evt)


/*
 **************************************************************************
 * Private rasevt information, not to be used by applications
 **************************************************************************
 */
#define rlf_filter_check(msg_obj, msg_def, filter) OK

/*
 * Note that pre 5.1 software will encode RLMD_ATTR_NONE for all ras events (raslog
 * messages) + some RLMD_ATTR_OLD_AUDIT values.  But they should be interpreted as
 * having the RLMD_ATTR_LOG bit set as well
*/
/* RAS Event attributes */
#define RLMD_ATTR_NONE  	0
#define RLMD_ATTR_OLD_AUDIT 	1
#define RLMD_ATTR_AUDIT 	2
#define RLMD_ATTR_LOG   	4
#define	RLMD_ATTR_FFDC		8
#define RLMD_ATTR_USER_CONF_SEVERITY_CRITICAL	0x100
#define RLMD_ATTR_USER_CONF_SEVERITY_ERROR		0x200
#define RLMD_ATTR_USER_CONF_SEVERITY_WARNING	0x400
#define RLMD_ATTR_USER_CONF_SEVERITY_INFO		0x800

#define	RLMD_ATTR_USER_CONF_SEVERITY_DEFAULT	\
						~(RLMD_ATTR_USER_CONF_SEVERITY_INFO |	\
							RLMD_ATTR_USER_CONF_SEVERITY_WARNING |	\
							RLMD_ATTR_USER_CONF_SEVERITY_ERROR |	\
							RLMD_ATTR_USER_CONF_SEVERITY_CRITICAL)
#define RLMD_ATTR_INT_SYSLOG	16	

/* RAS Event pre-defined classes, DMTF standard */
#define RASEVENT_CLASS_NONE			0
#define RASEVENT_CLASS_				0
#define RASEVENT_CLASS_ZONE			1
#define RASEVENT_CLASS_SECURITY		2
#define RASEVENT_CLASS_CFG			3
#define RASEVENT_CLASS_FIRMWARE		4	
#define RASEVENT_CLASS_FABRIC		5	
#define RASEVENT_CLASS_FW			6 /* Fabric Watch */
#define RASEVENT_CLASS_LS			7 /* LS */ 
#define RASEVENT_CLASS_CLI			8
#define RASEVENT_CLASS_MAPS			9 
#define RASEVENT_CLASS_RAS			10

#define RASEVENT_CLASS_COMM			11
#define RASEVENT_CLASS_QOS			12
#define RASEVENT_CLASS_PROC_ERR		13		
#define RASEVENT_CLASS_DEV			14	
#define RASEVENT_CLASS_MODEL_CHG	15

/* RAS Event pre-defined classes, Brocade defined */
#define RASEVENT_CLASS_SWITCH		16	
#define RASEVENT_CLASS_PORT			17
#define RASEVENT_CLASS_DEVICE		18
#define RASEVENT_CLASS_PLATFORM		19
#define RASEVENT_CLASS_CONNECTION	20
#define RASEVENT_CLASS_HA			21
#define RASEVENT_CLASS_AGENT		22

#define RASEVENT_CLASS_ENV			23

#define RASEVENT_CLASS_MAX_SUPPORTED		11

#define RASEVT_TRACE_EVENT_ID(trace_event_ID, constant_int, optional_fmt_string) \
        TRACE_EVENT_ID(trace_event_ID, constant_int, optional_fmt_string);\
        char *##trace_event_ID##_FMT = {optional_fmt_string};\
        extern	 char *##trace_event_ID##_FMT;\

#endif /* __RAS_EVENT_H__ */
