#!/bin/bash
# Function to reduce the number of core files in a directory to
# a maximum of $KEEP. Delete from the oldest to the newest
# Defaults, can be overridden using command parameters

DESIRED_SPACE=20000
DESIRED_PERCENT=80
DO_GZIP=0
GZIP=/bin/gzip
RM=/bin/rm

shopt -s extglob nullglob

# A newline, used to help read difficult filenames
nl='
'

# Remove files from directory $1 (get back by going to $2) that are either
# core or core.[0-9][0-9]*. However keep at least $KEEP of them, which are
# the newest.
# Make use of the GNU ls program, which can output filenames
# like "fred" or "bill\"", including quotes

trimdir(){
	cd "$1"
	olddir="$2"
	# Do a quick check before we bother calling ls
	set -- core?(.+([0-9]))?(.gz)
	if (( $# > $KEEP )) ; then
		# Read in the core file names, dealing with weird characters
		OIFS="$IFS"; IFS="$nl"
		set -- $(ls --quoting-style=c -t1d "$@" 2>/dev/null)
		IFS="$OIFS"
		if (( $# > $KEEP )) ; then
			shift $KEEP
			for filename ; do
				# Strip off the " that the ls put on.
				j=${filename#\"}; j=${j%\"}; j=${j//\\\"/\"}
				# Append full pathname to the togo array
				togo[$C]="$(printf %s/%b "$PWD" "$j" )"
				let C+=1
			done
		fi
	fi
	cd "$olddir"
}

compress_in_dir(){
	cd "$1"
	olddir="$2"
	set -- core?(.+([0-9]))
  [ $# -gt 0 ] && $GZIP "$@"
  cd "$olddir"
}

# loop over directories except panic and trim each one.

tidy(){
	let C=0
	unset togo
	for i in !(panic)/.
	do
        if [ $i != "sassvrd/." ] &&
           [ $i != "dpc/." ] &&
           [ $i != "sattoold/." ]
        then
			trimdir "$i" ..
		fi
	done
	if (( $C > 0 )) ; then
		$RM -- "${togo[@]}"
	fi
}

compress_all_core_files(){
	for i in !(panic)/.
	do
        if [ $i != "sassvrd/." ] &&
           [ $i != "dpc/." ] &&
           [ $i != "sattoold/." ]
        then
			compress_in_dir "$i" ..
		fi
	done
}


# Return 0 when then is enough space, 1 otherwise
space(){
	set -- $(df .)
	# Skip header ("filesystem 1k-blocks used available use% mounted on")
	shift 7
	# $1=where, $2=size, $3=used $4=free $5=free% $6=mnt_point
	# Can either use DESIRED_SPACE as a number of blocks
	if (( $4 < $DESIRED_SPACE )) ; then return 1 ; fi
	# Or as a percentage
	percent=${5%\%}
	if (( $percent > $DESIRED_PERCENT )) ; then return 1 ; else return 0 ; fi
}

# Loop, with smaller and smaller values of KEEP until we have enough space
# Returns 0 is there is enough space, 1 otherwise
tidy_till_space(){
	for KEEP in 4 3 2 ; do
		if space ; then return  0; fi
		tidy
	done
	# Do it one more time to set return code
	space
}

usage(){
	u="Usage: $0 [-c] [-p percent_space] [-n desired_free_blocks] [-P rm_program]" 
	echo "$u" >&2
	echo "Usage: $0 -h [this help]" >&2
	exit ${1:-2}
}

# Main program.
# Decode arguements
while getopts "hcp:n:P:" option ;do
	case "$option" in
	c)  DO_GZIP=1 ;;
	p)	DESIRED_PERCENT="$OPTARG" ;;
	n)	DESIRED_SPACE="$OPTARG" ;;
	P)	RM="$OPTARG" ;;
	h)	usage 0;;
	\?)	usage ;;
	esac
done
shift $(($OPTIND - 1))

if [ $# -ne 0 ] ; then
	usage
fi

# First attempt to compress files that can be compressed
if [ $DO_GZIP -eq 1 ]; then 
	compress_all_core_files
fi

# clean up additional files 
tidy_till_space
