/*
 *
 *  File Name:      ras_trace.h
 *
 *
 *
 *  Description:
 *
 *  This header file defines the user interface to the Trace Subsystem.
 *  Callers use the interface to create trace buffers and store
 *  program execution information into them.
 *
 *  It contains:
 *
 *  1.  Function prototypes 
 *  2.  Subsystem Identifiers assigned for use to other subsystems
 *  3.  Trace entry data type
 *  4.  Macro definitions that allow users to create Event IDs.
 *
 *
 *  This file is divided into two interfaces.  User-level software uses one
 *  and kernel-level software the other.  The __KERNEL__ flag is used to 
 *  distinguish between code compiled for user space and kernel-level code.
 *
 *  Note:  The management Interface to the Trace Subsystem is contained in 
 *         separate header file, "ras_trace_freeze.h"
 *
 *
 *  Warnings:
 *
 *  None.
 *
 *
 *
 *
 *  Copyright (C) 2003 Brocade Communications
 *  All rights reserved.
 *
 *  These coded instructions and statements contain unpublished trade
 *  secrets and proprietary information.  They are protected by federal
 *  copyright law and by trade secret law, and may not be disclosed to
 *  third parties or used, copied, or duplicated in any form, in whole
 *  or in part, without the prior written consent of Brocade Communications.
 *
 *
 *
 *
 *
 *  Change History:
 *
 *  Date        Author      Description
 *
 *  8/4/03      S. Stolper  Initial Version
 *  8/12/03	S. Stolper	Add include for module IDs.
 *
 */


/*  Avoid Duplication
 */
#ifndef __RAS_TRACE_H__
#define __RAS_TRACE_H__
#ifdef	__KERNEL__
#include <linux/types.h>
#else
#include <inttypes.h>
#endif

/*  Define Function Return Values */
#define TRACE_SUCCESS			(0)
#define TRACE_ERROR			(-1)
#define TRACE_ERR_BAD_STATE		(-2)
#define TRACE_ERR_NOCREATE		(-3)
#define TRACE_ERR_NORESOURCES		(-4)
#define TRACE_ERR_DRIVER		(-5)
#define	TRACE_ERR_LISTFULL		(-6)
#define	TRACE_ERR_LISTEMPTY		(-7)
#define	TRACE_ERR_BADINPUT		(-8)
#define	TRACE_ERR_DUPLICATE		(-9)
#define	TRACE_ERR_BADSLOT		(-10)
#define	TRACE_ERR_BADMSGID		(-11)
#define	TRACE_ERR_NOTREADY		(-12)

/* Trace severity levels */
#define TRACE_MIN_LVL	(TRACE_1)
#define TRACE_1			(1)
#define TRACE_2			(2)
#define TRACE_3			(3)
#define TRACE_4			(4)
#define TRACE_5			(5)
#define TRACE_6			(6)
#define TRACE_7			(7)
#define TRACE_8			(8)
#define TRACE_9			(9)
#define TRACE_10		(10)	/* level 10 is not stored */
#define TRACE_MAX_LVL		(TRACE_10)
#define TRACE_DEFAULT_LVL	(TRACE_5)

/*
 * Trace allows user to pass in 3 generic bytes and a level per entry.
 * The user can decide how these values are used, and when displaying data
 * or calling trace_snapshot, can pass in a filter on these values.
 */

#define TRACE_LVL(lvl)  (lvl & 0xff)
#define TRACE_G1(vu)	((vu & 0xff) << 8)
#define TRACE_G2(vu)	((vu & 0xff) << 16)
#define TRACE_G3(vu)	((vu & 0xff) << 24)
#define TRACE_G_NONE	(0)

#define TRACE_LVL_ON (0x000000FF)
#define TRACE_G1_ON	(0x0000FF00)
#define TRACE_G2_ON	(0x00FF0000)
#define TRACE_G3_ON	(0xFF000000)
#define TRACE_G_ALL_ON	(0xFFFFFFFF)
#define TRACE_SW(vu) (TRACE_G1(vu))
#define TRACE_LVL2G1(lvl)	((lvl >> 8) & 0xff)
#define TRACE_LVL2G2(lvl)	((lvl >> 16) & 0xff)
#define TRACE_LVL2G3(lvl)	((lvl >> 24) & 0xff)

#define TRACE_8_ENTRY_SZ	(8)
#define TRACE_16_ENTRY_SZ	(16)
#define TRACE_32_ENTRY_SZ	(32)
#define TRACE_64_ENTRY_SZ	(64)
#define TRACE_128_ENTRY_SZ	(128)
#define TRACE_256_ENTRY_SZ	(256)


typedef struct {
	unsigned int  td_word[2];
}  trace_8_data_t;

typedef struct {
	unsigned int  td_word[4];
}  trace_16_data_t;


typedef struct {
	unsigned int  td_word[8];
}  trace_32_data_t;

typedef struct {
	unsigned int  td_word[16];
}  trace_64_data_t;

typedef struct {
	unsigned int  td_word[32];
}  trace_128_data_t;

typedef struct {
	unsigned int  td_word[64];
}  trace_256_data_t;


/* Generic bytes filter */
typedef struct trace_filter {
	int	bmap; /* bitmap indicating which bytes of the data field to use */
	int	data;
} trace_filter_t;


/* Limits */
#define TRACE_MAX_NUM_ENTRIES	(1024 * 64) /* 64k entries */

/*  Module IDs
 *
 *  Module IDs identify software when it creates a 
 *  trace buffer or stores trace information.  The values
 *  are centrally managed by the RAS Subsystem and 
 *  assigned to user software subsystems.  The values can 
 *  also be used by operators from the CLI to manipulate
 *  trace buffers maintained by the system.
 */
#include <ras/ras_modules.h>


/*  Trace Entry Data Type
 *
 *  This union contains the bytes to be preserved in the
 *  trace buffer.  Developers fill the union and pass its
 *  address as an argument to the trace_store() function.
 *  The trace entry is accessible as both bytes and words.
 *  The format of the data is associated with a unique 
 *  trace event ID using the TRACE_EVENT_ID macro.
 */
#define TRACE_BYTES_PER_ENTRY		(64)  // was (32)
#define TRACE_UINT16S_PER_ENTRY		(TRACE_BYTES_PER_ENTRY / sizeof(uint16_t))
#define TRACE_WORDS_PER_ENTRY		(TRACE_BYTES_PER_ENTRY / sizeof(int))

typedef union {
	unsigned char td_byte[TRACE_BYTES_PER_ENTRY];
	uint16_t  td_uint16[TRACE_UINT16S_PER_ENTRY];
	unsigned int  td_word[TRACE_WORDS_PER_ENTRY];
}  trace_data_t;

/*
 * buffer entry sizes
 */
#define TRACE_DEF_ENTRY_SZ	(32)
#define TRACE_MIN_ENTRY_SZ	(8)
#define TRACE_MAX_ENTRY_SZ	(256)


typedef union {
                unsigned char td_byte[TRACE_MAX_ENTRY_SZ];
		uint16_t  td_uint16[TRACE_MAX_ENTRY_SZ / sizeof(uint16_t)];
                unsigned int  td_word[TRACE_MAX_ENTRY_SZ / sizeof(int)];
}  trace_long_data_t;



/*  Macros to define EVENT IDs
 *
 *  Users invoke this macro to create an event ID to be 
 *  used in calls to trace_store().
 *
 *  When compiled as part of an analytical tool, the macro
 *  creates an entry that can be used as part of an array
 *  to decode the message produced by the switch.
 *
 *  TRACE_EVENT_ID(event_ID,  constant_int,  fmt_string)
 *
 *  event_ID - Mnemonic used to identify trace entry format
 *  constant_int - constant value of mnemonic
 *  fmt_string - Optional printf-like formatting string.
 *  
 *  If supplied, the format string is compiled into a
 *  utility that post-processes trace dumps.  If it is 
 *  NULL, the entry is displayed as raw hexadecimal.
 *  The special characters in the format string must be in
 *  the same order as they appear in the trace entry.  
 *  Allowable values are:
 *  
 *  %d = display 4 bytes as integer value
 *  %x = display 4 bytes as hexadecimal value
 *  %c = display 1 byte as hexadecimal value 
 *  Note that this differs from how the printf() function 
 *  works.
 *
 *  EXAMPLE:
 *  
 *  TRACE_EVENT_ID(FW_TEMP_TRACE, 1, "tempID= %d, val= %x")
 *
 *  Callers place the IDs into a header file 
 *  "xxx_trace_id.h", where 'xxx' is the name of the 
 *  Subsystem Identifier assigned by the Trace Subsystem.  
 *  Developers include these files in their 'C' source 
 *  files and use the event IDs when storing trace 
 *  information.
 */
#ifdef __ANALYSIS_TOOL__

/*
 * the first module id is used as the entry identify
 */
#define TRACE_EVENT_ID(trace_event_ID, constant_int, optional_fmt_string) \
	{MODULE_ID, #trace_event_ID, constant_int, optional_fmt_string},

#else
#define TRACE_EVENT_ID(trace_event_ID, constant_int, optional_fmt_string) \
	enum { trace_event_ID = constant_int }; \
//        static char *##trace_event_ID##_FMT __attribute__ ((unused)) = {optional_fmt_string};
#endif


/*
 * we use the following format to get finer
 * granularity of buffer indexing inside one module id
 */
#define TRACE_BUFF_ID(modid, subid)	(((subid) << 16) + (modid))
#define TRACE_BUFF_MODID(id)		((id) & 0xFFFF)
#define TRACE_BUFF_SUBID(id)		(((id) >> 16) & 0xFFFF)

/*  Select interface by determining if the code being 
 *  compiled runs in kernel or user space.
 */
#ifdef __KERNEL__

int trace_core_notify(char *address, u_int size, int id);

/*
 * platform-specific dump funtions
 */
typedef struct trace_dump_func {
	int (*open)(unsigned int);
	int (*close)(int);
	int (*write)(int, char *buf, int size);
} trace_dump_func_t;

/*  ***** Kernel-level interface *****
 */


/*
 *
 *
 *  Function Name:  trace_init
 *
 *
 *  Inputs:	index        	-  Each module must identify itself when
 *					creating a trace buffer.  
 *		number_entries	-  Indicates how many entries stored in 
 *					buffer before wrapping (power of 2).
 *		module_string_p	-  Eight character string (not including 
 *					NULL) or NULL.
 *		prop		- trace buffer property
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	Returns zero if successful, non-zero if fails (See below)
 *
 *
 *  Description:
 *
 *  This function creates a trace buffer in kernel memory and alerts the 
 *  Trace Subsystem that the new buffer exists.  It also records the module ID
 *  in the buffer so that the trace information is associated with the new 
 *  module.
 *
 *  Each kernel level module creates a trace buffer by calling this function
 *  from its module initialization function.  Modules identify themselves by
 *  supplying a index preprocessor definition from "ras_trace.h".  The 
 *  module uses the same ID when tracing information into the buffer.  
 *  If supplied, the module character string is preserved in the trace buffer 
 *  for identification purposes.  Callers may use the string to identify a 
 *  kernel module.
 *  
 *  If this function is called more than once with the same module ID, it
 *  reports an error.  If it is unable to create the trace buffer, it LOGs
 *  an error and return TRACE_ERR_NOCREATE.  Subsequent calls to trace 
 *  information do not succeed, but the calling software continues to run
 *  normally.  If this function fails, it may be retried at a later time.
 *
 *
 *  NOTE: 
 *
 *  "number_entries" must be a power of two (2).  If not, the function 
 *  returns TRACE_ERR_BADPARAM and LOG a message.
 *
 */
int     
trace_init(int index, unsigned int number_entries, char *module_string_p, void *prop); 

/*
 *
 *
 *  Function Name:  trace_thread_identify
 *
 *
 *  Inputs:	thread_string_p  -  Eight character string (not including
 *					NULL)
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function should only be called from kernel threads.  It accepts a 
 *  string that is used to identify all trace entries that are generated 
 *  by the calling thread context.
 *
 *  Developers may choose to identify trace entries created by a thread with a 
 *  string to assist post-mortem debugging.  Each kernel module trace buffer 
 *  supports up to 16 thread identification entries.  If there are already 
 *  sixteen threads identified in the buffer, calling trace_thread_identify() 
 *  generates a LOG message and modifies the buffer metrics.
 *
 *  When a kernel thread exits normally, it should remove its identification 
 *  information from the trace buffer by calling 
 *  trace_thread_remove_identity().
 *
 *  If this function is called before calling trace_init(), it logs a message 
 *  to indicate a bug in the code.  It does not ASSERT() like the user-level 
 *  version of this function.  If it is called after a failed call to 
 *  trace_init(), or called after trace_cleanup(), it returns without 
 *  performing an operation.
 *
 *
 *  NOTE: 
 *
 *  This function cannot be called from interrupt level. It generates a 
 *  LOG message if called from an ISR or tasklet.
 *
 */
void
trace_thread_identify(char *thread_string_p);




/*
 *
 *
 *  Function Name:  trace_thread_remove_identity
 *
 *
 *  Inputs:	None.
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function should only be called from kernel threads.  It removes the
 *  association between PID and 8-character string created by 
 *  trace_thread_identify().  
 *
 *  Threads that terminate abnormally should not call this function.  This way,
 *  the buffer entries left behind by the thread are associated with the 
 *  "thread string" it placed into the buffer prior to making the entries.
 *
 *  If this function is called before calling trace_init(), it logs a message 
 *  to indicate a bug in the code.  It does not call ASSERT() like the user-
 *  space version of the function.  If it is called after a failed call to 
 *  trace_init(), or called after trace_cleanup(), it returns without 
 *  performing an operation.  If it is called when there is no 
 *  entry for the thread in the buffer, it returns without performing an 
 *  operation.
 *
 *
 *  NOTE: 
 *
 *  This function cannot be called from interrupt level. It generates a 
 *  LOG message if called from an ISR or tasklet.
 *
 */
void
trace_thread_remove_identity(void);




/*
 *
 *
 *  Function Name:  trace_store
 *
 *
 *  Inputs:	index		-  Identifies software performing the trace.
 *		event_ID	-  Identifies format of the trace entry.
 *		data_p		-  Address of a trace_data_t structure.  
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function allows software to store execution information into a trace 
 *  buffer.  Callers place the trace data into a trace_data_t structure 
 *  without regard to endianness or word alignment.  The function places the
 *  contents of the structure into the trace buffer indicated by the index.
 *  The event_ID indicates the format of the trace entry such that it can be 
 *  decoded and analyzed.  All entries with the same trace ID share the same 
 *  formatting.  
 *
 *  If the trace_store() function is called before trace_init(), it LOGs an
 *  error message, but does not ASSERT() like the user-level version.  If the 
 *  trace_init() function fails to create the trace buffer, this function does 
 *  not store information and but does not produce an error. 
 *
 *
 */
void trace_store(int index, unsigned short int event_ID, void* data_p);
void trace_store_lvl(int index, unsigned short int event_ID, int lvl, void* data_p);



/*
 *
 *
 *  Function Name:  trace_cleanup
 *
 *
 *  Inputs:	index  -  Identifies buffer to remove.
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function removes the process trace buffer and notifies the Trace 
 *  Subsystem that the application is exiting. When a module is unloaded, or 
 *  terminates normally, it must call this function.  Modules should place a 
 *  call to trace_cleanup() into their un-initialization module.  
 *
 *  The index must be the same one used to create the trace buffer the 
 *  caller was using.
 *
 *  If the application calls the trace_store() function after it invokes this 
 *  one, it does not place data into the buffer.  However, program execution 
 *  is not affected and the function does not generate an error.  This 
 *  handles a potential race condition where one thread calls trace_cleanup() 
 *  and another calls trace_store() before the first thread can terminate the 
 *  process. 
 *
 *  NOTE: 
 *
 *  This function cannot be called from interrupt level. It generates a 
 *  LOG message if called from an ISR or tasklet.
 *
 *
 */
void
trace_cleanup(int index);



/*
 *
 *
 *  Function Name:  trace_dump
 *
 *
 *  Inputs:	dump cause.
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	0 if success, negative if failed.
 *
 *
 *  Description:
 *
 *	This function triggers the trace device doing trace
 *	dump using the ATA polling driver, it is a
 *	non-recoverable event
 *
 */
int
trace_dump(int);

/*
 *
 *
 *  Function Name:  trace_dump_register
 *
 *
 *  Inputs:	func	- dump function block.
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *	This function provides the interface for platform
 *	software to register the platform specific dump functions
 *	such as ATA on CP, flash on BP
 *
 */
void
trace_dump_register(trace_dump_func_t *);


#else
/*  ***** User-level interface *****
 */
#include <sys/time.h>

/*
 * trace buffer handle
 */
typedef int trace_hndl_t; 

/*
 * exposed trace entry data to user application
 */
typedef struct trace_entry {
	uint16_t	mod_id;
	uint16_t	evt_id;
	int			generic;
	struct timeval	time;
	trace_long_data_t	data;
} trace_entry_t;

/*
 * trace callback function
 */
typedef void (*trace_cb_func_t)(trace_entry_t *entry_p);

/*
 * default trace buffer handle
 */
#define	TRACE_DEF_HANDLE	0x1010
#define	TRACE_FIRST_PID		0

/*
 *
 *
 *  Function Name:  trace_process_init
 *
 *
 *  Inputs:	module_ID	-  Each application must identify itself 
 *					when creating a trace buffer.  
 *		number_entries	-  Indicates how many entries stored in 
 *					buffer before wrapping (power of 2).
 *		module_string_p	-  Eight character string (not including 
 *					NULL) or NULL.
 *		prop		-  trace buffer property
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	Returns zero if successful, non-zero if fails (See below)
 *
 *
 *  Description:
 *
 *  This function creates a trace buffer in shared memory and alerts the 
 *  Trace Subsystem that the new buffer exists.  It also records the 
 *  application's module ID in the buffer so that the trace information is 
 *  associated with the new application process.
 *
 *  When a new user-level process begins execution it must call this function 
 *  before it can begin tracing information.  Every thread in the process 
 *  traces information into the buffer.  So, the size of the process's buffer 
 *  must account for the total number of threads the process expects to have.
 *  
 *  If this function is called more than once from the same application, it 
 *  reports an error, but returns normally.  If it is unable to create the trace 
 *  buffer, it LOGs an error and returns TRACE_ERR_NOCREATE.  Subsequent 
 *  calls to trace information do not succeed, but the calling software 
 *  continues to run normally.  If this function fails, it may be retried at a 
 *  later time.
 *
 *
 *  NOTE: 
 *
 *  "number_entries" must be a power of two (2).  If not, the function 
 *  ASSERT()s to indicate that ther is a bug in the software.
 *
 */
int     
trace_process_init(unsigned short int module_ID, 
		unsigned int number_entries, char * module_string_p,
		void *prop);

/*
 *
 *
 *  Function Name:  trace_module_init
 *
 *
 *  Inputs:	hndl		-  trace buffer handle
 *		module_ID	-  Each application must identify itself
 *					when creating a trace buffer.
 *		number_entries	-  Indicates how many entries stored in
 *					buffer before wrapping (power of 2).
 *		module_string_p	-  Eight character string (not including
 *					NULL) or NULL.
 *		prop		-  trace buffer property
 *
 *
 *  Outputs:    None.
 *
 *
 *  Returns:    Returns zero if successful, non-zero if fails (See below)
 *
 *
 *  Description:
 *
 *  This function nitializes the Library Object.  In the future, we may decide
 *  to have multiple Library Objects for a single process.  In this case, only
 *  this function needs to change.
 *
 *  When a new user-level process begins execution it must call this function
 *  before it can begin tracing information.  Every thread in the process
 *  traces information into the buffer.  So, the size of the process's buffer
 *  must account for the total number of threads the process expects to have.
 *
 *  If this function is called more than once from the same application, it
 *  reports an error, but returns normally.  If it is unable to create the
 *  trace buffer, it LOGs an error and returns TRACE_ERR_NOCREATE.  Subsequent
 *  calls to trace information do not succeed, but the calling software
 *  continues to run normally.  If this function fails, it may be retried at a
 *  later time.
 *
 *
 *  NOTE:
 *
 *  "number_entries" must be a power of two (2).  If not, the function
 *  ASSERT()s to indicate that there is a bug in the software.
 *
 */
int     
trace_module_init(trace_hndl_t *hndl, unsigned short int module_ID, 
		unsigned int number_entries, char * module_string_p,
		void *prop);


/*
 *
 *
 *  Function Name:  trace_data_init
 *
 *
 *  Inputs:	hndl		-  trace buffer handle
 *		module_ID	-  Each application must identify itself
 *					when creating a trace buffer.
 *		number_entries	-  Indicates how many entries stored in
 *					buffer before wrapping (power of 2).
 *		module_string_p	-  Eight character string (not including
 *					NULL) or NULL.
 *		prop		-  trace buffer property
 *
 *
 *  Outputs:    None.
 *
 *
 *  Returns:    Returns zero if successful, non-zero if fails (See below)
 *
 *
 *  Description:
 *
 *  This function nitializes the Library Object.  In the future, we may decide
 *  to have multiple Library Objects for a single process.  In this case, only
 *  this function needs to change.
 *
 *  When a new user-level process begins execution it must call this function
 *  before it can begin tracing information.  Every thread in the process
 *  traces information into the buffer.  So, the size of the process's buffer
 *  must account for the total number of threads the process expects to have.
 *
 *  If this function is called more than once from the same application, it
 *  reports an error, but returns normally.  If it is unable to create the
 *  trace buffer, it LOGs an error and returns TRACE_ERR_NOCREATE.  Subsequent
 *  calls to trace information do not succeed, but the calling software
 *  continues to run normally.  If this function fails, it may be retried at a
 *  later time.
 *
 *
 *  NOTE:
 *
 *  "number_entries" must be a power of two (2).  If not, the function
 *  ASSERT()s to indicate that there is a bug in the software.
 *
 */
int
trace_data_init(trace_hndl_t *hndl, unsigned short int module_ID,
	unsigned int size, char *module_string_p,
		int trace_sec, void **buf);


/*
 *
 *
 *  Function Name:  trace_thread_identify
 *
 *
 *  Inputs:	thread_string_p  -  Eight character string (not including
 *					NULL)
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function is called by threads in an application.  It accepts a 
 *  string that is used to identify all trace entries that are generated 
 *  by the calling thread context.
 *
 *  Every entry in the trace buffer contains the OS process ID of the thread 
 *  that created it.  The process ID (of the thread) uniquely identifies each 
 *  thread in a process.  Developers may choose to identify trace entries 
 *  created by a thread with a string to assist post-mortem debugging.  
 *  The application trace buffer supports up to 16 thread identification 
 *  entries.  If there are already sixteen threads identified in the buffer, 
 *  calling trace_thread_identify() generates a LOG message and modifies the 
 *  buffer metrics.
 *
 *  When an application thread exits normally, it should remove its 
 *  identification information from the trace buffer by calling 
 *  trace_thread_remove_identity().
 *
 *  If this function is called before calling trace_process_init(), it logs a
 *  message and ASSERT()s to indicate a bug in the code.  If it is called 
 *  after a failed call to trace_process_init(), or called after 
 *  trace_process_cleanup(), it returns without performing an operation.
 *
 *
 */
void
trace_thread_identify(char *thread_string_p);




/*
 *
 *
 *  Function Name:  trace_thread_remove_identity
 *
 *
 *  Inputs:	None.
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function is called by threads in an application.  It removes the
 *  association between PID and 8-character string created by 
 *  trace_thread_identify().  
 *
 *  Threads that terminate abnormally should not call this function.  This way,
 *  the buffer entries left behind by the thread are associated with the 
 *  "thread string" it placed into the buffer prior to making the entries.
 *
 *  If this function is called before calling trace_process_init(), it logs a 
 *  message and ASSERT()s to indicate a bug in the code.  If it is called after
 *  a failed call to trace_process_init(), or called after 
 *  trace_process_cleanup(), it returns without performing an operation. 
 *  If it is called when there is no entry for the thread in the 
 *  buffer, it logs a message and ASSERT()s to indicate a bug in the code.
 *
 *
 */
void
trace_thread_remove_identity(void);




/*
 *
 *
 *  Function Name:  trace_process_store
 *
 *
 *  Inputs:	module_ID  -  Identifies software performing the trace.
 *		event_ID   -  Identifies format of the trace entry.
 *		data_p     -  Address of a trace_data_t structure.  
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function allows software to store execution information into a trace 
 *  buffer.  Callers place the trace data into a trace_data_t structure 
 *  without regard to endianness or word alignment.  The function places the
 *  contents of the structure into the trace buffer that was created by the 
 *  application process context.  The event_ID indicates the format of the 
 *  trace entry such that it can be decoded and analyzed.  All entries with 
 *  the same trace ID share the same formatting.  
 *
 *  If the trace_store() function is called before trace_process_init(), it 
 *  LOGs an error message and ASSERT()s to indicate a bug in the code.  If the
 *  trace_process_init() function fails to create the trace buffer, this function 
 *  does not store information but does not produce an error. 
 *
 *
 */
#if 0
#define trace_process_store(module_ID, event_ID, data_p) \
	trace_process_store_lvl(module_ID, event_ID, TRACE_DEFAULT_LVL, data_p);
#else
void
trace_process_store(unsigned short int module_ID,
	unsigned short int event_ID, void* data_p);
#endif
void
trace_process_store_lvl(unsigned short int module_ID,
    unsigned short int event_ID, int lvl, void* data_p);

/*
 *
 *
 *  Function Name:  trace_module_store
 *
 *
 *  Inputs:	hndl       -  trace buffer handle
 *		module_ID  -  Identifies software performing the trace.
 *		event_ID   -  Identifies format of the trace entry.
 *		data_p     -  Address of a trace_data_t structure.
 *
 *
 *  Outputs:    None.
 *
 *
 *  Returns:    None.
 *
 *
 *  Description:
 *
 *  This function allows software to store execution information into a trace
 *  buffer.  Callers place the trace data into a trace_data_t structure
 *  without regard to endianness or word alignment.  The function places the
 *  contents of the structure into the trace buffer that was created by the
 *  application process context.  The event_ID indicates the format of the
 *  trace entry such that it can be decoded and analyzed.  All entries with
 *  the same trace ID share the same formatting.
 *
 *  If the trace_store() function is called before trace_process_init(), it
 *  LOGs an error message and ASSERT()s to indicate a bug in the code.  If the
 *  trace_process_init() function fails to create the trace buffer,
 *  this function does not store information but does not produce an error.
 *
 *
 */
#define trace_module_store(hndl, module_ID, event_ID, data_p) \
	trace_module_store_lvl(hndl, module_ID, event_ID, TRACE_DEFAULT_LVL, data_p)
void
trace_module_store_lvl(trace_hndl_t hndl, unsigned short int module_ID,
	unsigned short int event_ID, int lvl, void* data_p);


/*
 *
 *
 *  Function Name:  trace_process_cleanup
 *
 *
 *  Inputs:	None.
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function removes the process trace buffer and notifies the Trace 
 *  Subsystem that the application is exiting. When a process terminates 
 *  normally, it must call trace_process_cleanup(). 
 *
 *  If the application calls the trace_store() function after it invokes this 
 *  one, it does not place data into the buffer.  However, program execution 
 *  is not affected and the function does not generate an error.  This 
 *  handles a potential race condition where one thread calls 
 *  trace_process_cleanup() and another calls trace_store() before the first 
 *  thread can terminate the process. 
 *
 *
 */
void
trace_process_cleanup(void);

/*
 *
 *
 *  Function Name:  trace_module_cleanup
 *
 *
 *  Inputs:	hndl	- trace buffer handle.
 *
 *
 *  Outputs:	None.
 *
 *
 *  Returns:	None.
 *
 *
 *  Description:
 *
 *  This function removes the process trace buffer and notifies the Trace
 *  Subsystem that the application is exiting. When a process terminates
 *  normally, it must call trace_process_cleanup().
 *
 *  If the application calls the trace_store() function after it invokes this
 *  one, it does not place data into the buffer.  However, program execution
 *  is not affected and the function does not generate an error.  This
 *  handles a potential race condition where one thread calls
 *  trace_process_cleanup() and another calls trace_store() before the first
 *  thread can terminate the process.
 *
 *
 */
void
trace_module_cleanup(trace_hndl_t hndl);



/*
 * trace_hndl_t trace_attach(int mod_id, int pid, int number_entries)
 *
 * Attach to an existing trace buffer
 *
 * Inputs:
 *	mod_id	- module id of the buffer
 *	pid	- pid of the buffer
 *	number_entries - number of trace entries
 *
 * Outputs:
 *	N/A
 *
 * Returns:
 *	trace buffer handle, -1 if fail
 *
 */
trace_hndl_t
trace_attach(int mod_id, int pid, int number_entries);

/*
 * int trace_traverse(trace_hndl_t hndl, int back, trace_cb_func_t func)
 *
 * Traverse the buffer buffer backward or forward
 *
 * Inputs:
 *	hndl	- trace buffer handle
 *	back	- traverse forward (from earlier to latest) or backwards
 *	func	- call back function to process the entry
 *
 * Outputs:
 *	N/A
 *
 * Returns:
 *	TRACE_SUCCESS if succeed, otherwise TRACE_ERROR
 *
 */
int
trace_traverse(trace_hndl_t hndl, int back, trace_cb_func_t func);

/*
 * int trace_snapshot(trace_hndl_t hndl, int lock,
 *                      trace_entry_t **entry_list_p, int *num_p)
 *
 * Take a snapshot of the whole buffer
 *
 * Inputs:
 *	hndl	- trace buffer handle
 *	lock	- lock the buffer during taking snapshot
 *
 * Outputs:
 *	entry_list_p	- entry list (the caller needs to free the list)
 *	num_p		- number of entries
 *
 * Returns:
 *	TRACE_SUCCESS if succeed, otherwise TRACE_ERROR
 *
 */
int
trace_snapshot(trace_hndl_t hndl, int lock,
	trace_entry_t **entry_list_p, int *num_p);

int
trace_snapshot_filter(trace_hndl_t hndl, int lock,
	trace_entry_t **entry_list_p, int *num_p, trace_filter_t filter);

/*
 * int trace_resize(trace_hndl_t hndl, int new_num)
 *
 * Traverse the buffer buffer backward or forward
 *
 * Inputs:
 *	hndl	- trace buffer handle
 *	new_num	- new number of entries
 *
 * Outputs:
 *	N/A
 *
 * Returns:
 *	TRACE_SUCCESS if succeed, otherwise TRACE_ERROR
 *
 */
int
trace_resize(trace_hndl_t hndl, int new_num);

/*
 * int trace_reset(trace_hndl_t hndl)
 *
 * Clears trace buffer
 *
 * Inputs:
 *	hndl	- trace buffer handle
 *
 * Outputs:
 *	N/A
 *
 * Returns:
 *	TRACE_SUCCESS if succeed, otherwise TRACE_ERROR
 *
 */

int
trace_reset(trace_hndl_t hndl);

/*
 * int trace_get_buf_size(int mod_id)
 * Returns the max buffer size configured for the
 * module id passed as argument
 * Inputs:
 * mod_id - module ID
 *
 * Outputs:
 *	N/A
 *
 * Returns:
 * the max buffer size configured 
 */

int
trace_get_buf_size(int mod_id);
#endif

#endif  /*  __RAS_TRACE_H__ */
