/*
 * Copyright (c) 2006 Brocade Communications Systems, Inc.
 * All rights reserved.
 *
 * Description:
 *    CSMsg interface defines and function prototypes.  This interface
 *    provides the means for user-space applications on the BP to exchange
 *    messages with the BCM network processors.  Once device per network
 *    processor is provided for communication.  Multiple processes may
 *    open and share the same device.  Devices are named:
 *
 *    /dev/csmsg0		First network processor
 *    /dev/csmsg1		Second network processor
 *
 *    For legacy code reasons, a user-space library (libcsmsg.a) provides
 *    the API method csmsg_xbar_init() to open and initialize the devices
 *    listed above based on network processor circular queue (CQ) instance
 *    number which is 0 indexed.
 *
 */

#ifndef __CSMSG_H__
#define __CSMSG_H__

#ifdef __KERNEL__
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#else
#include <inttypes.h>
#include <sys/param.h>
#include <malloc.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <sys/uio.h>
#endif

#include <csmsg/csmsg_hdr.h>


#define CSMSG_DVR_NAME			"csmsg"


/*
 * Maximum number of csmsg devices.  Each device corresponds to one of
 * the network processors (BCM1250's).
 */
#define CSMSG_DEV_MAX			2


/* Prefix for csmsg devices, i.e. /dev/csmsg0 and /dev/csmsg1 */
#define CSMSG_DEV_PREFIX		"/dev/csmsg"

/* Device name for performing non-messaging control of CSMsg driver i/f */
#define CSMSG_DEV_CTRL			"/dev/csmsg_ctrl"
#define CSMSG_DEV_CTRL_NUM		2

/* Defines to indicate message destination circular queue to open */
#define CSMSG_NP_1			0		/* first 1250 on board */
#define CSMSG_NP_2			1		/* second 1250 on board */

#if 0

/* for EAP */
#define CSMSG_MAX_MSG_SIZE 		(1518 - 32) /* VRK - TODO */

#else

/* for production */
#define CSMSG_MAX_MSG_SIZE 		2232 /* Same as ps_LargeFrame_SIZE definition
                                      * in switch/port/include/portbuffers.h
                                      */

#endif

#define CSMSG_GET_PAYLOAD_PTR(_csmsg, _t) 							\
				(_t) (((char *) (_csmsg)) + sizeof(csmsg_hdr_t))

typedef enum {
	CS_OK 								= 0,
	CS_ERR 								= 1,
	CS_ERR_CSMSG_NO_ROUTE				= 2,
	CS_ERR_CSMSG_READ_FAILED 			= 3,
	CS_ERR_CSMSG_PAYLOAD_READ_FAILED 	= 4,
	CS_ERR_CSMSG_XBAR_WRITE_FAILED 		= 5,
} cserror_t;


#ifndef __KERNEL__

#define CSMSG_MAX_INST	2

#define CSMSG_CQ_1	CSMSG_NP_1		/* CQ number 1 */
#define CSMSG_CQ_2	CSMSG_NP_2		/* CQ number 2 */

/*
 * Config info structure
 */
typedef struct csmsg_config_info {
	union {
		u_char		wc[8];
		uint32_t	ww[2];
	} switch_wwn;			/* Switch WWN */
	uint8_t bp_slot_no;		/* Blade slot number */
	uint8_t num_chips;		/* Number of DPCs in the blade */
	uint8_t chip_list[CSMSG_MAX_INST];	/* DPC instances */
	uint8_t version;		/* Version */
	uint8_t	reserved[8];	/* Reserved fields */
} csmsg_config_info_t;

/*
 * Function: csmsg_blade_config_info()
 *
 * Description:
 *    Get the config information of the BP blade, including the switch
 *    WWN, from the BP.  If BP blade is booted in standalone mode, the
 *    information returned will not be defined.
 *
 * Input:
 *    *config_info	- pointer to structure for returning config info
 *
 * Output:
 *	  None.
 *
 * Returns:
 *    >= 0 	on success
 *    -1 	on error
 */
extern int csmsg_blade_config_info(csmsg_config_info_t *config_info);

/*
 * Function: csmsg_get_cq_num()
 *
 * Description:
 *    Get the circular queue number, given the DPC instance.
 *
 * Input:
 *    chip_inst	- DPC instance
 *
 * Output:
 *	  None.
 *
 * Returns:
 *    CQ number, on success
 *    -1, 		 on error
 */
extern uint8_t csmsg_get_cq_num(uint8_t chip_inst);

/*
 * Function: csmsg_blade_hw_addr()
 *
 * Description:
 *    Get the H/W addr on a blade given the interface name
 *
 * Input:
 *    if_name	- Interface name 
 *
 * Output:
 *	  H/W addr.
 *
 * Returns:
 *    0 on success
 *    errno on failure
 */
int csmsg_blade_hw_addr(char *if_name, uint8_t *hw_addr);

#endif /* ifndef __KERNEL__ */

#endif /* __CSMSG_H__ */
