

/*
 *    Copyright (c) 2007 Brocade Communications Systems, Inc.
 *    All rights reserved.
 *
 *    Module name:
 *        NPSAS driver (runs on 1250)
 *
 *    Description:
 *		Data structures and function prototypes for the
 *		APIs exported by the NPSAS driver.
 */


#ifndef __NPSAS_API_H__
#define	__NPSAS_API_H__

#include "sas_pmmr.h"	/* for egid_t */

#ifdef __cplusplus
extern "C" {
#endif


#ifdef __KERNEL__
#include <linux/types.h>
#elif defined __QNX__
#include "mpi/mpi.h"
#endif /* __KERNEL__ */

/*
 * The following are the events generated by FOS module(s) (running on CP) to
 * SAS Application running on NP (1250), generally meant to indicate somthing.
 *
 * The SAS Application can register its callbacks for the events upto
 * NPSAS_EVT_PRE_SLOT_POWEROFF (a total of NPSAS_MAX_REG_EVTS events)
 */
typedef enum npsas_event_type_s {
	/* Switch is Ready for virtual device registration */
	NPSAS_EVT_INITIATE_VD_LOGIN = 0,

	/* Sent just before the switch user ports are going to be disabled */
	NPSAS_EVT_PRE_SWITCH_DISABLE = 1,

	/* Sent just before the switch user ports are enabled */
	NPSAS_EVT_PRE_SWITCH_ENABLE = 2,

	/*
	 * Sent before the power on the blade is removed (and after
	 * the blade ports are disabled)
	 */
	NPSAS_EVT_PRE_SLOT_POWEROFF = 3,

	/* Sent just before the blade is faulted for whatever reason */
	NPSAS_EVT_PRE_BLADE_FAULT = 4,

	/* Max events that SAS Application could register its callbacks */
	NPSAS_MAX_REG_EVTS = NPSAS_EVT_PRE_BLADE_FAULT + 1,

	/* ACK message type (for pre/post notification events) */
	NPSAS_EVT_ACK = 100,

	/* Event indicating that list of areas available for SAS */
	NPSAS_EVT_AREA_LIST = 101,

	/* Event indicating that list of EGIDs available for NP cores */
	NPSAS_EVT_CORE_EGIDS = 102,

	/* EGID available for the condor trap (loopback) port */
	NPSAS_EVT_CDRTRAPPORT_EGID = 103,

	/* EGID available for the Anzio message port */
	NPSAS_EVT_AMSGPORT_EGID = 104,

} npsas_event_type_t;

/*
 * Data structure that describes the pre and post event notifications
 * that are in their simplest forms.
 *
 * Used for NPSAS_EVT_PRE_SWITCH_ENABLE, NPSAS_EVT_PRE_SWITCH_DISABLE,
 * NPSAS_EVT_INITIATE_VD_LOGIN and NPSAS_EVT_PRE_SLOT_POWEROFF events
 */
typedef struct npsas_prerpost_data_s {

	/* Sequence id that indicates the current incidence of the event */
	u_int		evt_seq_id;

	/* switch's local domain (confirmed) */
	u_int		sw_domain;

	u_int		reserved[3];

	/* Domain changed or not? */
	uint8_t		sw_domain_changed:1;
} npsas_prerpost_data_t;

/* Can we get around this local define and refer to cpsas_def.h? */
#define	NPSAS_ANZIO_NUM_DATA_PORTS		8

/* Data structure that describes the configuration information */
typedef struct npsas_cfg_data_s {
	u_int		np_instance; /* 1250 instance within the blade 0 or 1 */

	u_int		area_array[NPSAS_ANZIO_NUM_DATA_PORTS];

	uint8_t		slot_num;

	uint8_t		num_chips;	/* number of 1250s on this blade */

	/*
	 * Note not making use of wwn_t type since it needs to include
	 * bccb/sys/include/fc/wwn.h which we'd want to avoid
	 */
	u_int		sw_wwn[2];
} npsas_cfg_data_t;

#define	NPSAS_NUM_CORES		2

/* Data structure that describes the core egid information */
typedef struct npsas_core_data_s {
	u_int		slot_num;	/* slot num within the chassis */

	u_int		np_instance; /* 1250 instance within the blade, either 0 or 1 */

	u_int		egid_array[NPSAS_NUM_CORES];
} npsas_core_data_t;

/*
 * Data structure that describes the complete information about an
 * NPSAS event
 */
typedef struct npsas_event_s {
	npsas_event_type_t	evt_type;

	union {
		/* Data that pertains to Pre or Post notification event */
		npsas_prerpost_data_t	preevt_data;

		/* Event data that pertains to reserved area list, etc */
		npsas_cfg_data_t		cfg_data;

		/* Event data that carries the core egid information */
		npsas_core_data_t		core_data;

		/* Event data that carries the trap port's egid information */
		egid_t					trap_egid;

		/* Event data that carries the Anzio msg port's egid information */
		egid_t					amport_egid;
	} eventU;

	u_int	reserved[3];

} npsas_event_t;

/*
 * Data structure that describes the ACK event (acknowledgement for
 * any of the pre/post notification events).
 *
 * NOTE: The npsas_ack_event_t would go as part of the ackMsg_t which is
 *		 the actual message that would be deferenced by the CP component.
 *		 The message is constructed as shown below:
 *
 *	npsas_ack_event_t ackdata;
 *	ackMsg_t ackmsg;
 *
 *	ackmsg.km_type = MSG_ACK;
 *	ackmsg.msg.msg_state = NPSAS_EVT_ACK;
 *
 *  CPSAS driver converts port# to hdl before passing it up to the blade driver
 *	ackmsg.msg_handle = port#;
 *  ackmsg.msg_status = (int)&ackdata;
 */
typedef struct npsas_ack_event_s {

	/* Original event that this ACK is intended for */
	npsas_event_type_t		orig_event_type;

	npsas_prerpost_data_t	orig_event_data;

} npsas_ack_event_t;


#ifdef __QNX__

/* SAS callback routine prototype */
typedef int (*npsas_event_cb)(void *cb_data, npsas_event_type_t evt_type,
							  npsas_prerpost_data_t *evt_data);

typedef struct npsas_event_reg_s {
	npsas_event_type_t	evt_type;

	npsas_event_cb		cb_func;

	/*
	 * Data that can be optionally passed to the callback routine
	 * in addition to the evt_data
	 */
	void				*cb_data;
} npsas_event_reg_t;


/* Return codes that SAS driver APIs might return */
typedef enum npsas_status_s {
	NPSAS_STATUS_SUCCESS	= 0,

	/* Invalid input parameters */
	NPSAS_STATUS_INVINPUT	= 1,

	/* No route established for a given source and destination PIDs */
	NPSAS_STATUS_NOROUTE	= 2,

	/* Memory allocation failed, will this ever happen? */
	NPSAS_STATUS_NOMEMORY	= 3,

} npsas_status_t;

/*
 * ------------------------------------------------------------------------
 *								EXPORTED APIS to "main()"
 * ------------------------------------------------------------------------
 */
void npsas_init(void);
void npsas_exit(void);


/*
 * ------------------------------------------------------------------------
 *								EXPORTED APIS to MPI Dispatcher
 * ------------------------------------------------------------------------
 */
int npsas_msg_hdlr0(void *msg_data, unsigned int msg_len);
int npsas_msg_hdlr1(void *msg_data, unsigned int msg_len);


/*
 * ------------------------------------------------------------------------
 *								EXPORTED APIS to SAS Application
 * ------------------------------------------------------------------------
 */

/*
 * Description:
 *	This API provides a mechanism to SAS application ONLY for registering its
 *	callbacks' with the NPSAS driver.
 *
 * Note:
 *	No other module running on 1250 should register their callbacks.
 *
 * Input(s):
 *  cid - 1250's core number (0 or 1).
 *  event_table - Table of all events with their respective callbacks. An
 *		     invalid entry TABLE_END_MARKER which is defined as
 *		     "{-1, NULL, NULL}" would be used for the demarcation
 *		     and it's the responsibility of the caller to end the table
 *		     with that.
 *
 * Output(s):
 *
 * Returns:
 *  NPSAS_STATUS_SUCCESS - on success
 *
 * Context:
 *  Could be called from either Thread Context or Interrupt Context
 */
npsas_status_t
npsas_register_sas_cb(uint8_t cid, npsas_event_reg_t *event_cb_table);


/*
 * Description:
 *	This API provides a mechanism for de-registering the previously registered
 *	callbacks (at one shot) with the NPSAS driver. This API does nothing,
 *	if no callbacks were registered previously.
 *
 * Note:
 *	Selective de-registration of callbacks is not supported (there is no strong
 *	case/reason why SAS application would do selective de-register). Maybe
 *	SAS would never invoke this in its entire life but is provided for
 *	completeness.
 *
 * Input(s):
 *  cid - 1250's core number (0 or 1).
 *
 * Output(s):
 *
 * Returns:
 *  Nothing. The caller should expect that this is always successful
 *
 * Context:
 *  Could be called from either Thread Context or Interrupt Context
 */
void
npsas_deregister_sas_cb(uint8_t cid);


/*
 * Description:
 *  This API provides a mechanism for SAS application to send an
 *	acknowledgement for any event (the type to be passed in first argument)
 *	to a module that's running on CP. If the application needs to include
 *	any data in the message payload, it can specify through the second arg.
 *
 * Note: SUCCESS indicates that the msg was successfully enqueued to the CQ but
 *  it doesn't guarantee that the other side has picked-it up.
 *
 * Input(s):
 *  evt_type	- pre-notification event type that you are trying to acknowledge
 *  event		- ptr to the npsas_pre_event_t structure that describes the
 *			      original pre-notification event.
 *
 * Output(s):
 *
 * Returns:
 *  NPSAS_STATUS_INVINPUT - if evt is unknown type
 *  NPSAS_STATUS_SUCCESS  - on success
 *
 * Context:
 *  Could be called from either Thread Context or Interrupt Context
 */
npsas_status_t
npsas_send_event_ack(npsas_event_type_t evt, npsas_prerpost_data_t *event);


/*
 * Description:
 *	This API given the sid, did and oxid computes the hash value based on the
 *	policy configured and returns the output of the hash function.
 *
 * Input(s):
 *  s_id	- source port's fibre channel identifier
 *  d_id	- destination port's fibre channel identifier
 *  ox_id	- originator's exchange identifier
 *
 * Output(s):
 *  out_hashP - output generated by the hash routine. Caller should
 *				should allocate sufficient memory.
 *
 * Returns:
 *  NPSAS_STATUS_SUCCESS  - on success
 *  NPSAS_STATUS_INVINPUT - one of the given params is invalid
 *
 * Notes:
 *  The driver returns NPSAS_STATUS_INVINPUT even if the output arg
 *  is invalid (no memory allocated for out_hashP)
 *
 * Context:
 *  Could be called from either Thread Context or Interrupt Context
 */
extern inline npsas_status_t
npsas_get_hash_for_fcframe(uint32_t s_id, uint32_t d_id, uint32_t ox_id,
						   uint32_t *out_hashP);

/*
 * Description:
 *  This API returns the routing information (EGID and VC) given the FC frame
 *	parameters, S_ID, D_ID and OX_ID. If there exists more than one EGID
 *	that's reachable to the destination (D_ID), the API performs DPS internally
 *	based on the policy parameters configured and returns the EGID from the
 *	DPS output.
 *
 * Input(s):
 *  s_id	- source port's fibre channel identifier
 *  d_id	- destination port's fibre channel identifier
 *  ox_id	- originator's exchange identifier
 *
 * Output(s):
 *  out_egidP	- switch's egress identifier (final output port). Caller
 *		  should allocate memory sufficient enough to hold 1 EGID value.
 *  out_vcP	- external VC that the switch port should be using. Caller
 *		  should allocate memory sufficient enough to hold 1 VC value.
 *  out_stagP	- STAG value used by Pinball for SAS over FCR frames. Caller
 *		  should allocate memory sufficient enough to hold 1 STAG value.
 *
 * Returns:
 *  NPSAS_STATUS_SUCCESS	- on success
 *  NPSAS_STATUS_INVINPUT - one of the given params is invalid
 *  NPSAS_STATUS_NOROUTE	- no route exists for the given valid params
 *
 * Notes:
 *  The driver returns NPSAS_STATUS_INVINPUT even if one of the output args
 *  is invalid (like no memory allocated for either of the egid or vc or both)
 *
 * Context:
 *  Could be called from either Thread Context or Interrupt Context
 */
extern inline npsas_status_t
npsas_get_rinfo_for_fcframe(uint32_t s_id, uint32_t d_id, uint32_t ox_id,
							uint32_t *out_egidP, uint8_t *out_vcP,
							uint32_t *out_stagP);

/*
 * Description:
 *  Get the configuration information about the local NP cores like
 *	switch's WWN, slot number, total number of chips within this blade and
 *	the chip instance of the current core.
 *
 * Input(s):
 *  Nothing
 *
 * Output(s):
 *  out_switch_wwnP	- switch's WWN (caller should allocate 8 bytes of memory)
 *  out_slot_numP	- slot number where this blade is plugged-in
 *  out_num_chipsP	- number of chip (1250) instances within this slot. Caller
 *					  should assume the chip index is contiguous and starts
 *					  from 0 to *out_num_chipsP - 1
 *  out_chip_instP - chip instance for this 1250.
 *
 * Returns:
 *  NPSAS_STATUS_SUCCESS	- on success
 *  NPSAS_STATUS_INVINPUT - one of the given params is invalid
 *
 * Notes:
 *  The driver returns NPSAS_STATUS_INVINPUT even if one of the output args
 *  is invalid (like no memory allocated for either of out_switch_wwnP or
 *  out_slot_numP or out_num_chipsP).
 *
 * Context:
 *  Could be called from either Thread Context or Interrupt Context
 */
npsas_status_t
npsas_get_cinfo_for_npcores(uint8_t *out_switch_wwnP, uint8_t *out_slot_numP,
							uint8_t *out_num_chipsP, uint8_t *out_chip_instP);

/*
 * Description:
 *  This API returns the routing information (EGID and/or VC) given the
 *	slot number and the 1250 instance number within that blade. The routing
 *	information is to be used for the frames destined to either of the
 *	two cores of 1250. Frames originated by source 1250 to destination 1250
 *	would always go through the core (in the Sabre, there are direct links
 *	between the two condor chips).
 *
 *	The out_vc_listP would be filled with meaningful data only when the
 *	in_slot_num corresponds to the local slot, otherwise it'd be filled with
 *	invalid data (-1). Frames originated both by Anzio and CP destined to
 *	one core would take two different VCs and another two VCs are meant for
 *	reaching the 2nd core, a total of 4 different possible VC values.
 *
 * Input(s):
 *  in_slot_num		- input slot# (0,1-4 and 7-10, 0 is for Sabre)
 *  in_chip_num		- input chip# (0 or 1)
 *
 * Output(s):
 *  out_egid_listP	- list of egids assigned to two cores of NPSAS (caller
 *					  should allocate memory to hold 2 EGIDs)
 *  out_vc_listP	- list of all VCs, data frames would take to reach 1250
 *					  from Anzio (caller should allocate memory to hold 4 VCs)
 *
 *					  Note: this api returns valid value only if the in_slot_num
 *					  corresponds to the local slot
 *					  out_vc_listP[0] - VC for FCP frame to core in out_egid_listP[0]
 *					  out_vc_listP[1] - VC for CPU frame to core in out_egid_listP[0]
 *					  out_vc_listP[2] - VC for FCP frame to core in out_egid_listP[1]
 *					  out_vc_listP[3] - VC for CPU frame to core in out_edid_listP[1]
 *
 * Returns:
 *  NPSAS_STATUS_SUCCESS	- on success
 *  NPSAS_STATUS_INVINPUT	- one of the given params is invalid
 *
 * Notes:
 *  The driver returns NPSAS_STATUS_INVINPUT even if one of the output args
 *  is invalid (like no memory allocated for either of egid_list or vc_list,
 *	etc).
 *
 * Context:
 *  Could be called from either Thread Context or Interrupt Context
 */
npsas_status_t
npsas_get_rinfo_for_npcores(uint8_t in_slot_num, uint8_t in_chip_num,
							uint32_t *out_egid_listP, int8_t *out_vc_listP);


/*
 * Description:
 *	This API returns the routing information (EGID and VC) given an area
 *	that corresponds to one of the 8 condor-to-anzio data ports. This
 *	information would typically be used for the frames destined to Anzio
 *	from either of the two cores of 1250. The 1250 core uses EGID to reach
 *	Anzio and VC on the condor-to-sflipper port and as well as on the
 *	condor-to-anzio message port.
 *
 * Input(s):
 *  area	- area that corresponds to one of the 8 cdr-to-anzio data ports.
 *
 * Output(s):
 *  out_egidP	- egid that's reachable to the given area (that corresponds to
 *				  one of the 8 condor-to-anzio data ports). The caller should
 *				  allocated memory sufficient enough to hold 1 egid value.
 *  out_vcP		- VC to be used by 1250 cores to transmit frames to Anzio on
 *				  the cdr-to-sflipper port and as well as on the
 *				  cdr-to-anzio msg port (the caller should allocate memory to
 *				  hold 1 VC value)
 *
 * Returns:
 *  NPSAS_STATUS_SUCCESS	- on success
 *  NPSAS_STATUS_INVINPUT - one of the given params is invalid
 *
 * Notes:
 *  The driver returns NPSAS_STATUS_INVINPUT even if one of the output args
 *  is invalid (like no memory allocated for either of egid_list or vc_list or
 *  both).
 *
 * Context:
 *  Could be called from either Thread Context or Interrupt Context
 */
npsas_status_t
npsas_get_rinfo_for_ctrlframes(uint32_t area, uint32_t *out_egidP,
							   uint8_t *out_vcP);


/*
 * Description:
 *  Return the EGID assigned for the condor-to-anzio message port.
 *
 * Input(s):
 *  None
 *
 * Output(s):
 *  out_egidP	- egress identifier assigned to condor-to-anzio msg port. Caller
 *		  should allocate memory sufficient enough to hold 1 egid value.
 *
 * Returns:
 *  NPSAS_STATUS_SUCCESS	- on success
 *  NPSAS_STATUS_INVINPUT - one of the given params is invalid
 *
 * Notes:
 *  The driver returns NPSAS_STATUS_INVINPUT even if the output arg
 *  is invalid (like no memory allocated for the egid)
 *
 * Context:
 *  Could be called from either Thread Context or Interrupt Context
 */
npsas_status_t
npsas_get_egid_for_anziomsgport(uint32_t *out_egidP);


/*
 * Description:
 *  Return the EGID and VC assigned for frame to condor loopback port.
 *
 * Input(s):
 *  None
 *
 * Output(s):
 *  out_egidP	- egress identifier assigned to condor loopback port. Caller
 *		  should allocate memory sufficient enough to hold 1 egid value.
 *  out_vcP - VC assigned to sflipper to condor link for the trap port frame.
 *
 * Returns:
 *  NPSAS_STATUS_SUCCESS	- on success
 *  NPSAS_STATUS_INVINPUT - one of the given params is invalid
 *
 * Notes:
 *  The driver returns NPSAS_STATUS_INVINPUT even if one of the output args
 *  is invalid (like no memory allocated for the egid)
 */
npsas_status_t
npsas_get_rinfo_for_trapport(uint32_t *out_egidP, uint8_t *out_vcP);


/*
 * Description:
 *  This API returns the list of all the areas that were assigned to the
 *  8 condor-to-anzio data ports.
 *
 * Input(s):
 *  Nothing
 *
 * Output(s):
 *  out_area_listP - list of areas that were assigned to
 *			 8 cdr-to-anzio data ports
 *
 * Returns:
 *  NPSAS_STATUS_SUCCESS	- on success
 *  NPSAS_STATUS_INVINPUT - The given input param is invalid
 *
 * Notes:
 *  The driver returns NPSAS_STATUS_INVINPUT even if the output arg
 *  is invalid (like no memory allocated for area_list)
 *
 * Context:
 *  Could be called from either Thread Context or Interrupt Context
 */
npsas_status_t
npsas_get_areas_for_anziodataports(uint32_t *out_area_listP);


/*
 * Description:
 *  This API returns the list of other instances of 1250.
 *
 * Input(s):
 *  Nothing
 *
 * Output(s):
 *  *out_chip_instP - list of other instances of 1250.  User should
 *				  	  allocate space sufficient for all the other instances
 *
 * Returns:
 *  NPSAS_STATUS_SUCCESS	- on success
 *  NPSAS_STATUS_INVINPUT 	- The given input param is invalid
 *
 * Notes:
 *  The driver returns NPSAS_STATUS_INVINPUT even if the output arg
 *  is invalid (like no memory allocated for out_chip_instP)
 *
 * Context:
 *  Could be called from either Thread Context or Interrupt Context
 */
npsas_status_t
npsas_get_other_chip_inst(uint8_t *out_chip_instP);

/*
 * Description:
 *  This API returns the CQ number, given a 1250 instance.
 *
 * Input(s):
 *  Nothing
 *
 * Output(s):
 *  chip_inst - 1250 chip instance
 *  *out_cqP  - circular queue number
 *
 * Returns:
 *  NPSAS_STATUS_SUCCESS	- on success
 *  NPSAS_STATUS_INVINPUT 	- The given input param is invalid
 *
 * Notes:
 *  The driver returns NPSAS_STATUS_INVINPUT even if the output arg
 *  is invalid (like no memory allocated for out_chip_instP)
 *
 * Context:
 *  Could be called from either Thread Context or Interrupt Context
 */
npsas_status_t
npsas_get_cq_num(uint8_t chip_inst, uint8_t *out_cqP);

/*
 * Description:
 *  Initialize FOS drivers and utilities.
 *
 * Input(s):
 *  cpu - core
 *
 * Output(s):
 *  N/A
 *
 * Returns:
 *  N/A
 *
 */
void
fos_init(int cpu, int ch_id);


#endif	/*  __QNX__ */

#ifdef __cplusplus
}
#endif

#endif	/* __NPSAS_API_H__ */
