/*
 * Copyright (c) 2005-2007 Brocade Communications Systems, Inc.
 * All rights reserved.
 *
 * Description:
 *     Internal Message Passing Interface (MPI) defines and function 
 *		prototypes.
 */

#ifndef __MPI_PRIV_H__
#define __MPI_PRIV_H__

#include <mpi/mpi.h>
#include <mpi/mpi_mach_dep.h>

#ifdef __cplusplus
extern "C" {
#endif


/*
 * ----------------------------------------------------------------------------
 *                               D E F I N E S
 * ----------------------------------------------------------------------------
 */

#define	MPI_HDL_MAGIC			0x60629396
#define MPI_HDL_INVALID(x)		(*((int *)(x)) != MPI_HDL_MAGIC ? 1 : 0)
#define MPI_IHDR_SZ				(sizeof (mpi_imsg_t))

#define MPI_NULL_PAGE			-1
#define MPI_NULL_MBLK			-1

/*
 * Invalid mblk/page management value.  Used to detect when shared
 * memory suddenly goes away.
 */
#define MPI_INVALID_MGMT_VAL	0xff

typedef enum {
	MPI_FALSE = 0,
	MPI_TRUE  = 1
} mpi_bool_t;


/*
 * Circular Queue defines and types.
 */

/* Consumer/producer index type */
typedef uint16_t mpi_cpi_idx_t;
#define MPI_CPI_SZ			(sizeof (mpi_cpi_idx_t))


/* CQ storage and state information */
typedef struct mpi_cq {
	volatile mpi_cpi_idx_t	*r_ci_base;		/* Rx consumer index base addr */
	volatile mpi_cpi_idx_t	*r_pi_base;		/* Rx producer index base addr */
	volatile mpi_cpi_idx_t	*t_ci_base;		/* Tx consumer index base addr */
	volatile mpi_cpi_idx_t	*t_pi_base;		/* Tx producer index base addr */
	volatile uint8_t		*t_q_base;		/* Tx CQ base addr */
	volatile uint8_t		*r_q_base;		/* Rx CQ base addr */
	volatile uint8_t		*mblk_base;		/* mem block base addr */
	volatile uint8_t		*mblk_mgmt_base;/* mem block mgmt data base addr */
	int						mblk_start_idx;	/* starting mblk index */
	int						mblk_end_idx;	/* ending mblk index */
	volatile uint8_t		*page_base;		/* page table base addr */
	volatile uint8_t		*page_mgmt_base;/* page tbl mgmt data baddr */
	int						page_start_idx;	/* starting page index */
	int						page_end_idx;	/* ending page index */
	mpi_stats_t				*stats;			/* CQ statistics */
	int						next_alloc_pg;	/* next page to allocate */
	int						next_alloc_mblk;/* next mblk to allocate */
} mpi_cq_t;


/* Memory block management data.  Tracks the state of an mblk. */
typedef union mpi_mblk_mgmt {
	uint8_t val;
	struct {
		uint8_t unused	: 7;
		uint8_t in_use	: 1;
	} fld;
} mpi_mblk_mgmt_t;


/* Page table management data.  Tracks the state of a page. */
typedef union mpi_page_mgmt {
	uint8_t val;
	struct {
		uint8_t unused	: 7;
		uint8_t in_use	: 1;
	} fld;
} mpi_page_mgmt_t;


/* Align CQ elements on double word boundary */
#define MPI_CQ_ALIGN_SZ		8
#define MPI_CQ_ALIGNED(x)	((x) + (((unsigned)(x)) % MPI_CQ_ALIGN_SZ))
#define MPI_CQ_ALIGN(x)		x = MPI_CQ_ALIGNED(x)


/* --- CQ size, ci/pi reference, entry reference, and CQ state --- */

/* Size of an entry in a CQ */
#define MPI_CQ_ENTRY_SZ		(sizeof (mpi_imsg_t))

/* Two CQ per Rx/Tx pair */
#define MPI_RX_TX_PAIR		2

/* Total number of consumer/producer indices */
#define MPI_CPI_NUM			4

/* Consumer/Producer index references */
#define MPI_CQ_RX_CI(hdl_, cq_)		*((hdl_)->cq.r_ci_base + (cq_))
#define MPI_CQ_RX_PI(hdl_, cq_)		*((hdl_)->cq.r_pi_base + (cq_))
#define MPI_CQ_TX_CI(hdl_, cq_)		*((hdl_)->cq.t_ci_base + (cq_))
#define MPI_CQ_TX_PI(hdl_, cq_)		*((hdl_)->cq.t_pi_base + (cq_))

/* Base address of a Tx CQ */
#define MPI_CQ_TX_BASE(hdl_, cq_)	((hdl_)->cq.t_q_base + \
	((MPI_CQ_ENTRY_SZ * (hdl_)->cq_sz) * (cq_)))

/* Base address of a Rx CQ */
#define MPI_CQ_RX_BASE(hdl_, cq_)	((hdl_)->cq.r_q_base + \
	((MPI_CQ_ENTRY_SZ * (hdl_)->cq_sz) * (cq_)))

/* Get address of a Tx CQ entry */
#define MPI_CQ_TX_ENTRY(hdl_, cq_, i_)	\
	(MPI_CQ_TX_BASE(hdl_, cq_) + (MPI_CQ_ENTRY_SZ * (i_)))

/* Get address of a Rx CQ entry */
#define MPI_CQ_RX_ENTRY(hdl_, cq_, i_)	\
	(MPI_CQ_RX_BASE(hdl_, cq_) + (MPI_CQ_ENTRY_SZ * (i_)))

/* Increment a Rx CQ consumer index */
#define MPI_INCR_RX_CI(hdl_, cq_)	\
	MPI_CQ_INCR(MPI_CQ_RX_CI(hdl_, cq_), (hdl_)->cq_sz, 1)

/* Increment a Tx CQ consumer index (only for CQ flush) */
#define MPI_INCR_TX_CI(hdl_, cq_)	\
	MPI_CQ_INCR(MPI_CQ_TX_CI(hdl_, cq_), (hdl_)->cq_sz, 1)

/* Increment a Tx CQ producer index */
#define MPI_INCR_TX_PI(hdl_, cq_)	\
	MPI_CQ_INCR(MPI_CQ_TX_PI(hdl_, cq_), (hdl_)->cq_sz, 1)

/* Check if Tx CQ full */
#define MPI_TX_CQ_FULL(hdl_, cq_)	(MPI_CQ_TX_CI(hdl_, cq_) == \
	((MPI_CQ_TX_PI(hdl_, cq_) + 1) % (hdl_)->cq_sz) ? 1 :0)

/* Check if Rx CQ empty */
#define MPI_RX_CQ_EMPTY(hdl_, cq_) \
    ((MPI_CQ_RX_CI(hdl_, cq_) == MPI_CQ_RX_PI(hdl_, cq_)) ? 1 : 0)

/* Check if Tx CQ empty */
#define MPI_TX_CQ_EMPTY(hdl_, cq_) \
    ((MPI_CQ_TX_CI(hdl_, cq_) == MPI_CQ_TX_PI(hdl_, cq_)) ? 1 : 0)

/* #msgs pending in Tx CQ */
#define MPI_TX_CQ_PENDING(hdl_, cq_)	\
	(((unsigned)MPI_CQ_TX_PI(hdl_, cq_) - \
	  (unsigned)MPI_CQ_TX_CI(hdl_, cq_)) % (hdl_)->cq_sz)

/* #msgs pending in Rx CQ */
#define MPI_RX_CQ_PENDING(hdl_, cq_)	\
	(((unsigned)MPI_CQ_RX_PI(hdl_, cq_) - \
	  (unsigned)MPI_CQ_RX_CI(hdl_, cq_)) % (hdl_)->cq_sz)

/* #msgs that may be enqueued to Tx CQ */
#define MPI_TX_CQ_CAPACITY(hdl_, cq_)	\
	(hdl_->cq_sz - MPI_TX_CQ_PENDING(hdl_, cq_) - 1)

/* #msgs that may be enqueued to Rx CQ */
#define MPI_RX_CQ_CAPACITY(hdl_, cq_)	\
	(hdl_->cq_sz - MPI_RX_CQ_PENDING(hdl_, cq_) - 1)


/* --- Memory block sizes, mem usage, mgmt data, and references --- */

#define MPI_MBLK_SZ			(sizeof (mpi_mblk_t))

/* #mem blks = 2 * total number of CQ entries  */
#define MPI_MBLKS(num_cq_, cq_sz_)	((num_cq_) * (cq_sz_) * 2)

#define MPI_MBLK_MGMT_ENTRY_SZ	(sizeof (mpi_mblk_mgmt_t))

/* Get base address of a memory block */
#define MPI_MBLK_ADDR(hdl_, blk_)	\
	(((blk_) == MPI_NULL_MBLK) ? NULL : \
	 (mpi_mblk_t *)((hdl_)->cq.mblk_base + (MPI_MBLK_SZ * (blk_))))

/* Get base address of a memory management entry */
#define MPI_MBLK_MGMT_ADDR(hdl_, blk_)	\
	(((mpi_mblk_mgmt_t *) (hdl_)->cq.mblk_mgmt_base) + (blk_))

/* Reference a block management entry */
#define MPI_MBLK_MGMT(hdl_, blk_)	\
	((mpi_mblk_mgmt_t *) MPI_MBLK_MGMT_ADDR(hdl_, blk_))


/* --- Page sizes, mem usage, mgmt data, and references --- */

#define MPI_PAGE_SZ(hdl_)		(hdl_)->page_sz
#define MPI_PAGES(hdl_)			(hdl_)->num_pages
#define MPI_PAGE_MGMT_SZ(hdl_)	MPI_PAGES(hdl_)
#define MPI_PAGE_MGMT_ENTRY_SZ	(sizeof (mpi_page_mgmt_t))

/* Get base address of a page */
#define MPI_PAGE_ADDR(hdl_, pg_)	\
	((hdl_)->cq.page_base + (MPI_PAGE_SZ(hdl_) * (pg_)))

/* Get base address of a page table management entry */
#define MPI_PAGE_MGMT_ADDR(hdl_, pg_)	((hdl_)->cq.page_mgmt_base + (pg_))

/* Reference a page management entry */
#define MPI_PAGE_MGMT(hdl_, pg_)	\
	((mpi_page_mgmt_t *) MPI_PAGE_MGMT_ADDR(hdl_, pg_))

/* Returns true if current MPI instance is the memory owner */
#define MPI_MEM_OWNER(hdl_) hdl_->cq_mem_owner

/* Validate consumer/producer index value */
#define MPI_CPI_VALID(hdl_, cpi_)		((cpi_) < hdl_->cq_sz)

/* Validate mblk linked list index value (includes NULL mblk pointer) */
#define MPI_MBLK_LLIST_IDX_VALID(hdl_, idx_)	\
	(((idx_) >= 0 && (idx_) < hdl_->num_mblks) || ((idx_) == MPI_NULL_MBLK))

/* Validate mblk index value (includes NULL mblk pointer) */
#define MPI_MBLK_IDX_VALID(hdl_, idx_)	\
	((idx_) >= 0 && (idx_) < hdl_->num_mblks)

/* Validate page index value */
#define MPI_PG_IDX_VALID(hdl_, idx_)	((idx_) < hdl_->num_pages)


/* Internal MPI handle type.  Holds the MPI instantiation parameters. */
typedef struct mpi_ihandle {
	int					magic;
	int					num_cq;
	int 				cq_sz;
	int					num_mblks;
	int 				page_sz;
	int 				num_pages;
	void 				*cq_base_addr;
	int					cq_mem_owner;
	mpi_dma_info_t 		dma_info;
	mpi_notify_fn_t		rx_notify_fn;
	void				*rx_notify_arg;
	mpi_notify_fn_t		tx_notify_fn;
	void				*tx_notify_arg;
	mpi_sap_app_map_t	sap_app_map[MPI_MAX_SAP_HDLRS];
	int					num_sap_app;
	mpi_cq_t			cq;
	mpi_sem_type_t		alloc_mutex;
	mpi_sem_type_t		rx_mutex;
	mpi_sem_type_t		tx_mutex;
} mpi_ihandle_t;


/*
 * Message memory block defines.  These datastructures allow
 * definition and management of SGL messages as well as dynamic
 * memory blocks that need to be allocated from MPI shared memory.
 */

/* Type of mblk allocated */
typedef enum {
	MPI_MBLK_TYPE_HEAP = 1,		/* mem blk from heap, e.g. malloc() */
	MPI_MBLK_TYPE_SM			/* mem blk from shared memory */
} mpi_mblk_type_t;

/* Type of mblk buffer memory */
typedef enum {
	MPI_MBLK_MEM_TYPE_HEAP = 3,	/* mem blk from heap, e.g. malloc() */
	MPI_MBLK_MEM_TYPE_SM, 		/* mem blk from shared memory */
	MPI_MBLK_MEM_TYPE_USER		/* addr points to user-provided buffer */
} mpi_mblk_mem_type_t;


/*
 * MPI memory block.  Describes a block of memory used for message payload
 * storage.  If the MPI instance is the CQ memory owner, memory is
 * allocated PCI shared memory organized as pages, otherwise it is
 * allocated from local HEAP via malloc().  
 */
typedef struct mpi_mblk {
	void *addr;					/* addr of malloc'd or shared mem buf */

	int	cur_len;				/* current length in bytes */

	uint16_t type;				/* mblk type (mpi_mblk_type_t) */
	union {
		struct {
			int  mblk_idx;		/* index into mblk table */
			int  next;			/* index into shared mem pool of next mblk */
		} sm;

		struct {
			struct mpi_mblk *next;	/* ptr to next mblk */
		} heap;
	} inst;

	uint16_t mem_type;			/* mblk mem type (mpi_mblk_mem_type_t) */
	union {
		/* type == MPI_MBLK_MEM_TYPE_SM */
		struct {				/* 1 or more pages alloc'd from shared mem */
			uint16_t start_pg;	/* starting index into page table */
			uint16_t num_pgs;	/* number of pages allocated */
		} sm;

		/* type == MPI_MBLK_MEM_TYPE_HEAP */
		struct {
			int  malloc_len;	/* len of buffer allocated */
		} heap;

		/* type == MPI_MBLK_MEM_TYPE_USER */
		struct {
			mpi_tx_cb_fn_t cb;	/* user callback function */
		} user;
	} mem;

} mpi_mblk_t;


/*
 * Internal MPI msg type.  Superset of mpi_msg_t in that it includes
 * an additional mblk union which can be either an mblk index or a
 * pointer to an mblk.
 */
typedef struct mpi_imsg {
	mpi_msg_type_t	type;		/* MPI message type */
	int				msg_len;	/* msg payload length in bytes */
	void			*msg_buf;	/* pointer to message buffer */

	union {
		mpi_mblk_t	*ptr;
		int			index;
	} mblk;
} mpi_imsg_t;


/*
 * ----------------------------------------------------------------------------
 *                      F U N C T I O N  P R O T O T Y P E S
 * ----------------------------------------------------------------------------
 */

/*
 * Function: mpi_malloc_()
 *
 * Description:
 *    Wrapper function for machine-dependent dynamic memory allocation.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *    size			Size of memory block in bytes to allocate
 *
 * Output:
 *	  None.
 *
 * Returns:
 *    Pointer to allocated memory block
 *    NULL on failure
 */
extern void *mpi_malloc_(int size);


/*
 * Function: mpi_free_()
 *
 * Description:
 *    Wrapper function for machine-dependent memory free.
 *
 * Input:
 *    hdl			MPI handle returned from mpi_init()
 *    objp			Pointer to memory block allocated by mpi_malloc_()
 *
 * Output:
 *	  None.
 *
 * Returns:
 *    None.
 */
extern void mpi_free_(void *objp);


/*
 * Function: mpi_alloc_mblk_()
 *
 * Description:
 *    Internal function that allocates a memory block.  If the caller
 *    is the memory owner, a memory block is allocated from the pool in
 *    shared memory.  Othewise, it is dynamically allocated.
 *
 * Input:
 *    ihdl			Internal MPI handle
 *    type			mblk type (heap, or shared mem)
 *    mem_type		mblk memory type (heap, shared mem, or user buffer)
 *    size			Size of memory block in bytes
 *
 * Output:
 *    mblk			Pointer to newly allocated mblk
 *
 * Returns:
 *    0				On success
 *    -EINVAL		Invalid ihdl or mblk
 *    -ENOMEM		Failed to allocate memory block
 */
extern int
mpi_alloc_mblk_(mpi_ihandle_t *ihdl, mpi_mblk_type_t type,
				mpi_mblk_mem_type_t mem_type, int size, mpi_mblk_t **mblk);


/*
 * Function: mpi_alloc_sm_mblk_()
 *
 * Description:
 *    Allocates a memory block memory from shared memory pool.  This
 *    function is used to force allocation from shared memory for
 *    enqueuing operations where CQ accessible memory must be used
 *    for storing message payload.
 *
 * Input:
 *    ihdl			Internal MPI handle
 *
 * Output:
 *    mblk			Pointer to newly allocated mblk
 *
 * Returns:
 *    0				On success
 *    -EINVAL		Invalid ihdl or mblk
 *    -ENOMEM		Failed to allocate memory block
 */
extern int mpi_alloc_sm_mblk_(mpi_ihandle_t *ihdl, mpi_mblk_t **mblk);


/*
 * Function: mpi_free_mblk_()
 *
 * Description:
 *    Internal function that frees a memory block memory.  If the caller
 *    is the memory owner, a memory block is returned to the pool in
 *    shared memory.  Othewise, it is dynamically freed.
 *
 * Input:
 *    ihdl			Internal MPI handle
 *
 * Output:
 *    mblk			Pointer to mblk to be freed
 *
 * Returns:
 *    0				On success
 *    -EINVAL		Invalid ihdl, size, or mblk
 */
extern int mpi_free_mblk_(mpi_ihandle_t *ihdl, mpi_mblk_t *mblk);


/*
 * Function: mpi_alloc_mblk_mem_()
 *
 * Description:
 *    Internal function that allocates actual memory block memory.
 *    Depending on the platform, this may be a simple system call to
 *    malloc() or an allocation from shared memory.
 *
 * Input:
 *    ihdl			Internal MPI handle
 *    size			Size in bytes
 *
 * Output:
 *    mblk			Pointer to mblk that will track allocated block of memory
 *
 * Returns:
 *    0				On success
 *    -EINVAL		Invalid ihdl, size, or mblk
 *    -ENOMEM		Failed to allocate requested size
 */
extern int mpi_alloc_mblk_mem_(mpi_ihandle_t *ihdl, int size, mpi_mblk_t *mblk);


/*
 * Function: mpi_free_mblk_mem_()
 *
 * Description:
 *    Internal function that frees memory block memory.  Depending on
 *    the platform, this may be a simple system call to free() or
 *    freeing a block allocated from shared memory.
 *
 * Input:
 *    hdl			Internal MPI handle 
 *    mblk			Pointer to allocated block of memory
 *
 * Output:
 *    None.
 *
 * Returns:
 *    0				On success
 *    -EINVAL		Invalid hdl, or mblk
 */
extern int mpi_free_mblk_mem_(mpi_ihandle_t *hdl, mpi_mblk_t *mblk);


/*
 * Function: mpi_alloc_pages_()
 *
 * Description:
 *    Internal function that allocates a series of pages from shared memory.
 *
 * Input:
 *    ihdl			Internal MPI handle
 *    size			Size in bytes
 *
 * Output:
 *    mblk			Pointer to mblk which will track allocated pages
 *
 * Returns:
 *    0				On success
 *    -EINVAL		Invalid ihdl, size, or mblk
 *    -ENOMEM		Failed to allocate pages for requested size
 */
extern int mpi_alloc_pages_(mpi_ihandle_t *ihdl, int size, mpi_mblk_t *mblk);


/*
 * Function: mpi_free_pages_()
 *
 * Description:
 *    Internal function that frees a series of pages from shared memory.
 *
 * Input:
 *    ihdl			Internal MPI handle
 *    mblk			Pointer to mblk containg pages to be freed
 *
 * Output:
 *    None.
 *
 * Returns:
 *    0				On success
 *    -EINVAL		Invalid ihdl, mblk ptr, or mblk type
 */
extern int mpi_free_pages_(mpi_ihandle_t *ihdl, mpi_mblk_t *mblk);


/*
 * Function: mpi_validate_hdl_()
 *
 * Description:
 *    Verify that the specified handle is non-null and a valid MPI one.
 *
 * Input:
 *    hdl			Internal MPI handle to verify.
 *    func			Invoking function name.
 *
 * Output:
 *	  None.
 *
 * Returns:
 *    0				On success.
 *    -EINVAL		Invalid hdl.
 */
extern int mpi_validate_hdl_(mpi_ihandle_t *hdl, const char *func);


/*
 * Function: mpi_validate_type_()
 *
 * Description:
 *    Verify that the specified type is one of the valid MPI msg types.
 *
 * Input:
 *    type			Message type to verify.
 *    func			Invoking function name.
 *
 * Output:
 *	  None.
 *
 * Returns:
 *    0				On success.
 *    -EINVAL		Invalid type.
 */
extern int mpi_validate_type_(mpi_msg_type_t type, const char *func);


/*
 * Function: mpi_dump_pg_mgmt_()
 *
 * Description:
 *    Dump the contents of the page management table
 *
 * Input:
 *    ihdl			Internal MPI handle
 *
 * Output:
 *	  None.
 *
 * Returns:
 *    None.
 */
extern void mpi_dump_pg_mgmt_(mpi_ihandle_t *hdl);


/*
 * Function: mpi_dump_mblk_mgmt_()
 *
 * Description:
 *    Dump the contents of the mblk management table
 *
 * Input:
 *    ihdl			Internal MPI handle
 *
 * Output:
 *	  None.
 *
 * Returns:
 *    None.
 */
extern void mpi_dump_mblk_mgmt_(mpi_ihandle_t *hdl);


/*
 * Function: mpi_dump_msg_()
 *
 * Description:
 *    Dump the message attributes
 *
 * Input:
 *    ihdl			Internal MPI handle
 *    msg			Pointer to message to dump
 *    dump_payload	Flags if payload should be dumped as well
 *    chksum		Flags if checksum should be computed
 *
 * Output:
 *	  None.
 *
 * Returns:
 *    None.
 */
extern void
mpi_dump_msg_(mpi_ihandle_t *hdl, mpi_msg_t *msg, int dump_payload, int chksum);


/*
 * Function: mpi_dump_msg_()
 *
 * Description:
 *    Creates an internal MPI message instance from a message on a CQ.
 *    Copies the header portion and fixes up the mblk reference from an
 *    index value to a local pointer.
 *
 * Input:
 *    ihdl			Internal MPI handle
 *    cq_buf		Pointer to CQ memory where message resides
 *
 * Output:
 *	  cq_msg		Pointer to message returned from CQ.
 *
 * Returns:
 * 	  0				On success
 *    -ENOMEM		Failed to allocate space for the new message.
 */
extern int
mpi_get_cq_msg_(mpi_ihandle_t *ihdl, volatile uint8_t *cq_buf,
					mpi_imsg_t **cq_msg);


/*
 * Function: mpi_mach_dep_init()
 *
 * Description:
 *    This function is called during MPI initialization to perform any
 *    machine dependent initialization.  Even if no init needs to be
 *    done, every platform must provide an implementation of this routine.
 *
 * Input:
 *    ihdl			Internal MPI handle
 *    num_cq			Number of CQ Rx/Tx pairs to create
 *
 *    cq_sz				Number of entries in each CQ Rx/Tx pair
 *
 *    page_sz			Size of MPI storage page in bytes
 *
 *    num_pages			Number of pages in free page pool
 *
 *    cq_mem_owner		Flags if CQ memory resides on this platform
 *
 * Output:
 *	  None.
 *
 * Returns:
 * 	  0				On success
 * 	  !0			Failed to perform machine depenent init.  Fatal error.
 */
extern int mpi_mach_dep_init(int num_cq, int cq_sz, int page_sz,
								int num_pages, int cq_mem_owner);


#ifdef __cplusplus
}
#endif

#endif
